/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.processus;

import java.sql.Date;
import java.util.HashMap;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.email.JSBMailbox;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.config.PropertyHelper;
import com.kportal.core.security.MySQLHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.univ.objetspartages.om.Changementmotpasse;
import com.univ.objetspartages.om.Utilisateur;
import com.univ.utils.ContexteDao;
import com.univ.utils.EscapeString;
import com.univ.utils.URLResolver;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.operande.TypeOperande;

/**
 * The Class MotDePasseCtrl.
 * 
 * @author pguiomar
 * 
 *         Gestion du changement de mot de passe pour un utilisateur identifie.
 */
public class MotDePasseCtrl {

	/** The utilisateur. */
	private Utilisateur utilisateur = null;

	/**
	 * Récupère du jtf la propriété permettant de connaitre la longueur max du mot de passe
	 */
	private static final String PROPRIETE_LONGUEUR_MOT_DE_PASSE = "utilisateur.password.longueur.max";

	/**
	 * Valeur par défaut si rien n'est spécifié dans la conf
	 */
	public static final int LONGUEUR_MOT_DE_PASSE_DEFAUT = 64;

	/**
	 * Retourne la longeur du champ mot de passe a utiliser dans les formulaires
	 * 
	 * @return la valeur de la propriété ou {@link MotDePasseCtrl#LONGUEUR_MOT_DE_PASSE_DEFAUT} si non défini
	 */
	public static int getLongueurChampMotDePasse() {
		int tailleMotDePasse = LONGUEUR_MOT_DE_PASSE_DEFAUT;
		final String prop = PropertyHelper.getCoreProperty(PROPRIETE_LONGUEUR_MOT_DE_PASSE);
		if (StringUtils.isNotBlank(prop) && StringUtils.isNumeric(prop)) {
			try {
				tailleMotDePasse = Integer.parseInt(prop);
			} catch (final NumberFormatException e) {
				tailleMotDePasse = LONGUEUR_MOT_DE_PASSE_DEFAUT;
			}
		}
		return tailleMotDePasse;
	}

	/**
	 * 1er phase de la gestion du mot de passe (regeneration) recuperation des donnees de l'utilisateur, enregistrement des donnees de l'utilsateur dans un objet provisoire
	 * (ChangementMotDePasse), envoi d'email vers l'utilisateur avec un lien HTTPS de retour vers ce ctrl pour traiter la suite de la gestion du mot de passe.
	 * 
	 * @param infoBean
	 *            l'infobean pour avoir le champ email et login
	 * @param ctx
	 *            the ctx
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public void demande(final InfoBean infoBean, final OMContext ctx) throws Exception {
		final String email = infoBean.getString("EMAIL");
		final ClauseWhere where = new ClauseWhere(ConditionHelper.egalVarchar("ADRESSE_MAIL", email));
		where.and(ConditionHelper.notEgal("MOT_DE_PASSE", StringUtils.EMPTY, TypeOperande.VARCHAR));
		final String code = infoBean.getString("LOGIN");
		if (StringUtils.isNotEmpty(code)) {
			where.and(ConditionHelper.egalVarchar("CODE", code));
		}
		utilisateur = new Utilisateur();
		utilisateur.setCtx(ctx);
		utilisateur.init();
		final int nb = utilisateur.select(where.formaterSQL());
		if (nb == 0) {
			throw new ErreurApplicative(MessageHelper.getCoreMessage(StringUtils.isNotEmpty(code) ? "ST_ERR_DEMANDE_MDP_EMAIL_LOGIN" : "ST_ERR_DEMANDE_MDP_EMAIL"));
		} else if (nb > 1) {
			throw new ErreurApplicative(MessageHelper.getCoreMessage("ST_ERR_DEMANDE_MDP_COMPTES"));
		}
		utilisateur.nextItem();
		final String id = this.enregistrementDEMANDE_MDP(utilisateur.getCode(), email);
		this.envoiEmailDEMANDE_MDP(id, email, ctx);
	}

	/**
	 * 2e phase de la generation du mot de passe. recuperation des donnees a partir de l'objet temporaire (Changementmotpasse) grace a ID (timestamp), generation du mot de passe,
	 * enregistrement dans la base du nouveau mot de passe avec eventuellement le nouvel email
	 * 
	 * @param id
	 *            the id
	 * @param ctx
	 *            the ctx
	 * 
	 * @return le mot de passe
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public HashMap<String, String> fabrication(final String id, final OMContext ctx) throws Exception {
		utilisateur = new Utilisateur();
		utilisateur.setCtx(ctx);
		utilisateur.init();
		HashMap<String, String> presentation = new HashMap<>();
		if (id == null) {
			return presentation;
		}
		presentation = this.recuperationDEMANDE_MDP(id);
		if (StringUtils.isEmpty(presentation.get("code"))) {
			return presentation;
		}
		final String motDePasse = this.generationMDP();
		presentation.put("motDePasse", motDePasse);
		final String code = presentation.get("code");
		final String email = presentation.get("email");
		this.enregistrementUtilisateur(code, email, motDePasse);
		this.suppressionTouteDEMANDE_MDP(code);
		return presentation;
	}//fabrication

	/**
	 * Enregistrement demand e_ mdp.
	 * 
	 * @param code
	 *            the code
	 * @param email
	 *            the email
	 * 
	 * @return the string
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private String enregistrementDEMANDE_MDP(String code, String email) throws Exception {
		//evite injection SQL
		code = EscapeString.escapeSql(code);
		email = EscapeString.escapeSql(email);
		final Changementmotpasse chgMdp = new Changementmotpasse();
		try (ContexteDao ctx = new ContexteDao()) {
			chgMdp.setCtx(ctx);
			chgMdp.init();
			chgMdp.setCode(code);
			chgMdp.setEmail(email);
			chgMdp.setId(this.generationID());
			chgMdp.setDateDemande(new Date(System.currentTimeMillis()));
			chgMdp.add();
		}
		return chgMdp.getId();
	}

	/**
	 * supprime ds la base une demande en fct de son id @ id.
	 * 
	 * @param id
	 *            the id
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private void suppressionDEMANDE_MDP(final String id) throws Exception {
		final Changementmotpasse chgMdp = new Changementmotpasse();
		try (ContexteDao ctx = new ContexteDao()) {
			chgMdp.setCtx(ctx);
			chgMdp.init();
			final ClauseWhere whereID = new ClauseWhere(ConditionHelper.egalVarchar("ID", id));
			if (chgMdp.select(whereID.formaterSQL()) > 0) {
				while (chgMdp.nextItem()) {
					chgMdp.delete();
				}
			}
		}
	}

	/**
	 * supprime toutes les demandes pour un utilisateur.
	 * 
	 * @param code
	 *            the code
	 * 
	 * @throws Exception
	 *             the exception
	 * 
	 * @author root
	 */
	private void suppressionTouteDEMANDE_MDP(final String code) throws Exception {
		final Changementmotpasse chgMdp = new Changementmotpasse();
		try (ContexteDao ctx = new ContexteDao()) {
			chgMdp.setCtx(ctx);
			chgMdp.init();
			final ClauseWhere whereCode = new ClauseWhere(ConditionHelper.egalVarchar("code", code));
			if (chgMdp.select(whereCode.formaterSQL()) > 0) {
				while (chgMdp.nextItem()) {
					chgMdp.delete();
				}
			}
		}
	}

	/**
	 * Enregistrement utilisateur.
	 * 
	 * @param code
	 *            the code
	 * @param email
	 *            the email
	 * @param motDePasse
	 *            the mot de passe
	 * @throws Exception
	 *             the exception
	 */
	private void enregistrementUtilisateur(final String code, final String email, final String motDePasse) throws Exception {
		final Utilisateur utilisateur = new Utilisateur();
		try (ContexteDao ctx = new ContexteDao()) {
			utilisateur.setCtx(ctx);
			utilisateur.init();
			final ClauseWhere whereCode = new ClauseWhere(ConditionHelper.egalVarchar("code", code));
			if (utilisateur.select(whereCode.formaterSQL()) > 0) {
				if (utilisateur.nextItem()) {
					utilisateur.setMotDePasse(MySQLHelper.encodePassword(motDePasse));
					if ((utilisateur.getAdresseMail().length() == 0) && (!email.equals(""))) {
						utilisateur.setAdresseMail(email);
					}
					utilisateur.update();
				}
			}
		}
	}

	/**
	 * Envoi email demand e_ mdp.
	 * 
	 * @param id
	 *            the id
	 * @param to
	 *            the to
	 * @param ctx
	 *            the ctx
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private void envoiEmailDEMANDE_MDP(final String id, final String to, final OMContext ctx) throws Exception {
		final String universite = (ctx.getInfosSite() != null ? ctx.getInfosSite().getIntitule() : "K-Portal");
		final String sujet = universite + " " + MessageHelper.getCoreMessage(ctx.getLocale(), "ST_OBJET_DSI_CONNEXION_INTRANET");
		final String url = WebAppUtil.SG_PATH + "?PROC=IDENTIFICATION_FRONT&ACTION=PRESENTER_MDP&ID=" + id;
		String message = MessageHelper.getCoreMessage(ctx.getLocale(), "ST_CLIQUEZ_LIEN");
		message += URLResolver.getAbsoluteUrl(url, ctx);
		message += MessageHelper.getCoreMessage(ctx.getLocale(), "ST_EQUIPE_WEB") + universite;
		final JSBMailbox mailbox = new JSBMailbox(false);
		try {
			mailbox.sendSystemMsg(to, sujet, message);
		} catch (final Exception e) {
			this.suppressionDEMANDE_MDP(id);
			throw e;
		}
	}

	/**
	 * Generation mdp.
	 * 
	 * @return the string
	 */
	private String generationMDP() {
		final String charMin = "abcdefghijklmnopqrstuvwxyz";
		final String charMaj = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
		final String charNb = "0123456789";
		final String charTout = charMin + charMaj + charNb;
		String mdp = "";
		for (int i = 0; i < 7; i++) {
			mdp = mdp.concat("" + charTout.charAt((int) ((Math.random() * 1000) % charTout.length())));
		}//for
		return mdp;
	}//generationMDP

	/**
	 * Generation id.
	 * 
	 * @return the string
	 */
	private String generationID() {
		final String charMin = "abcdefghijklmnopqrstuvwxyz";
		final String charMaj = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
		final String charNb = "0123456789";
		final String charTout = charMin + charMaj + charNb;
		String id = "";
		for (int i = 0; i < 8; i++) {
			id = id.concat("" + charTout.charAt((int) ((Math.random() * 1000) % charTout.length())));
		}//for
		return id;
	}

	private HashMap<String, String> recuperationDEMANDE_MDP(final String id) throws Exception {
		final HashMap<String, String> demande = new HashMap<>();
		final Changementmotpasse chgMdp = new Changementmotpasse();
		try (ContexteDao ctx = new ContexteDao()) {
			chgMdp.setCtx(ctx);
			chgMdp.init();
			final ClauseWhere whereID = new ClauseWhere(ConditionHelper.egalVarchar("ID", id));
			if (chgMdp.select(whereID.formaterSQL()) > 0) {
				while (chgMdp.nextItem()) {
					final ClauseWhere whereCode = new ClauseWhere(ConditionHelper.egalVarchar("code", chgMdp.getCode()));
					if (utilisateur.select(whereCode.formaterSQL()) > 0) {
						utilisateur.nextItem();
						demande.put("code", chgMdp.getCode());
						demande.put("email", chgMdp.getEmail());
						demande.put("nom", utilisateur.getNom());
						demande.put("prenom", utilisateur.getPrenom());
					}
				}
			}
		}
		return demande;
	}
}
