/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.processus;

import java.sql.Date;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.univ.objetspartages.om.Article;
import com.univ.objetspartages.om.Libelle;

/**
 * Processus de gestion des articles.
 */
public class SaisieArticle extends SaisieFiche {

	/** The article. */
	private Article article = null;

	/**
	 * Constructeur.
	 * 
	 * @param infoBean
	 *            the info bean
	 */
	public SaisieArticle(final InfoBean infoBean) {
		super(infoBean);
	}

	/**
	 * Point d'entree du processus.
	 * 
	 * @return true, if traiter action
	 * 
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public boolean traiterAction() throws Exception {
		infoBean.set("CODE_OBJET", "0015");
		article = new Article();
		article.init();
		traiterActionParDefaut(article);
		infoBean.set("NOM_ONGLET", "article");
		// on continue si on n'est pas a la FIN !!!
		return (etat == FIN);
	}

	/**
	 * Affichage de l'ecran des criteres de recherche d'un(e) article.
	 * 
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected void preparerRECHERCHE() throws Exception {
		ecranLogique = ECRAN_RECHERCHE;
		infoBean.set("LISTE_LANGUES", LangueUtil.getListeLangues(getLocale()));
		infoBean.set("LISTE_THEMATIQUES", Libelle.getListe("04", getLocale()));
	}

	/**
	 * Affichage de l'ecran de saisie d'un(e) article.
	 * 
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected void preparerPRINCIPAL() throws Exception {
		ecranLogique = ECRAN_PRINCIPAL;
		infoBean.set("ID_ARTICLE", article.getIdFiche().toString());
		infoBean.set("TITRE", article.getTitre());
		infoBean.set("CHAPEAU", article.getChapeau());
		infoBean.set("CORPS", article.getCorps());
		infoBean.set("THEMATIQUE", article.getThematique());
		infoBean.set("LIBELLE_THEMATIQUE", article.getLibelleThematique());
		infoBean.set("LISTE_THEMATIQUES", Libelle.getListe("04", LangueUtil.getLocale(article.getLangue())));
		if (Formateur.estSaisie(article.getDateArticle())) {
			infoBean.set("DATE_ARTICLE", article.getDateArticle());
		} else {
			infoBean.set("DATE_ARTICLE", new Date(System.currentTimeMillis()));
		}
		infoBean.set("SOUS_TITRE", article.getSousTitre());
		if (1000 == article.getOrdre()) {
			infoBean.set("ORDRE", "");
		} else {
			infoBean.set("ORDRE", article.getOrdre().toString());
		}
		infoBean.set("CODE_RUBRIQUE", article.getCodeRubrique());
		infoBean.set("LIBELLE_CODE_RUBRIQUE", article.getInfosRubrique().getLibelleAffichable());
		infoBean.set("CODE_RATTACHEMENT", article.getCodeRattachement());
		infoBean.set("LIBELLE_CODE_RATTACHEMENT", article.getInfosStructure().getLibelleAffichable());
		infoBean.set("LIBELLE_AFFICHABLE", article.getLibelleAffichable());
		infoBean.set("SOUS_ONGLET", "PRINCIPAL"); // onglet par defaut
		ControleurUniv.preparerPRINCIPAL(infoBean, article, this);
	}

	/**
	 * Traite l'ecran de saisie d'une fiche.
	 * 
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected void traiterPRINCIPAL() throws Exception {
		// chargement de l'objet
		if (InfoBean.ETAT_OBJET_CREATION.equals(infoBean.getEtatObjet())) {
			article.init();
		} else {
			article.setIdFiche(new Long(infoBean.getString("ID_ARTICLE")));
			article.retrieve();
		}
		// changement d'onglet
		if (InfoBean.ACTION_ONGLET.equals(action)) {
			infoBean.set("SOUS_ONGLET", infoBean.getString("SOUS_ONGLET_DEMANDE"));
		}
		// validation de l'ecran
		else if (InfoBean.ACTION_ENREGISTRER.equals(action)) {
			alimenteDonneesCreation(article, false);
			article.setTitre((String) infoBean.get("TITRE"));
			article.setThematique((String) infoBean.get("THEMATIQUE"));
			article.setChapeau(infoBean.getString("CHAPEAU"));
			article.setSousTitre((String) infoBean.get("SOUS_TITRE"));
			if (Formateur.estSaisie((Date) infoBean.get("DATE_ARTICLE"))) {
				article.setDateArticle((Date) infoBean.get("DATE_ARTICLE"));
			}
			article.setCorps((String) infoBean.get("CORPS"));
			if (StringUtils.isNotBlank(infoBean.getString("ORDRE"))) {
				try {
					final Integer ordreArticle = new Integer(infoBean.getString("ORDRE"));
					article.setOrdre(ordreArticle);
				} catch (final Exception e) {
					throw new ErreurApplicative("Format de l'ordre incorrect");
				}
			} else {
				article.setOrdre(1000); // 1000 par defaut
			}
			if (infoBean.get("CODE_RUBRIQUE") != null && !"0000".equals(infoBean.getString("CODE_RUBRIQUE"))) {
				article.setCodeRubrique(infoBean.getString("CODE_RUBRIQUE"));
			} else {
				article.setCodeRubrique("");
			}
			article.setCodeRattachement(infoBean.getString("CODE_RATTACHEMENT"));
		}
		// appel au traitement general
		ecranLogique = ControleurUniv.traiterPRINCIPAL(infoBean, article, this);
		if (ecranLogique.length() == 0) {
			etat = FIN;
		}
	}
}
