/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.processus;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.text.StrTokenizer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.ProcessusBean;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.jsbsoft.jtf.upload.UploadedFile;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.kportal.extension.module.composant.ComposantLibelle;
import com.univ.datagrid.processus.LibelleDatagrid;
import com.univ.datagrid.utils.DatagridUtils;
import com.univ.objetspartages.cache.CacheLibelleManager;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Libelle;
import com.univ.objetspartages.util.CritereRecherche;
import com.univ.objetspartages.util.CritereRechercheUtil;
import com.univ.utils.Chaine;
import com.univ.utils.RechercheFicheHelper;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * Processus de saisie d'un libellé.
 */
public class SaisieLibelle extends ProcessusBean {

	private static final Logger LOG = LoggerFactory.getLogger(SaisieLibelle.class);

	private final CacheLibelleManager cache = (CacheLibelleManager) ApplicationContextManager.getCoreContextBean(CacheLibelleManager.ID_BEAN);

	/** The mode. */
	private int mode = -1;

	/** The Constant MODE_AJOUT. */
	private static final int MODE_AJOUT = 0;

	/** The Constant MODE_MODIFICATION. */
	private static final int MODE_MODIFICATION = 1;

	/** The Constant ECRAN_PRINCIPAL. */
	private static final String ECRAN_PRINCIPAL = "PRINCIPAL";

	private static final String ECRAN_LISTE = "LISTE";

	/** The Constant ECRAN_IMPORT. */
	private static final String ECRAN_IMPORT = "SAISIE_IMPORT";

	/** The Constant ECRAN_RESULTAT. */
	private static final String ECRAN_RESULTAT = "RESULTAT";

	private static final String ECRAN_RECHERCHE = "RECHERCHE";

	/** The libelle. */
	private Libelle libelle = null;

	/**
	 * Constructeur.
	 *
	 * @param ciu
	 *            com.jsbsoft.jtf.core.InfoBean
	 */
	public SaisieLibelle(final InfoBean ciu) {
		super(ciu);
	}

	/**
	 * Point d'entrée du processus.
	 *
	 * @return true, if traiter action
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public boolean traiterAction() throws Exception {
		try {
			final AutorisationBean autorisations = (AutorisationBean) getGp().getSessionUtilisateur().getInfos().get(SessionUtilisateur.AUTORISATIONS);
			if (!ComposantLibelle.isAutoriseParActionProcessus(autorisations, null)) {
				if (autorisations == null) {
					infoBean.setEcranRedirection(WebAppUtil.CONNEXION_BO);
					infoBean.setEcranLogique("LOGIN");
				} else {
					throw new ErreurApplicative(MessageHelper.getCoreMessage("BO_OPERATION_INTERDITE"));
				}
			} else {
				ecranLogique = infoBean.getEcranLogique();
				action = infoBean.getActionUtilisateur();
				if (etat == DEBUT) {
					initialisation();
				} else {
					if (ecranLogique.equals(ECRAN_RECHERCHE)) {
						traiterRECHERCHE();
					}
					if (ecranLogique.equals(ECRAN_PRINCIPAL)) {
						traiterPRINCIPAL();
					} else if (ecranLogique.equals(ECRAN_IMPORT)) {
						traiterIMPORT();
					}
				}
				//placer l'état dans le composant d'infoBean
				infoBean.setEcranLogique(ecranLogique);
			}
		} catch (final Exception e) {
			LOG.error("erreur lors du traitement des libelles", e);
			infoBean.addMessageErreur(e.toString());
		}
		// On continue si on n'est pas à la FIN !!!
		return (etat == FIN);
	}

	/**
	 * Initialisation du processus.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void initialisation() throws Exception {
		// initialisation objets métiers
		libelle = new Libelle();
		libelle.setCtx(this);
		libelle.init();
		etat = EN_COURS;
		if (action.equals(InfoBean.ACTION_AJOUTER)) {
			mode = MODE_AJOUT;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_CREATION);
			preparerPRINCIPAL();
		} else if (action.equals(InfoBean.ACTION_MODIFIER)) {
			mode = MODE_MODIFICATION;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
			traiterRECHERCHE();
		} else if (action.equals(InfoBean.ACTION_RECHERCHER)) {
			mode = MODE_MODIFICATION;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
			preparerRECHERCHE();
		} else if (action.equals("IMPORTER")) {
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_CREATION);
			ecranLogique = ECRAN_IMPORT;
		} else if (action.equals("MODIFIERPARID")) {
			mode = MODE_MODIFICATION;
			retrouveLibelleDepuisInfoBean();
			preparerPRINCIPAL();
		} else if (action.equals("SUPPRIMERPARID")) {
			retrouveLibelleDepuisInfoBean();
			supprimerLibelle();
			preparerPRINCIPAL();
			etat = FIN;
			cache.flushListeInfosLibelles();
		}
	}

	private void preparerRECHERCHE() {
		ecranLogique = ECRAN_RECHERCHE;
		infoBean.set("LISTE_TYPES_LIBELLES", cache.getListeTypesLibelles());
		final Hashtable<String, String> langues = LangueUtil.getListeLangues(getLocale());
		infoBean.set("LISTE_LANGUES", langues);
	}

	private void retrouveLibelleDepuisInfoBean() throws Exception {
		final String indice = infoBean.getString("ID_LIBELLE");
		libelle.init();
		libelle.setIdLibelle(Long.valueOf(indice));
		libelle.retrieve();
	}

	/**
	 * Traitement associé à l'écran de saisie des critères. LIBELLE CODE TYPE LANGUE
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterRECHERCHE() throws Exception {
		final List<CritereRecherche> criteres = new ArrayList<>();
		CollectionUtils.addIgnoreNull(criteres, CritereRechercheUtil.getCritereTexteNonVide(infoBean, "LIBELLE"));
		CollectionUtils.addIgnoreNull(criteres, CritereRechercheUtil.getCritereTexteNonVide(infoBean, "CODE"));
		CollectionUtils.addIgnoreNull(criteres, CritereRechercheUtil.getCritereTexteNonVideSansValeurDefaut(infoBean, "LANGUE"));
		final String type = infoBean.getString("TYPE");
		if (StringUtils.isNotBlank(type) && !"0000".equals(type)) {
			criteres.add(new CritereRecherche("TYPE", type, cache.getListeTypesLibelles().get(type)));
		}
		criteres.add(new CritereRecherche(DatagridUtils.PARAM_BEAN_DATAGRID, LibelleDatagrid.ID_BEAN));
		infoBean.set(RechercheFicheHelper.ATTRIBUT_INFOBEAN_CRITERES, criteres);
		ecranLogique = ECRAN_LISTE;
	}

	/**
	 * Affichage de l'écran de saisie d'un Libelle.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerPRINCIPAL() throws Exception {
		ecranLogique = ECRAN_PRINCIPAL;
		infoBean.set("ACTION_DEMANDEE", action);
		infoBean.set("TYPE", libelle.getType());
		infoBean.set("LISTE_TYPES_LIBELLES", cache.getListeTypesLibelles());
		if (mode == MODE_AJOUT) {
			libelle.setCode(determinerCode());
		}
		infoBean.set("CODE", libelle.getCode());
		// infoBean.set("LIBELLE", libelle.getLibelle());
		final Hashtable<String, String> langues = LangueUtil.getListeLangues(getLocale());
		infoBean.set("LISTE_LANGUES", langues);
		for (final String codeLangue : langues.keySet()) {
			final Libelle libelle2 = new Libelle();
			libelle2.setCtx(this);
			final ClauseWhere where = new ClauseWhere(ConditionHelper.egalVarchar("TYPE", libelle.getType()));
			where.and(ConditionHelper.egalVarchar("CODE", libelle.getCode()));
			where.and(ConditionHelper.egalVarchar("LANGUE", codeLangue));
			libelle2.select(where.formaterSQL());
			libelle2.nextItem();
			infoBean.set("LIBELLE#" + codeLangue, libelle2.getLibelle() == null ? "" : libelle2.getLibelle());
		}
		final String langueInfoBean = infoBean.getString("LANGUE_LIBELLE");
		if (StringUtils.isNotBlank(langueInfoBean) && StringUtils.isNumeric(langueInfoBean)) {
			libelle.setLangue(langueInfoBean);
		}
		if (StringUtils.isBlank(libelle.getLangue())) {
			libelle.setLangue("0");
		}
		infoBean.set("LANGUE", libelle.getLangue());
		if (StringUtils.isNotBlank(libelle.getLibelleAffichable())) {
			infoBean.setTitreEcran(libelle.getLibelleAffichable());
		}
	}

	private void supprimerLibelle() throws Exception {
		final Libelle libelle2 = new Libelle();
		libelle2.setCtx(this);
		final ClauseWhere where = new ClauseWhere(ConditionHelper.egalVarchar("CODE", libelle.getCode()));
		where.and(ConditionHelper.egalVarchar("TYPE", libelle.getType()));
		while (libelle2.select(where.formaterSQL()) > 0) {
			libelle2.nextItem();
			libelle.setIdLibelle(libelle2.getIdLibelle());
			libelle.delete();
		}
		final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_SUPPRESSION_LIBELLE"), libelle.getCode());
		infoBean.addMessageConfirmation(confirmation);
	}

	private void ajouterLibelle() throws Exception {
		Chaine.controlerCodeMetier((String) infoBean.get("CODE"));
		final Libelle libelle2 = new Libelle();
		libelle2.setCtx(this);
		final ClauseWhere where = new ClauseWhere(ConditionHelper.egalVarchar("CODE", infoBean.getString("CODE")));
		where.and(ConditionHelper.egalVarchar("TYPE", infoBean.getString("TYPE")));
		if (libelle2.select(where.formaterSQL()) > 0) {
			throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_LIBELLE.ERREUR.CODE_EXISTANT"));
		}
		final Hashtable<String, String> listeLangue = LangueUtil.getListeLangues(getLocale());
		int nbLibelleSaisie = 0;
		for (final String codeLangue : listeLangue.keySet()) {
			final String valeurLibelle = infoBean.getString("LIBELLE#" + codeLangue);
			if (StringUtils.isNotBlank(valeurLibelle)) {
				libelle.init();
				libelle.setType(infoBean.getString("TYPE"));
				libelle.setCode(infoBean.getString("CODE"));
				libelle.setLangue(codeLangue);
				libelle.setLibelle(valeurLibelle);
				libelle.add();
				nbLibelleSaisie++;
			}
		}
		if (nbLibelleSaisie == 0) {
			throw new ErreurApplicative(MessageHelper.getCoreMessage("ERREUR_SAISIE_SANS_LIBELLE"));
		}
		final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_CREATION_LIBELLE"), libelle.getCode());
		infoBean.addMessageConfirmation(confirmation);
	}

	private void mettreLibelleAJour() throws Exception {
		final Hashtable<String, String> listeLangue = LangueUtil.getListeLangues(getLocale());
		final Collection<Long> idsLibelleToDelete = new ArrayList<>();
		int nbLibelleSaisie = 0;
		for (final String codeLangue : listeLangue.keySet()) {
			final String valeurLibelle = infoBean.getString("LIBELLE#" + codeLangue);
			final ClauseWhere where = new ClauseWhere(ConditionHelper.egalVarchar("CODE", infoBean.getString("CODE")));
			where.and(ConditionHelper.egalVarchar("TYPE", infoBean.getString("TYPE")));
			where.and(ConditionHelper.egalVarchar("LANGUE", codeLangue));
			final Libelle libellePourRequete = new Libelle();
			libellePourRequete.setCtx(this);
			if (StringUtils.isNotBlank(valeurLibelle)) {
				libelle.init();
				libelle.setType(infoBean.getString("TYPE"));
				libelle.setCode(infoBean.getString("CODE"));
				libelle.setLangue(codeLangue);
				libelle.setLibelle(valeurLibelle);
				if (libellePourRequete.select(where.formaterSQL()) > 0) {
					libellePourRequete.nextItem();
					libelle.setIdLibelle(libellePourRequete.getIdLibelle());
					libelle.update();
				} else {
					libelle.add();
				}
				nbLibelleSaisie++;
			} else if (libellePourRequete.select(where.formaterSQL()) > 0) {
				libellePourRequete.nextItem();
				idsLibelleToDelete.add(libellePourRequete.getIdLibelle());
			}
		}
		if (nbLibelleSaisie == 0) {
			throw new ErreurApplicative(MessageHelper.getCoreMessage("ERREUR_SAISIE_SANS_LIBELLE"));
		} else {
			for (final Long idLibelle : idsLibelleToDelete) {
				libelle.setIdLibelle(idLibelle);
				libelle.delete();
			}
		}
		final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_MODIFICATION_LIBELLE"), libelle.getCode());
		infoBean.addMessageConfirmation(confirmation);
	}

	/**
	 * Gestion de l'écran de saisie d'un Libelle.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterPRINCIPAL() throws Exception {
		if (action == InfoBean.ACTION_SUPPRIMER) {
			supprimerLibelle();
		} else {
			verifierLibelle();
			if (mode == MODE_AJOUT) {
				ajouterLibelle();
			} else if (mode == MODE_MODIFICATION) {
				mettreLibelleAJour();
			}
		}
		etat = FIN;
		cache.flushListeInfosLibelles();
	}

	private void verifierLibelle() throws Exception {
		final Hashtable<String, String> listeLangue = LangueUtil.getListeLangues(getLocale());
		for (final String codeLangue : listeLangue.keySet()) {
			final String lib = infoBean.getString("LIBELLE#" + codeLangue);
			if (lib.indexOf(",") > -1 || lib.indexOf(";") > -1) {
				throw new ErreurApplicative(MessageHelper.getCoreMessage(getLocale(), "SAISIE_LIBELLE.ERREUR.REMPLACEMENT_CARACTERE"));
			}
		}
	}

	/**
	 * Détermine un code libellé aléatoire.
	 *
	 * @return the string
	 */
	private String determinerCode() {
		final String chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
		final StringBuffer code = new StringBuffer();
		int pos;
		for (int i = 0; i < 8; i++) {
			pos = (int) (Math.random() * chars.length());
			if (pos >= chars.length()) {
				pos = 0;
			}
			code.append(chars.charAt(pos));
		}
		return code.toString();
	}

	/**
	 * Gestion de l'écran de saisie d'un Libelle.
	 * Cette méthode est appelée lors de l'import des libellés.
	 * Les données importées doivent respecter le modèle suivant ;'typeLibelle','codelibelle','libelle','languelibelle'. Le delimiteur utilisé pour parser les données est ',' (cote virgule cote).
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterIMPORT() throws Exception {
		if (action.equals(InfoBean.ACTION_VALIDER)) {
			final UploadedFile file = (UploadedFile) infoBean.get("LIBELLE_FICHIER_FILE");
			File f = null;
			if ((file != null) && (file.getContentFilename().length() != 0)) {
				f = file.getTemporaryFile();
			} else {
				throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_LIBELLE.ERREUR.FICHIER_INEXISTANT"));
			}
			final FileReader freader = new FileReader(f);
			try (final BufferedReader fichLogique = new BufferedReader(freader)) {
				String ligneLue = fichLogique.readLine();
				int nbLibelle = 0;
				int nbLigne = 0;
				List<Integer> libellesKo = new ArrayList<>();
				while (ligneLue != null) {
					nbLigne++;
					String typeLibelle = StringUtils.EMPTY;
					String code = StringUtils.EMPTY;
					String libelle = StringUtils.EMPTY;
					String langue = StringUtils.EMPTY;
					final StrTokenizer st = StrTokenizer.getCSVInstance(ligneLue.substring(1,ligneLue.length() - 1));
					st.setIgnoreEmptyTokens(false);
					st.setDelimiterString("','");
					int ind = 0;
					while (st.hasNext()) {
						final String item = st.nextToken();
						if (ind == 0) {
							typeLibelle = item;
						}
						if (ind == 1) {
							code = item;
						}
						if (ind == 2) {
							libelle = item;
						}
						if (ind == 3) {
							langue = item;
						}
						ind++;
					}
					if ((typeLibelle.length() > 0) && (code.length() > 0) && (libelle.length() > 0) && (langue.length() > 0)) {
						nbLibelle++;
						Libelle.addLibelle(this, typeLibelle, code, libelle, langue);
					} else {
						libellesKo.add(nbLigne);
					}
					ligneLue = fichLogique.readLine();
				}
				cache.flushListeInfosLibelles();
				String resultatImport;
				if(nbLibelle > 0) {
					if (nbLibelle == 1) {
						resultatImport = MessageHelper.getCoreMessage("BO_LIBELLE_RESULTAT_IMPORT_LIBELLE");
					} else {
						resultatImport = MessageHelper.getCoreMessage("BO_LIBELLE_RESULTAT_IMPORT_LIBELLES");
						resultatImport = String.format(resultatImport, nbLibelle);
					}
					infoBean.addMessageConfirmation(resultatImport);
					traiterRECHERCHE();
				}
				if(CollectionUtils.isNotEmpty(libellesKo)) {
					infoBean.addMessageAlerte(MessageHelper.getCoreMessage("BO_LIBELLE_RESULTAT_KO") + " [" + StringUtils.join(libellesKo, "; ") + "]");
					ecranLogique = ECRAN_IMPORT;
				}
			} catch (final Exception e) {
				LOG.error("Format du fichier d'import des libellés invalide", e);
				infoBean.addMessageErreur(MessageHelper.getCoreMessage("BO_LIBELLE_RESULTAT_ERREUR"));
				infoBean.setEtatObjet(InfoBean.ETAT_OBJET_CREATION);
				ecranLogique = ECRAN_IMPORT;
			}
		}
	}

	/**
	 * Enleve les quotes de debut et fin de chaine.
	 *
	 * @param chaine
	 *            the chaine
	 *
	 * @return the string
	 */
	private String enleverQuote(final String chaine) {
		String res = chaine.trim();
		if (res.charAt(0) == '\'') {
			res = res.substring(1);
		}
		if (res.charAt(res.length() - 1) == '\'') {
			res = res.substring(0, res.length() - 1);
		}
		return res;
	}
}
