/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.processus;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;

import org.apache.commons.lang3.CharEncoding;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.database.ProcessusBean;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.jsbsoft.jtf.upload.UploadedFile;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Groupedsi;
import com.univ.objetspartages.om.InfosPreferences;
import com.univ.objetspartages.om.Service;
import com.univ.objetspartages.om.ServiceBean;
import com.univ.portail.deploiment.Deploy;
import com.univ.portail.deploiment.DeployWarService;
import com.univ.portail.deploiment.util.PlutoAdminContext;
import com.univ.utils.Chaine;
import com.univ.utils.FileUtil;
import com.univ.utils.ServicesUtil;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * processus saisie Preferences.
 */
public class SaisiePreferences extends ProcessusBean {

	/** The autorisations. */
	private AutorisationBean autorisations = null;

	/** The dir delim. */
	private static String dirDelim = System.getProperty("file.separator");

	/** The Constant ECRAN_ORDRE. */
	private static final String ECRAN_ORDRE = "ORDRE";

	/** The Constant ECRAN_SERVICE. */
	private static final String ECRAN_SERVICE = "SERVICE";

	/** The Constant ECRAN_PREFS_GENERALES. */
	private static final String ECRAN_PREFS_GENERALES = "PREFS_GENERALES";

	/** The Constant ECRAN_DEPLOIEMENT. */
	private static final String ECRAN_DEPLOIEMENT = "DEPLOIEMENT";

	/** The Constant ECRAN_RESULTAT. */
	private static final String ECRAN_RESULTAT = "RESULTAT";

	/** The Constant ECRAN_RECHERCHE. */
	private static final String ECRAN_RECHERCHE = "RECHERCHE";

	/** The Constant ECRAN_LISTE. */
	private static final String ECRAN_LISTE = "LISTE";

	/** The service bean. */
	private ServiceBean serviceBean = null;

	/** The mode. */
	private int mode = -1;

	/** The Constant MODE_AJOUT. */
	private static final int MODE_AJOUT = 0;

	/** The Constant MODE_MODIFICATION. */
	private static final int MODE_MODIFICATION = 1;

	/** The Constant MODE_SUPPRESSION. */
	private static final int MODE_SUPPRESSION = 2;

	private static org.slf4j.Logger LOG = LoggerFactory.getLogger(SaisiePreferences.class);

	/**
	 * Processus SaisiePreferences
	 *
	 * @param ciu
	 *            com.jsbsoft.jtf.core.InfoBean
	 */
	public SaisiePreferences(final InfoBean ciu) {
		super(ciu);
	}

	/**
	 * Initialisation du processus.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void initialisation() throws Exception {
		etat = EN_COURS;
		if (action.equals("DEPLOYER")) {
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_CREATION);
			preparerDEPLOIEMENT();
		} else if (action.equals("AJOUTER")) {
			mode = MODE_AJOUT;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_CREATION);
			serviceBean = new ServiceBean();
			preparerSERVICE();
		} else if (action.equals("MODIFIER")) {
			mode = MODE_MODIFICATION;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
			preparerRECHERCHE();
		} else if (action.equals("MODIFIERPARCODE")) {
			mode = MODE_MODIFICATION;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
			traiterRECHERCHE();
		} else if (action.equals("SUPPRIMER")) {
			mode = MODE_SUPPRESSION;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_SUPPRESSION);
			preparerRECHERCHE();
		} else if (action.equals("SUPPRIMERPARCODE")) {
			mode = MODE_SUPPRESSION;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_SUPPRESSION);
			final ServiceBean serviceASupprimer = ServicesUtil.getService(infoBean.getString("CODE_SERVICE"));
			ServicesUtil.removeInstanceService(this, infoBean.getString("CODE_SERVICE"));
			ServicesUtil.getCacheServiceManager().asyncRefresh();
			final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_SUPPRESSION_SERVICE"), serviceASupprimer.getIntitule());
			infoBean.addMessageConfirmation(confirmation);
			etat = FIN;
		} else if (action.equals("ORDONNER")) {
			preparerORDRE();
		}
	}

	/**
	 * Affichage de l'écran des critères de recherche d'un Role.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerRECHERCHE() throws Exception {
		ecranLogique = ECRAN_RECHERCHE;
		infoBean.set("CODE_SERVICE", "");
		final Hashtable<String, String> listeServices = new Hashtable<String, String>();
		for (final ServiceBean service : ServicesUtil.getServices().values()) {
			listeServices.put(service.getCode(), service.getIntitule());
		}
		infoBean.set("LISTE_SERVICES", listeServices);
		infoBean.set("SERVICES", ServicesUtil.getServices().values());
		infoBean.setTitreEcran(MessageHelper.getCoreMessage("BO.SAISIE_PREFERENCES.SERVICES.TITRE"));
	}

	/**
	 * Affichage de la liste des Roles.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerLISTE() throws Exception {
		ecranLogique = ECRAN_LISTE;
		infoBean.set("CODE#0", serviceBean.getCode());
		infoBean.set("LIBELLE#0", serviceBean.getIntitule());
		infoBean.setInt("LISTE_NB_ITEMS", 1);
	}

	/**
	 * Affichage de l'écran de déploiement.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerDEPLOIEMENT() throws Exception {
		ecranLogique = ECRAN_DEPLOIEMENT;
	}

	/**
	 * Affichage de l'écran de saisie des préférences
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerPREFS_GENERALES() throws Exception {
		infoBean.setInt("PREFERENCES_NB_ITEMS", 0);
		infoBean.set("PREFERENCE_NOM", "");
		infoBean.set("PREFERENCE_VALEUR", "");
		infoBean.set("TMP_PREFERENCE_VALEUR", "");
		infoBean.set("LIBELLE_PREFERENCE_VALEUR", "");
		infoBean.set("PREFERENCES_NB_ITEMS", infoBean.get("SERVICE_PREFERENCES_NB_ITEMS"));
		final int nbPreferences = infoBean.getInt("PREFERENCES_NB_ITEMS");
		for (int k = 0; k < nbPreferences; k++) {
			infoBean.set("PREFERENCES_NOM#" + k, infoBean.get("SERVICE_PREFERENCES_NOM#" + k));
			infoBean.set("PREFERENCES_VALEUR#" + k, infoBean.get("SERVICE_PREFERENCES_VALEUR#" + k));
		}
		ecranLogique = ECRAN_PREFS_GENERALES;
	}

	/**
	 * Affichage de l'écran de saisie des préférences
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerORDRE() throws Exception {
		ecranLogique = ECRAN_ORDRE;
		/* intialisation services */
		final Collection<ServiceBean> vServices = ServicesUtil.getServices().values();
		infoBean.setInt("SERVICES_NB_ITEMS", vServices.size());
		int i = 0;
		for (final ServiceBean service : vServices) {
			infoBean.set("SERVICE_CODE#" + i, service.getCode());
			infoBean.set("SERVICE_INTITULE#" + i, service.getIntitule());
			i++;
		}
		rafraichirORDRE();
	}

	/**
	 * Données à recalculer à chaque affichage de page page principale.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void rafraichirORDRE() throws Exception {
		ecranLogique = ECRAN_ORDRE;
	}

	/**
	 * Point d'entrée du processus.
	 *
	 * @return true, if traiter action
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public boolean traiterAction() throws Exception {
		autorisations = (AutorisationBean) getGp().getSessionUtilisateur().getInfos().get(SessionUtilisateur.AUTORISATIONS);
		if (autorisations == null) {
			infoBean.setEcranRedirection(WebAppUtil.CONNEXION_BO);
			infoBean.setEcranLogique("LOGIN");
		} else {
			try {
				ecranLogique = infoBean.getEcranLogique();
				action = infoBean.getActionUtilisateur();
				if (etat == DEBUT) {
					initialisation();
				} else {
					if (ecranLogique.equals(ECRAN_ORDRE)) {
						traiterORDRE();
					} else if (ecranLogique.equals(ECRAN_RECHERCHE)) {
						traiterRECHERCHE();
					} else if (ecranLogique.equals(ECRAN_SERVICE)) {
						traiterSERVICE();
					} else if (ecranLogique.equals(ECRAN_PREFS_GENERALES)) {
						traiterPREFS_GENERALES();
					} else if (ecranLogique.equals(ECRAN_DEPLOIEMENT)) {
						traiterDEPLOIEMENT();
					}
				}
				// placer l'état dans le composant d'infoBean
				infoBean.setEcranLogique(ecranLogique);
			} catch (final Exception e) {
				LOG.error("erreur de traitement sur le processus", e);
				infoBean.addMessageErreur(e.toString());
			}
		}
		// On continue si on n'est pas à la FIN !!!
		return (etat == FIN);
	}

	/**
	 * Traitement associé à l'écran de saisie des critères.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterRECHERCHE() throws Exception {
		serviceBean = ServicesUtil.getService(infoBean.getString("CODE_SERVICE"));
		if (mode == MODE_MODIFICATION) {
			preparerSERVICE();
		} else if (mode == MODE_SUPPRESSION) {
			preparerLISTE();
		}
	}

	/**
	 * Gestion de l'écran d'ordre des services.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterORDRE() throws Exception {
		if (action.indexOf("REMONTER_SERVICE#") != -1) {
			/* Suppression de l'élément courant */
			final int indiceDiese = action.indexOf("#");
			final int indice = Integer.parseInt(action.substring(indiceDiese + 1));
			final String tmpCODE = infoBean.getString("SERVICE_CODE#" + indice);
			final String tmpINTITULE = infoBean.getString("SERVICE_INTITULE#" + indice);
			infoBean.set("SERVICE_CODE#" + indice, infoBean.getString("SERVICE_CODE#" + (indice - 1)));
			infoBean.set("SERVICE_INTITULE#" + indice, infoBean.getString("SERVICE_INTITULE#" + (indice - 1)));
			infoBean.set("SERVICE_CODE#" + (indice - 1), tmpCODE);
			infoBean.set("SERVICE_INTITULE#" + (indice - 1), tmpINTITULE);
		} else if (action.indexOf("BAISSER_SERVICE#") != -1) {
			/* Suppression de l'élément courant */
			final int indiceDiese = action.indexOf("#");
			final int indice = Integer.parseInt(action.substring(indiceDiese + 1));
			final String tmpCODE = infoBean.getString("SERVICE_CODE#" + indice);
			final String tmpINTITULE = infoBean.getString("SERVICE_INTITULE#" + indice);
			infoBean.set("SERVICE_CODE#" + indice, infoBean.getString("SERVICE_CODE#" + (indice + 1)));
			infoBean.set("SERVICE_INTITULE#" + indice, infoBean.getString("SERVICE_INTITULE#" + (indice + 1)));
			infoBean.set("SERVICE_CODE#" + (indice + 1), tmpCODE);
			infoBean.set("SERVICE_INTITULE#" + (indice + 1), tmpINTITULE);
		} else if (InfoBean.ACTION_VALIDER.equals(action)) {
			/* liste services */
			final List<String> vCodesServices = new ArrayList<>();
			for (int i = 0; i < infoBean.getInt("SERVICES_NB_ITEMS"); i++) {
				vCodesServices.add(infoBean.getString("SERVICE_CODE#" + i));
			}
			ServicesUtil.ordonnerInstancesServices(this, vCodesServices);
			ServicesUtil.getCacheServiceManager().asyncRefresh();
			etat = FIN;
		}
	}

	/**
	 * Affichage de l'écran de saisie d'un service.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerSERVICE() throws Exception {
		infoBean.set("ACTION_SERVICE", action);
		infoBean.set("SERVICE_CODE", serviceBean.getCode());
		infoBean.set("SERVICE_INTITULE", serviceBean.getIntitule());
		infoBean.set("SERVICE_TYPE", serviceBean.getType());
		if (serviceBean.getExpirationCacheService().equals(new Integer(-1))) {
			infoBean.set("SERVICE_EXPIRATION_CACHE", "");
		} else {
			infoBean.set("SERVICE_EXPIRATION_CACHE", serviceBean.getExpirationCacheService().toString());
		}
		infoBean.set("SERVICE_JETON_KPORTAL", serviceBean.getJetonKportal());
		infoBean.set("SERVICE_PROXY_CAS", serviceBean.getProxyCas());
		infoBean.set("SERVICE_URL", serviceBean.getUrl());
		infoBean.set("SERVICE_PORTLET_DEFINITION_ID", serviceBean.getPortletDefinitionId());
		infoBean.set("SERVICE_URL_POPUP", serviceBean.getUrlPopup());
		infoBean.set("SERVICE_VUE_REDUITE_TYPE", serviceBean.getVueReduiteType());
		infoBean.set("SERVICE_VUE_REDUITE_URL", serviceBean.getVueReduiteUrl());
		infoBean.set("SERVICE_VUE_REDUITE_CTX", serviceBean.getVueReduiteCtx());
		infoBean.set("SERVICE_VUE_REDUITE_URL_CROSS_CTX", serviceBean.getVueReduiteUrlCrossCtx());
		infoBean.set("SERVICE_DIFFUSION_MODE", serviceBean.getDiffusionMode());
		infoBean.set("SERVICE_DIFFUSION_PUBLIC_VISE", Chaine.convertirPointsVirgules(serviceBean.getDiffusionPublicVise()));
		infoBean.set("SERVICE_DIFFUSION_MODE_RESTRICTION", serviceBean.getDiffusionModeRestriction());
		infoBean.set("SERVICE_DIFFUSION_PUBLIC_VISE_RESTRICTION", Chaine.convertirPointsVirgules(serviceBean.getDiffusionPublicViseRestriction()));
		final List<InfosPreferences> listePreferences = serviceBean.getPreferencesGenerales();
		InfosPreferences infosPref = null;
		infoBean.setInt("SERVICE_PREFERENCES_NB_ITEMS", listePreferences.size());
		int k = 0;
		for (final Object element : listePreferences) {
			infosPref = (InfosPreferences) element;
			infoBean.set("SERVICE_PREFERENCES_NOM#" + k, infosPref.getNom());
			infoBean.set("SERVICE_PREFERENCES_VALEUR#" + k, infosPref.getValeursAsString());
			k++;
		}
		/*
		 * type de service U (URL) -> 0 P (PORTLET) -> 1
		 */
		infoBean.set("SERVICE_TYPE_SAISIE", "0");
		if (infoBean.get("SERVICE_TYPE").equals("U")) {
			infoBean.set("SERVICE_TYPE_SAISIE", "0");
		}
		if (infoBean.get("SERVICE_TYPE").equals("P")) {
			infoBean.set("SERVICE_TYPE_SAISIE", "1");
		}
		infoBean.set("LIBELLE_SERVICE_DIFFUSION_PUBLIC_VISE", Groupedsi.getIntitule(infoBean.getString("SERVICE_DIFFUSION_PUBLIC_VISE")));
		infoBean.set("LIBELLE_SERVICE_DIFFUSION_PUBLIC_VISE_RESTRICTION", Groupedsi.getIntitule(infoBean.getString("SERVICE_DIFFUSION_PUBLIC_VISE_RESTRICTION")));
		// 20060130 : FBI : filtre l'arbre des groupes en fonction des autorisations
		infoBean.set("GRS_FILTRE_ARBRE_GROUPE", "1");
		infoBean.set("GRS_FILTRE_ARBRE_GROUPE_TYPE", "TECH");
		infoBean.set("GRS_FILTRE_ARBRE_GROUPE_OBJET", "dsi");
		infoBean.set("GRS_FILTRE_ARBRE_GROUPE_ACTION", "");
		ecranLogique = ECRAN_SERVICE;
		if (StringUtils.isNotBlank(serviceBean.getIntitule())) {
			infoBean.setTitreEcran(serviceBean.getIntitule());
		}
	}

	/**
	 * Gestion de l'écran de saisie d'un service.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterSERVICE() throws Exception {
		if ((action.equals("MODIFIER_PREFS_GENERALES"))) {
			preparerPREFS_GENERALES();
		} else if (InfoBean.ACTION_VALIDER.equals(action)) {
			/* Controles */
			if (infoBean.getEtatObjet().equals(InfoBean.ETAT_OBJET_CREATION)) {
				if (infoBean.getString("SERVICE_CODE").length() == 0) {
					throw new ErreurApplicative("Le code du service est obligatoire");
				}
				final String serviceCode = infoBean.getString("SERVICE_CODE");
				// Vérifie que le code ne contient que des caractères alphanumériques ou '-' ou '_'
				Chaine.controlerCodeMetier(serviceCode);
				final ClauseWhere whereCode = new ClauseWhere(ConditionHelper.egalVarchar("CODE", serviceCode));
				/* On teste si le code existe déjà */
				final Service service = new Service();
				service.setCtx(this);
				service.init();
				if (service.select(whereCode.formaterSQL()) > 0) {
					throw new ErreurApplicative("Il existe déjà un service ayant ce code.");
				}
			}
			if (infoBean.getString("SERVICE_INTITULE").length() == 0) {
				throw new ErreurApplicative("L'intitulé du service est obligatoire");
			}
			if (infoBean.getString("SERVICE_INTITULE").indexOf(";") != -1) {
				throw new ErreurApplicative("Le service ne doit pas contrenir de point-virgule");
			}
			if (infoBean.getString("SERVICE_VUE_REDUITE_TYPE").equals("1")) {
				if (infoBean.getString("SERVICE_VUE_REDUITE_URL").length() == 0) {
					throw new ErreurApplicative("L'url de la vue réduite est obligatoire");
				}
			}
			if (infoBean.getString("SERVICE_VUE_REDUITE_TYPE").equals("2")) {
				if (infoBean.getString("SERVICE_VUE_REDUITE_URL_CROSS_CTX").length() == 0) {
					throw new ErreurApplicative("L'url cross-context de la vue réduite est obligatoire");
				}
			}
			Integer expirationCache = new Integer(-1);
			if (infoBean.getString("SERVICE_EXPIRATION_CACHE").length() > 0) {
				try {
					expirationCache = new Integer(infoBean.getString("SERVICE_EXPIRATION_CACHE"));
				} catch (final Exception e) {
					throw new ErreurApplicative("Format de la durée de cache incorrect");
				}
			}
			/*
			 * type de service U (URL) -> 0 P (PORTLET) -> 1
			 */
			if (infoBean.get("SERVICE_TYPE_SAISIE").equals("0")) {
				infoBean.set("SERVICE_TYPE", "U");
			}
			if (infoBean.get("SERVICE_TYPE_SAISIE").equals("1")) {
				infoBean.set("SERVICE_TYPE", "P");
			}
			/* récupération indice de la ligne dans InfoBean */
			InfosPreferences infoPref = null;
			List<InfosPreferences> listeInfosPref = null;
			String code = "";
			final String typeService = infoBean.getString("SERVICE_TYPE");
			// TODO prévoir de ne pas avoir à faire ce test..
			if (Service.TYPE_PORTLET.equals(typeService)) {
				code = infoBean.getString("SERVICE_PORTLET_DEFINITION_ID");
				if (StringUtils.isEmpty(code)) {
					throw new ErreurApplicative("L'id du portlet est obligatoire");
				}
			} else {
				code = infoBean.getString("SERVICE_CODE");
			}
			listeInfosPref = new ArrayList<InfosPreferences>();
			for (int k = 0; k < infoBean.getInt("SERVICE_PREFERENCES_NB_ITEMS"); k++) {
				// il s'agit d'une préférence générale, dc le code User de
				// la préférence est vide..
				infoPref = new InfosPreferences("", code, typeService, infoBean.getString("SERVICE_PREFERENCES_NOM#" + k), infoBean.getString("SERVICE_PREFERENCES_VALEUR#" + k));
				listeInfosPref.add(infoPref);
			}
			serviceBean = new ServiceBean(infoBean.getString("SERVICE_CODE"), infoBean.getString("SERVICE_INTITULE"), typeService, CharEncoding.UTF_8, expirationCache,
				expirationCache, infoBean.getString("SERVICE_JETON_KPORTAL"), infoBean.getString("SERVICE_PROXY_CAS"), infoBean.getString("SERVICE_URL"),
				infoBean.getString("SERVICE_PORTLET_DEFINITION_ID"), infoBean.getString("SERVICE_URL_POPUP"), infoBean.getString("SERVICE_VUE_REDUITE_TYPE"),
				infoBean.getString("SERVICE_VUE_REDUITE_URL"), infoBean.getString("SERVICE_VUE_REDUITE_CTX"), infoBean.getString("SERVICE_VUE_REDUITE_URL_CROSS_CTX"),
				infoBean.getString("SERVICE_DIFFUSION_MODE"), Chaine.getVecteurPointsVirgules(infoBean.getString("SERVICE_DIFFUSION_PUBLIC_VISE")),
				infoBean.getString("SERVICE_DIFFUSION_MODE_RESTRICTION"), Chaine.getVecteurPointsVirgules(infoBean.getString("SERVICE_DIFFUSION_PUBLIC_VISE_RESTRICTION")),
				listeInfosPref);
			ServicesUtil.setInstanceService(this, serviceBean);
			ServicesUtil.getCacheServiceManager().asyncRefresh();
			String messageConfirmation = MessageHelper.getCoreMessage("CONFIRMATION_MODIFICATION_SERVICE");
			if (infoBean.getEtatObjet().equals(InfoBean.ETAT_OBJET_CREATION)) {
				messageConfirmation = MessageHelper.getCoreMessage("CONFIRMATION_CREATION_SERVICE");
			}
			final String confirmation = String.format(messageConfirmation, serviceBean.getIntitule());
			infoBean.addMessageConfirmation(confirmation);
			etat = FIN;
		} else if (action.equals("SUPPRIMER")) {
			ServicesUtil.removeInstanceService(this, serviceBean.getCode());
			ServicesUtil.getCacheServiceManager().asyncRefresh();
			final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_SUPPRESSION_SERVICE"), serviceBean.getIntitule());
			infoBean.addMessageConfirmation(confirmation);
			etat = FIN;
		}
	}

	/**
	 * Gestion de l'écran de saisie d'un service.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterPREFS_GENERALES() throws Exception {
		/* récupération indice de la ligne dans InfoBean */
		int indice = 0;
		if (action.equals("AJOUTER_PREFERENCE")) {
			if (infoBean.getString("PREFERENCE_NOM").length() == 0) {
				throw new ErreurApplicative("Le nom doit etre renseigné");
			}
			indice = infoBean.getInt("PREFERENCES_NB_ITEMS");
			infoBean.setInt("PREFERENCES_NB_ITEMS", indice + 1);
			/* Mise à jour InfoBean */
			infoBean.set("PREFERENCES_NOM#" + indice, infoBean.getString("PREFERENCE_NOM"));
			infoBean.set("PREFERENCES_VALEUR#" + indice, infoBean.getString("PREFERENCE_VALEUR"));
			infoBean.set("PREFERENCE_NOM", "");
			infoBean.set("PREFERENCE_VALEUR", "");
			infoBean.set("TMP_PREFERENCE_VALEUR", "");
			infoBean.set("LIBELLE_PREFERENCE_VALEUR", "");
		} else if (action.indexOf("SUPPRIMER_PREFERENCE") != -1) {
			/* Suppression de l'élément courant */
			final int indiceDiese = action.indexOf("#");
			indice = Integer.parseInt(action.substring(indiceDiese + 1));
			for (int j = indice; j < infoBean.getInt("PREFERENCES_NB_ITEMS") - 1; j++) {
				infoBean.set("PREFERENCES_NOM#" + j, infoBean.get("PREFERENCES_NOM#" + (j + 1)));
				infoBean.set("PREFERENCES_VALEUR#" + j, infoBean.get("PREFERENCES_VALEUR#" + (j + 1)));
			}
			infoBean.setInt("PREFERENCES_NB_ITEMS", infoBean.getInt("PREFERENCES_NB_ITEMS") - 1);
		} else if (InfoBean.ACTION_VALIDER.equals(action)) {
			infoBean.setInt("SERVICE_PREFERENCES_NB_ITEMS", infoBean.getInt("PREFERENCES_NB_ITEMS"));
			for (int j = indice; j < infoBean.getInt("PREFERENCES_NB_ITEMS"); j++) {
				infoBean.set("SERVICE_PREFERENCES_NOM#" + j, infoBean.get("PREFERENCES_NOM#" + j));
				infoBean.set("SERVICE_PREFERENCES_VALEUR#" + j, infoBean.get("PREFERENCES_VALEUR#" + j));
			}
			ecranLogique = ECRAN_SERVICE;
		} else if (InfoBean.ACTION_REVENIR.equals(action)) {
			ecranLogique = ECRAN_SERVICE;
		}
	}

	/**
	 * Gestion de l'écran de déploiement.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterDEPLOIEMENT() throws Exception {
		if (action.equals(InfoBean.ACTION_VALIDER)) {
			final UploadedFile file = (UploadedFile) infoBean.get("PORTLET_WAR_FILE");
			String context = "";
			String serverFileName = "";
			String workDefaultDir = "";
			File f = null;
			String pathFichierDest = null;
			if ((file != null) && (file.getContentFilename().length() != 0)) {
				f = file.getTemporaryFile();
				workDefaultDir = WebAppUtil.getWorkDefaultPath();
				serverFileName = file.getContentFilename();
				pathFichierDest = workDefaultDir + File.separator + serverFileName;
				//Suppression de l'ancien .war si il existe
				new File(pathFichierDest).delete();
				// duplication du fichier source
				FileUtil.copierFichier(f, new File(pathFichierDest), true);
			} else {
				throw new ErreurApplicative("Fichier non renseigné");
			}
			final int index = serverFileName.indexOf(".war");
			if (index != -1) {
				context = serverFileName.substring(0, index);
			} else {
				context = serverFileName;
			}
			String deployDir = PlutoAdminContext.getDeploymentPath();
			if (!deployDir.endsWith(dirDelim)) {
				deployDir += dirDelim;
			}
			Deploy.deployArchive(deployDir, pathFichierDest);
			final DeployWarService dws = new DeployWarService();
			dws.updateWebXml(context);
			final String msg = "Déploiement effectué";
			ecranLogique = ECRAN_RESULTAT;
			infoBean.set("RESULTAT", msg);
			// suppression fichier temporaire
			f.delete();
		}
	}
}
