/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.processus;

import java.util.ArrayList;
import java.util.Collection;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.database.ProcessusBean;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.kportal.extension.module.composant.ComposantProfil;
import com.univ.objetspartages.cache.CacheProfilDsiManager;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Groupedsi;
import com.univ.objetspartages.om.Profildsi;
import com.univ.objetspartages.om.Rubrique;
import com.univ.utils.Chaine;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * processus saisie Profildsi.
 */
public class SaisieProfildsi extends ProcessusBean {

	private final CacheProfilDsiManager cache = (CacheProfilDsiManager) ApplicationContextManager.getCoreContextBean(CacheProfilDsiManager.ID_BEAN);

	/** The mode. */
	private int mode = -1;

	/** The Constant MODE_AJOUT. */
	private static final int MODE_AJOUT = 0;

	/** The Constant MODE_MODIFICATION. */
	private static final int MODE_MODIFICATION = 1;

	/** The Constant MODE_SUPPRESSION. */
	private static final int MODE_SUPPRESSION = 2;

	/** The Constant MODE_RECHERCHE. */
	private static final int MODE_RECHERCHE = 3;

	/** The Constant ECRAN_RECHERCHE. */
	private static final String ECRAN_RECHERCHE = "RECHERCHE";

	/** The Constant ECRAN_PRINCIPAL. */
	private static final String ECRAN_PRINCIPAL = "PRINCIPAL";

	/** The Constant ECRAN_LISTE. */
	private static final String ECRAN_LISTE = "LISTE";

	private static org.slf4j.Logger LOG = LoggerFactory.getLogger(SaisieProfildsi.class);

	/** The profildsi. */
	Profildsi profildsi = null;

	/** The liste identifiant. */
	Long listeIdentifiant[] = null;

	/**
	 * processus saisie Profildsi.
	 *
	 * @param ciu
	 *            com.jsbsoft.jtf.core.InfoBean
	 */
	public SaisieProfildsi(final InfoBean ciu) {
		super(ciu);
	}

	/**
	 * Initialisation du processus.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void initialisation() throws Exception {
		// initalisation objets métiers
		profildsi = new Profildsi();
		profildsi.setCtx(this);
		profildsi.init();
		etat = EN_COURS;
		if (action.equals("AJOUTER")) {
			mode = MODE_AJOUT;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_CREATION);
			preparerPRINCIPAL();
		}
		if (action.equals("MODIFIER")) {
			mode = MODE_MODIFICATION;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
			preparerRECHERCHE();
		}
		if (action.equals("SUPPRIMER")) {
			mode = MODE_SUPPRESSION;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_SUPPRESSION);
			preparerRECHERCHE();
		}
		if (action.equals("RECHERCHER")) {
			mode = MODE_RECHERCHE;
			infoBean.setEtatObjet(InfoBean.ETAT_OBJET_RECHERCHE);
			preparerRECHERCHE();
		}
		if (action.equals("MODIFIERPARID")) {
			modifierParId();
		}
		if (action.equals("SUPPRIMERPARID")) {
			supprimerParId();
		}
	}

	/**
	 * Affichage de la liste des Profildsis.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerLISTE() throws Exception {
		int i = 0;
		ecranLogique = ECRAN_LISTE;
		final ClauseWhere whereIdProfil = new ClauseWhere(ConditionHelper.egalVarchar("ID_PROFILDSI", infoBean.getString("ID_PROFILDSI")));
		final int count = profildsi.select(whereIdProfil.formaterSQL());
		final Collection<Profildsi> profilsSelectionnes = new ArrayList<>();
		listeIdentifiant = new Long[count];
		while (profildsi.nextItem()) {
			profilsSelectionnes.add(profildsi.clone());
			listeIdentifiant[i] = profildsi.getIdProfildsi();
			i++;
		}
		infoBean.set("LISTE_PROFILDSI", profilsSelectionnes);
	}

	/**
	 * Affichage de l'écran de saisie d'un Profildsi.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerPRINCIPAL() throws Exception {
		ecranLogique = ECRAN_PRINCIPAL;
		infoBean.set("CODE", profildsi.getCode());
		infoBean.set("ID_PROFIL", profildsi.getIdProfildsi());
		infoBean.set("LIBELLE", profildsi.getLibelle());
		final String codeRubrique = profildsi.getCodeRubriqueAccueil();
		infoBean.set("LIBELLE_CODE_RUBRIQUE_ACCUEIL", Rubrique.getIntitule(codeRubrique));
		infoBean.set("CODE_RUBRIQUE_ACCUEIL", profildsi.getCodeRubriqueAccueil());
		//JSS 20040409 : gestion des roles
		/*
		infoBean.set("ROLES", profildsi.getRoles ());
		ControleurAffectationRole.preparerPRINCIPAL(infoBean, this);
		 */
		// JSS 20050510 : profils dynamiques
		final String sTmp = Chaine.convertirPointsVirgules(profildsi.getVecteurGroupes());
		infoBean.set("GROUPES", sTmp);
		infoBean.set("LIBELLE_GROUPES", Groupedsi.getIntitule(sTmp));
		if (StringUtils.isNotBlank(profildsi.getLibelle())) {
			infoBean.setTitreEcran(profildsi.getLibelle());
		}
	}

	/**
	 * Affichage de l'écran des critères de recherche d'un Profildsi.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerRECHERCHE() throws Exception {
		ecranLogique = ECRAN_RECHERCHE;
		infoBean.set("ID_PROFILDSI", profildsi.getIdProfildsi());
		infoBean.set("LISTE_PROFILDSI", Profildsi.getListeProfilsDSI(this));
	}

	/**
	 * Point d'entrée du processus.
	 *
	 * @return true, if traiter action
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public boolean traiterAction() throws Exception {
		try {
			ecranLogique = infoBean.getEcranLogique();
			action = infoBean.getActionUtilisateur();
			final AutorisationBean autorisations = (AutorisationBean) getGp().getSessionUtilisateur().getInfos().get(SessionUtilisateur.AUTORISATIONS);
			if (!ComposantProfil.isAutoriseParActionProcessus(autorisations, null)) {
				if (autorisations == null) {
					infoBean.setEcranRedirection(WebAppUtil.CONNEXION_BO);
					infoBean.setEcranLogique("LOGIN");
				} else {
					throw new ErreurApplicative(MessageHelper.getCoreMessage("BO_OPERATION_INTERDITE"));
				}
			} else {
				if (etat == DEBUT) {
					initialisation();
				} else if (ecranLogique.equals(ECRAN_RECHERCHE)) {
					traiterRECHERCHE();
				} else if (ecranLogique.equals(ECRAN_LISTE)) {
					traiterLISTE();
				} else if (ecranLogique.equals(ECRAN_PRINCIPAL)) {
					traiterPRINCIPAL();
				}
				//placer l'état dans le composant d'infoBean
				infoBean.setEcranLogique(ecranLogique);
			}
		} catch (final Exception e) {
			LOG.error("erreur de traitement sur le processus", e);
			infoBean.addMessageErreur(e.toString());
		}
		// On continue si on n'est pas à la FIN !!!
		return (etat == FIN);
	}

	/**
	 * Traitement de l'écran de sélection d'un Profildsi.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterLISTE() throws Exception {
		if (action.equals(InfoBean.ACTION_MODIFIER)) {
			modifierParId();
		}
		if (action.equals(InfoBean.ACTION_VALIDER)) {
			final int indice = Integer.parseInt((String) infoBean.get("LISTE_INDICE"));
			profildsi.setIdProfildsi(listeIdentifiant[indice]);
			profildsi.retrieve();
			remplirDonneesRecherche();
			etat = FIN;
		}
		if (action.equals(InfoBean.ACTION_SUPPRIMER)) {
			final int indice = Integer.parseInt((String) infoBean.get("LISTE_INDICE"));
			profildsi.setIdProfildsi(listeIdentifiant[indice]);
			profildsi.retrieve();
			profildsi.delete();
			cache.flush();
			etat = FIN;
		}
		if (action.equals(InfoBean.ACTION_ANNULER)) {
			infoBean.set("ID_PROFILDSI", null);
			etat = FIN;
		}
	}

	private void modifierParId() throws Exception {
		mode = MODE_MODIFICATION;
		final String idProfil = infoBean.getString("ID_PROFIL");
		profildsi.setIdProfildsi(Long.valueOf(idProfil));
		profildsi.retrieve();
		preparerPRINCIPAL();
	}

	/**
	 * Gestion de l'écran de saisie d'un Profildsi.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterPRINCIPAL() throws Exception {
		if (action.equals(InfoBean.ACTION_VALIDER)) {
			profildsi.setCode((String) infoBean.get("CODE"));
			profildsi.setLibelle((String) infoBean.get("LIBELLE"));
			profildsi.setCodeRubriqueAccueil((String) infoBean.get("CODE_RUBRIQUE_ACCUEIL"));
			profildsi.setVecteurGroupes(Chaine.getVecteurPointsVirgules(infoBean.getString("GROUPES")));
			if (mode == MODE_AJOUT) {
				/* On teste si le code existe déjà */
				final Profildsi profildsi2 = new Profildsi();
				profildsi2.setCtx(this);
				profildsi2.init();
				if (profildsi2.select(profildsi.getCode(), "") > 0) {
					throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "PROFIL_DSI.ERREUR.CODE_EXISTANT"));
				}
				profildsi.add();
				final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_CREATION_PROFILDSI"), profildsi.getLibelle());
				infoBean.addMessageConfirmation(confirmation);
			}
			if (mode == MODE_MODIFICATION) {
				profildsi.update();
				final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_MODIFICATION_PROFILDSI"), profildsi.getLibelle());
				infoBean.addMessageConfirmation(confirmation);
			}
			cache.flush();
			etat = FIN;
		} else if (action.equals(InfoBean.ACTION_SUPPRIMER)) {
			supprimerParId();
		}
	}

	private void supprimerParId() throws Exception {
		final Long idProfil = Long.parseLong((String) infoBean.get("ID_PROFIL"));
		profildsi.setIdProfildsi(idProfil);
		profildsi.retrieve();
		profildsi.delete();
		cache.flush();
		final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_SUPPRESSION_PROFILDSI"), profildsi.getLibelle());
		infoBean.addMessageConfirmation(confirmation);
		etat = FIN;
	}

	/**
	 * Traitement associé à l'écran de saisie des critères.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterRECHERCHE() throws Exception {
		if (action.equals(InfoBean.ACTION_VALIDER)) {
			final ClauseWhere whereIdProfil = new ClauseWhere(ConditionHelper.egalVarchar("ID_PROFILDSI", infoBean.getString("ID_PROFILDSI")));
			int count = profildsi.select(whereIdProfil.formaterSQL());
			if (count == 0) {
				infoBean.addMessageErreur("Aucun enregistrement sélectionné");
			}
			if (count == 1) {
				profildsi.nextItem();
				if (mode == MODE_RECHERCHE) {
					remplirDonneesRecherche();
					etat = FIN;
				}
				if (mode == MODE_MODIFICATION) {
					preparerPRINCIPAL();
				}
				if (mode == MODE_SUPPRESSION) {
					preparerLISTE();
				}
			}
			if (count > 1) {
				preparerLISTE();
			}
		}
		if (action.equals(InfoBean.ACTION_ANNULER)) {
			if (mode == MODE_RECHERCHE) {
				infoBean.set("ID_PROFILDSI", null);
				etat = FIN;
			}
		}
	}

	/**
	 * Préparation des données à renvoyer pour une rechercher Profildsi.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void remplirDonneesRecherche() throws Exception {
		infoBean.set("ID_PROFILDSI", profildsi.getIdProfildsi());
		infoBean.set("CODE", profildsi.getCode());
		infoBean.set("LIBELLE", profildsi.getLibelle());
		infoBean.set("CODE_RUBRIQUE_ACCUEIL", profildsi.getCodeRubriqueAccueil());
	}
}
