/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.processus;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.Vector;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.CodeLibelle;
import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.database.ProcessusBean;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.cms.objetspartages.Objetpartage;
import com.kportal.core.autorisation.ActionPermission;
import com.kportal.core.autorisation.Permission;
import com.kportal.core.autorisation.Permission.Type;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.config.PropertyHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.kportal.extension.ExtensionHelper;
import com.kportal.extension.ExtensionManager;
import com.kportal.extension.IExtension;
import com.kportal.extension.module.IModule;
import com.kportal.extension.module.ModuleHelper;
import com.kportal.extension.module.ModuleManager;
import com.kportal.extension.module.composant.ComposantRole;
import com.univ.collaboratif.om.Espacecollaboratif;
import com.univ.objetspartages.cache.CacheRoleManager;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Groupedsi;
import com.univ.objetspartages.om.Perimetre;
import com.univ.objetspartages.om.PermissionBean;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Role;
import com.univ.objetspartages.om.Rubrique;
import com.univ.objetspartages.om.Structure;
import com.univ.utils.Chaine;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

/**
 * processus saisie Role.
 */
public class SaisieRole extends ProcessusBean {

    /** The Constant MODE_AJOUT. */
    private static final int MODE_AJOUT = 0;

    /** The Constant MODE_MODIFICATION. */
    private static final int MODE_MODIFICATION = 1;

    /** The Constant MODE_SUPPRESSION. */
    private static final int MODE_SUPPRESSION = 2;

    /** The Constant MODE_RECHERCHE. */
    private static final int MODE_RECHERCHE = 3;

    /** The Constant ECRAN_RECHERCHE. */
    private static final String ECRAN_RECHERCHE = "RECHERCHE";

    /** The Constant ECRAN_PRINCIPAL. */
    private static final String ECRAN_PRINCIPAL = "PRINCIPAL";

    /** The Constant ECRAN_LISTE. */
    private static final String ECRAN_LISTE = "LISTE";

    private static org.slf4j.Logger LOG = LoggerFactory.getLogger(SaisieRole.class);

    private final CacheRoleManager cache = (CacheRoleManager) ApplicationContextManager.getCoreContextBean(CacheRoleManager.ID_BEAN);

    /** The role. */
    Role role = null;

    /** The autorisations. */
    AutorisationBean autorisations = null;

    /** The liste identifiant. */
    Long listeIdentifiant[] = null;

    /** The mode. */
    private int mode = -1;

    /**
     * processus saisie Role.
     *
     * @param ciu
     *            com.jsbsoft.jtf.core.InfoBean
     */
    public SaisieRole(final InfoBean ciu) {
        super(ciu);
    }

    /**
     * Initialisation du processus.
     *
     * @throws Exception
     *             the exception
     */
    private void initialisation() throws Exception {
        // initalisation objets métiers
        role = new Role();
        role.setCtx(this);
        role.init();
        // initalisation liste objets permissions
        infoBean.set("LISTE_TECHS", CodeLibelle.lireTable(infoBean.getNomExtension(), "permission_objet_tech", null));
        etat = EN_COURS;
        if (action.equals("AJOUTER")) {
            mode = MODE_AJOUT;
            infoBean.setEtatObjet(InfoBean.ETAT_OBJET_CREATION);
            preparerPRINCIPAL();
        } else if (action.equals("MODIFIER")) {
            mode = MODE_MODIFICATION;
            infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
            preparerRECHERCHE();
        } else if (action.equals("MODIFIERPARID")) {
            final String paramIdRole = infoBean.getString("ID_ROLE");
            long idRole = 0L;
            if (StringUtils.isNotBlank(paramIdRole) && StringUtils.isNumeric(paramIdRole)) {
                idRole = Long.valueOf(paramIdRole);
            }
            role.setIdRole(idRole);
            try {
                role.retrieve();
            } catch (final Exception e) {
                throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "ROLE.ERREUR.ROLE_INEXISTANT"));
            }
            mode = MODE_MODIFICATION;
            infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
            preparerPRINCIPAL();
        } else if (action.equals("SUPPRIMER")) {
            mode = MODE_SUPPRESSION;
            infoBean.setEtatObjet(InfoBean.ETAT_OBJET_SUPPRESSION);
            preparerRECHERCHE();
        } else if (action.equals("SUPPRIMERPARID")) {
            supprimerParId(Long.valueOf(infoBean.getString("ID_ROLE")));
        } else if (action.equals("RECHERCHER")) {
            mode = MODE_RECHERCHE;
            infoBean.setEtatObjet(InfoBean.ETAT_OBJET_RECHERCHE);
            preparerRECHERCHE();
        }
    }

    /**
     * Affichage de la liste des Roles.
     *
     * @throws Exception
     *             the exception
     */
    private void preparerLISTE() throws Exception {
        int i = 0;
        ecranLogique = ECRAN_LISTE;
        final ClauseWhere whereIdRole = new ClauseWhere(ConditionHelper.egalVarchar("ID_ROLE", infoBean.getString("ID_ROLE")));
        final int count = role.select(whereIdRole.formaterSQL());
        listeIdentifiant = new Long[count];
        while (role.nextItem()) {
            infoBean.set("CODE#" + i, role.getCode());
            infoBean.set("LIBELLE#" + i, role.getLibelle());
            infoBean.set("ID_ROLE#" + i, role.getIdRole());
            listeIdentifiant[i] = role.getIdRole();
            i++;
        }
        infoBean.set("LISTE_NB_ITEMS", i);
    }

    /**
     * Affichage de l'écran de saisie d'un Role.
     *
     * @throws Exception
     *             the exception
     */
    private void preparerPRINCIPAL() throws Exception {
        ecranLogique = ECRAN_PRINCIPAL;
        infoBean.set("CODE", role.getCode());
        infoBean.set("LIBELLE", role.getLibelle());
        final Perimetre perimetre = new Perimetre(role.getPerimetre());
        // JSS 20051104 : périmètres
        // Structure
        infoBean.set("STRUCTURE_TYPE_PERIMETRE", "");
        infoBean.set("STRUCTURE_PERIMETRE", "");
        infoBean.set("LIBELLE_STRUCTURE_PERIMETRE", "");
        if (perimetre.getCodeStructure().equals("")) {
            infoBean.set("STRUCTURE_TYPE_PERIMETRE", "0");
        } else if (perimetre.getCodeStructure().equals("-")) {
            infoBean.set("STRUCTURE_TYPE_PERIMETRE", "2");
        } else {
            infoBean.set("STRUCTURE_TYPE_PERIMETRE", "1");
            infoBean.set("STRUCTURE_PERIMETRE", perimetre.getCodeStructure());
            infoBean.set("LIBELLE_STRUCTURE_PERIMETRE", Structure.getLibelleAffichable(perimetre.getCodeStructure(), ""));
        }
        // Rubrique
        infoBean.set("RUBRIQUE_TYPE_PERIMETRE", "");
        infoBean.set("RUBRIQUE_PERIMETRE", "");
        infoBean.set("LIBELLE_RUBRIQUE_PERIMETRE", "");
        if (perimetre.getCodeRubrique().equals("")) {
            infoBean.set("RUBRIQUE_TYPE_PERIMETRE", "0");
        } else if (perimetre.getCodeRubrique().equals("-")) {
            infoBean.set("RUBRIQUE_TYPE_PERIMETRE", "2");
        } else {
            infoBean.set("RUBRIQUE_TYPE_PERIMETRE", "1");
            infoBean.set("RUBRIQUE_PERIMETRE", perimetre.getCodeRubrique());
            infoBean.set("LIBELLE_RUBRIQUE_PERIMETRE", Rubrique.renvoyerItemRubrique(perimetre.getCodeRubrique()).getIntitule());
        }
        // Groupes
        infoBean.set("GROUPE_TYPE_PERIMETRE", "");
        infoBean.set("GROUPE_PERIMETRE", "");
        infoBean.set("LIBELLE_GROUPE_PERIMETRE", "");
        if (perimetre.getCodeGroupe().equals("")) {
            infoBean.set("GROUPE_TYPE_PERIMETRE", "0");
        } else if (perimetre.getCodeGroupe().equals("-")) {
            infoBean.set("GROUPE_TYPE_PERIMETRE", "2");
        } else {
            infoBean.set("GROUPE_TYPE_PERIMETRE", "1");
            infoBean.set("GROUPE_PERIMETRE", perimetre.getCodeGroupe());
            infoBean.set("LIBELLE_GROUPE_PERIMETRE", Groupedsi.getIntitule(perimetre.getCodeGroupe()));
        }
        if (Espacecollaboratif.isExtensionActivated()) {
            if (perimetre.getCodeEspaceCollaboratif().equals("*")) {
                infoBean.set("ESPACE_COLLABORATIF_COURANT_PERIMETRE", "1");
            } else {
                infoBean.set("ESPACE_COLLABORATIF_COURANT_PERIMETRE", "0");
            }
        }
        infoBean.set("PERMISSIONS", role.getPermissions());
        // Initialisation des permissions par extension par type
        final ModuleManager moduleManager = ModuleHelper.getModuleManager();
        final ExtensionManager extensionManager = ExtensionHelper.getExtensionManager();
        final Map<Type, List<Permission>> permissionsByType = new HashMap<>();
        if (moduleManager.getModules().size() > 0) {
            // boucle par extension
            for (final IExtension extension : extensionManager.getExtensions().values()) {
                final List<IModule> modules = new ArrayList<>();
                modules.addAll(extension.getModules());
                Collections.sort(modules, new Comparator<IModule>() {

                    @Override
                    public int compare(final IModule o1, final IModule o2) {
                        int nbActionsO1 = 0;
                        int nbActionsO2 = 0;
                        for (final Permission permission : o1.getPermissions()) {
                            nbActionsO1 += permission.getActions().size();
                        }
                        for (final Permission permission : o2.getPermissions()) {
                            nbActionsO2 += permission.getActions().size();
                        }
                        return nbActionsO2 - nbActionsO1;
                    }
                });
                for (final IModule module : modules) {
                    if (module.getEtat() == IModule.ETAT_ACTIF) {
                        for (final Permission permission : module.getPermissions()) {
                            if (permissionsByType.get(permission.getType()) == null) {
                                permissionsByType.put(permission.getType(), new ArrayList<Permission>());
                            }
                            permissionsByType.get(permission.getType()).add(permission);
                        }
                    }
                }
            }
        }
        /* 0 - ONGLET 0 CONTENU : fiches et gestion éditoriale */
        infoBean.setInt("DEBUT_ONGLET_0", 0);
        int i = 0;
        int j = 0;
        int k = 0;
        int indiceOnglet = 0;
        Enumeration<PermissionBean> enumPermissions = null;
        final String ordre[] = ReferentielObjets.getOrdreOnglets();
        // Boucle sur les fiches
        while (k < ordre.length) {
            final String codeObjet = ordre[k];
            Objetpartage objet = ReferentielObjets.getObjetByCode(codeObjet);
            if (ReferentielObjets.instancierFiche(codeObjet) == null || objet.isStrictlyCollaboratif()) {
                k++;
                continue;
            }
            int nbActions= 0;
            if (i == 0) {
                infoBean.set("PERMISSION_ONGLET_" + i, "contenu");
                infoBean.set("PERMISSION_PERIMETRE_" + i + "_0", "rubrique");
                infoBean.set("PERMISSION_PERIMETRE_" + i + "_1", "structure");
                infoBean.set("PERMISSION_TITRE_" + i, "Fiches");
                indiceOnglet++;
            }
            infoBean.set("PERMISSION_TYPE_" + i, "FICHE");
            infoBean.set("PERMISSION_CODE_OBJET_" + i, codeObjet);
            infoBean.set("PERMISSION_INTITULE_" + i, ReferentielObjets.getLibelleObjet(codeObjet));
            infoBean.set("PERMISSION_HORS_PERIMETRE_" + i, "0");
            infoBean.set("PERMISSION_ACTION_" + i + "_" + nbActions, "C");
            if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "C")) {
                infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_" + nbActions, "1");
            }
            nbActions++;
            infoBean.set("PERMISSION_ACTION_" + i + "_" + nbActions, "M");
            if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "M")) {
                infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_" + nbActions, "1");
            }
            nbActions++;
            infoBean.set("PERMISSION_ACTION_" + i + "_" + nbActions, "D");
            if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "D")) {
                infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_" + nbActions, "1");
            }
            nbActions++;
            infoBean.set("PERMISSION_ACTION_" + i + "_" + nbActions, "S");
            if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "S")) {
                infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_" + nbActions, "1");
            }
            nbActions++;
            // validation
            infoBean.set("PERMISSION_ACTION_" + i + "_" + nbActions, "V");
            if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "V")) {
                infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_" + nbActions, "1");
            }
            nbActions++;
            // notification
            infoBean.set("PERMISSION_ACTION_" + i + "_" + nbActions, "N");
            if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "N")) {
                infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_" + nbActions, "1");
            }
            nbActions++;
            // JSS 20051115 : ajout droit mise en ligne
            infoBean.set("PERMISSION_ACTION_" + i + "_" + nbActions, "U");
            if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "U")) {
                infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_" + nbActions, "1");
            }
            nbActions++;
            infoBean.set("PERMISSION_ACTION_" + i + "_" + nbActions, "R");
            if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "R")) {
                infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_" + nbActions, "1");
            }
            if (i == 0) {
                nbActions = 0;
                infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_" + nbActions++, "Création");
                infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_" + nbActions++, "Modification");
                infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_" + nbActions++, "Traduction");
                infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_" + nbActions++, "Suppression");
                infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_" + nbActions++, "Validation");
                infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_" + nbActions++, "Notification");
                infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_" + nbActions++, "Mise en ligne");
                infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_" + nbActions++, "Suppression");
            }
            // Boucle sur les permissions
            enumPermissions = role.getVecteurPermissions().elements();
            while (enumPermissions.hasMoreElements()) {
                final PermissionBean permission = (enumPermissions.nextElement());
                if ((permission.getType().equals("FICHE")) && (permission.getObjet().equals(codeObjet))) {
                    if (permission.getAction().equals("C")) {
                        infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_0", "1");
                    }
                    if (permission.getAction().equals("M")) {
                        infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_1", "1");
                    }
                    if (permission.getAction().equals("D")) {
                        infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_2", "1");
                    }
                    if (permission.getAction().equals("S")) {
                        infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_3", "1");
                    }
                    if (permission.getAction().equals("V")) {
                        infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_4", "1");
                    }
                    if (permission.getAction().equals("N")) {
                        infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_5", "1");
                    }
                    if (permission.getAction().equals("U")) {
                        infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_6", "1");
                    }
                    if (permission.getAction().equals("R")) {
                        infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_7", "1");
                    }
                }
            }
            infoBean.setInt("PERMISSIONS_NB_ACTIONS_" + i, 8);
            i++;
            k++;
        }// fin de boucle sur les objets

        /************************************************ Gestion éditoriale **************************************/

        final Hashtable<String, String> hDejaAffiche = new Hashtable<>();
        final Hashtable<String, String> hPermissionsTechniques = CodeLibelle.lireTable(infoBean.getNomExtension(), "permission_objet_tech", null);
        i = ajouterPermissionTechnique(i, "rub", hPermissionsTechniques.get("rub"), "Gestion éditoriale", hDejaAffiche, new String[] {"C", "M", "S", "P"}, null);
        // RP 20060217 !!!!!!!! le droit de publication P/"pub" est ajouté dans la liste des droits "rub"
        // donc traité en spécifique dans la méthode ajouterPermissionTechnique et le traiter principal
        // i = ajouterPermissionTechnique( i, "pub", (String) hPermissionsTechniques.get( "pub"),  null, hDejaAffiche, null, "" );
        i = ajouterPermissionTechnique(i, "pho", hPermissionsTechniques.get("pho"), null, hDejaAffiche, new String[] {"U", "C", "M", "S"}, null);
        // RP 20060217 !!!!!!!! le droit d'utilisation U à la photothèque "acp" est ajouté dans la liste des droits "pho"
        // donc traité en spécifique dans la méthode ajouterPermissionTechnique et le traiter principal
        //i = ajouterPermissionTechnique( i, "acp", (String) hPermissionsTechniques.get( "acp"),  null, hDejaAffiche, null, "" );
        i = ajouterPermissionTechnique(i, "enc", hPermissionsTechniques.get("enc"), null, hDejaAffiche, new String[] {"M"}, null);
        // on ajoute les permissions du socle de type contenu
        if (CollectionUtils.isNotEmpty(permissionsByType.get(Type.CONTENU))) {
            i = ajouterPermissionExtension(i, permissionsByType.get(Type.CONTENU));
        }
        infoBean.setInt("FIN_ONGLET_0", i - 1);
        /************************************************ MULTI NIVEAU **************************************/
        final Hashtable<String, String> hPermissionsValidation = AutorisationBean.getListeNiveauxApprobation();
        if (MapUtils.isNotEmpty(hPermissionsValidation)) {
            infoBean.setInt("DEBUT_ONGLET_" + indiceOnglet, i);
            i = this.mapRoleMultiNiveau(i,ordre,hPermissionsValidation);
            infoBean.setInt("FIN_ONGLET_" + indiceOnglet, i - 1);
            indiceOnglet++;
        }
        /************************************************ CONTRIBUTION AVANCEE **************************************/
        /* 1 - ONGLET 1 CONTRIBUTION AVANCEE : mode et fonctionnalités */
        infoBean.setInt("DEBUT_ONGLET_" + indiceOnglet, i);
        infoBean.set("PERMISSION_ONGLET_" + i, "contribution-avancee");
        i = ajouterPermissionTechnique(i, "mda", hPermissionsTechniques.get("mda"), "Mode de contribution", hDejaAffiche, null, null);
        i = ajouterPermissionTechnique(i, "flf", hPermissionsTechniques.get("flf"), "Fonctionnalités avancées", hDejaAffiche, null, null);
        i = ajouterPermissionTechnique(i, "fat", hPermissionsTechniques.get("fat"), null, hDejaAffiche, null, null);
        i = ajouterPermissionTechnique(i, "fra", hPermissionsTechniques.get("fra"), null, hDejaAffiche, null, null);
        i = ajouterPermissionTechnique(i, "fgm", hPermissionsTechniques.get("fgm"), null, hDejaAffiche, null, null);
        //		i = ajouterPermissionTechnique(i, "fps", (String) hPermissionsTechniques.get("fps"), null, hDejaAffiche, null, null);
        i = ajouterPermissionTechnique(i, "fpg", hPermissionsTechniques.get("fpg"), null, hDejaAffiche, null, null);
        i = ajouterPermissionTechnique(i, "mdh", hPermissionsTechniques.get("mdh"), null, hDejaAffiche, null, null);
        // on ajoute les permissions de type contribution
        if (CollectionUtils.isNotEmpty(permissionsByType.get(Type.CONTRIBUTION))) {
            i = ajouterPermissionExtension(i, permissionsByType.get(Type.CONTRIBUTION));
        }
        infoBean.setInt("FIN_ONGLET_" + indiceOnglet, i - 1);
        indiceOnglet++;
        // RP 20060104 : activation du collaboratif
        final String activationDsi = PropertyHelper.getCoreProperty("dsi.activation");
        infoBean.set("GRS_SAISIE_DSI", "0");
        if ("1".equals(activationDsi)) {
            infoBean.set("GRS_SAISIE_DSI", "1");
			/* 2 - ONGLET 2 DIFFUSION : gestion de la dsi */
            infoBean.setInt("DEBUT_ONGLET_" + indiceOnglet, i);
            infoBean.set("PERMISSION_PERIMETRE_" + i + "_0", "groupe");
            infoBean.set("PERMISSION_ONGLET_" + i, "diffusion");
            // on ajoute les permissions de type diffusion
            if (CollectionUtils.isNotEmpty(permissionsByType.get(Type.DIFFUSION))) {
                i = ajouterPermissionExtension(i, permissionsByType.get(Type.DIFFUSION));
            }
            i = ajouterPermissionTechnique(i, "dsi", hPermissionsTechniques.get("dsi"), "Groupes", hDejaAffiche, null, null);
            infoBean.setInt("FIN_ONGLET_" + indiceOnglet, i - 1);
            indiceOnglet++;
        } else {
            hPermissionsTechniques.remove("dsi");
        }
		/* 3 - ONGLET 3 COLLABORATIF */
        if (Espacecollaboratif.isExtensionActivated()) {
            infoBean.setInt("DEBUT_ONGLET_" + indiceOnglet, i);
            k = 0;
            j = i;
            infoBean.set("PERMISSION_ONGLET_" + i, "collaboratif");
            // Boucle sur les fiches
            while (k < ordre.length) {
                final String codeObjet = ordre[k];
                if (!ReferentielObjets.getObjetByCode(codeObjet).isCollaboratif()) {
                    k++;
                    continue;
                }
                if (j == i) {
                    infoBean.set("PERMISSION_TITRE_" + i, "Fiches");
                }
                infoBean.set("PERMISSION_TYPE_" + i, "FICHE");
                infoBean.set("PERMISSION_CODE_OBJET_" + i, codeObjet);
                infoBean.set("PERMISSION_INTITULE_" + i, ReferentielObjets.getLibelleObjet(codeObjet));
                infoBean.set("PERMISSION_HORS_PERIMETRE_" + i, "0");
                infoBean.set("PERMISSION_ACTION_" + i + "_0", "C");
                if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "C")) {
                    infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_0", "1");
                }
                infoBean.set("PERMISSION_ACTION_" + i + "_1", "M");
                if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "M")) {
                    infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_1", "1");
                }
                infoBean.set("PERMISSION_ACTION_" + i + "_2", "S");
                if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "S")) {
                    infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_2", "1");
                }
                infoBean.set("PERMISSION_ACTION_" + i + "_3", "V");
                if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), "V")) {
                    infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_3", "1");
                }
                if (j == i) {
                    infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_0", "Création");
                    infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_1", "Modification");
                    infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_2", "Suppression");
                    infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_3", "Validation");
                }
                // Boucle sur les permissions
                enumPermissions = role.getVecteurPermissions().elements();
                while (enumPermissions.hasMoreElements()) {
                    final PermissionBean permission = (enumPermissions.nextElement());
                    if ((permission.getType().equals("FICHE")) && (permission.getObjet().equals(codeObjet))) {
                        if (permission.getAction().equals("C")) {
                            infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_0", "1");
                        }
                        if (permission.getAction().equals("M")) {
                            infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_1", "1");
                        }
                        if (permission.getAction().equals("S")) {
                            infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_2", "1");
                        }
                        if (permission.getAction().equals("V")) {
                            infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_3", "1");
                        }
                    }
                }
                infoBean.setInt("PERMISSIONS_NB_ACTIONS_" + i, 4);
                k++;
                i++;
            }
            // on ajoute les permissions de type administration
            if (CollectionUtils.isNotEmpty(permissionsByType.get(Type.COLLABORATIF))) {
                i = ajouterPermissionExtension(i, permissionsByType.get(Type.COLLABORATIF));
            }
            i = ajouterPermissionTechnique(i, "ges", hPermissionsTechniques.get("ges"), "Gestion des espaces", hDejaAffiche, null, null);
            infoBean.setInt("FIN_ONGLET_" + indiceOnglet, i - 1);
            indiceOnglet++;
        } else {
            hPermissionsTechniques.remove("ges");
        }
		/* 4 - ONGLET 4 Administration  */
        infoBean.setInt("DEBUT_ONGLET_" + indiceOnglet, i);
        infoBean.set("PERMISSION_ONGLET_" + i, "administration");
        // on ajoute les permissions de type administration
        if (CollectionUtils.isNotEmpty(permissionsByType.get(Type.ADMINISTRATION))) {
            i = ajouterPermissionExtension(i, permissionsByType.get(Type.ADMINISTRATION));
        }
        i = ajouterPermissionTechnique(i, "wmg", hPermissionsTechniques.get("wmg"), "Super administrateur", hDejaAffiche, null, null);
        infoBean.setInt("FIN_ONGLET_" + indiceOnglet, i - 1);
        indiceOnglet++;
		/* 5 - ONGLET 5 Modules */
        if (CollectionUtils.isNotEmpty(permissionsByType.get(Type.MODULE))) {
            infoBean.setInt("DEBUT_ONGLET_" + indiceOnglet, i);
            infoBean.set("PERMISSION_ONGLET_" + i, "modules");
            // on ajoute les permissions de type module
            i = ajouterPermissionExtension(i, permissionsByType.get(Type.MODULE));
            infoBean.setInt("FIN_ONGLET_" + indiceOnglet, i - 1);
            indiceOnglet++;
            infoBean.set("GRS_SAISIE_MODULES", "1");
        }
		/* 6 - ONGLET 6 Scripts */
        if (CollectionUtils.isNotEmpty(permissionsByType.get(Type.SCRIPT))) {
            infoBean.setInt("DEBUT_ONGLET_" + indiceOnglet, i);
            infoBean.set("PERMISSION_ONGLET_" + i, "scripts");
            // on ajoute les permissions de type module
            i = ajouterPermissionExtension(i, permissionsByType.get(Type.SCRIPT));
            infoBean.setInt("FIN_ONGLET_" + indiceOnglet, i - 1);
            indiceOnglet++;
            infoBean.set("GRS_SAISIE_MODULES", "1");
        }
        infoBean.setInt("PERMISSIONS_NB_ITEMS", i);
        infoBean.setInt("ONGLETS_NB_ITEMS", indiceOnglet);
        if (StringUtils.isNotBlank(role.getLibelle())) {
            infoBean.setTitreEcran(role.getLibelle());
        }
    }

    private int mapRoleMultiNiveau(int i, final String[] ordre, final Hashtable<String, String> hPermissionsValidation) {
        int debutIndex = i;
        final TreeSet<String> tmPermissionsValidation = AutorisationBean.getListePermissionsValidation();
        infoBean.set("PERMISSION_ONGLET_" + i, "validation-multi-niveau");
        infoBean.set("PERMISSION_TITRE_" + i, "Fiches");

        int j = 0;
        int k = 0;
        while (k < ordre.length) {
            final String codeObjet = ordre[k];
            Objetpartage objet = ReferentielObjets.getObjetByCode(codeObjet);
            if (ReferentielObjets.instancierFiche(codeObjet) == null || objet.isStrictlyCollaboratif()) {
                k++;
                continue;
            }
            j = 0;
            int indexPermission = 0;
            infoBean.set("PERMISSION_CODE_OBJET_" + i, codeObjet);
            infoBean.set("PERMISSION_TYPE_" + i, "FICHE");
            infoBean.set("PERMISSION_HORS_PERIMETRE_" + i, "0");
            infoBean.set("PERMISSION_INTITULE_" + i, ReferentielObjets.getLibelleObjet(codeObjet));
            for (String code : tmPermissionsValidation) {
                infoBean.set("PERMISSION_ACTION_" + i + "_" + j, code);
                if (i == debutIndex) {
                    infoBean.set("PERMISSION_ACTION_LIBELLE_" + i + "_" + j, hPermissionsValidation.get(code));
                }
                if (!ReferentielObjets.isActionParametrableDansRole(ReferentielObjets.getNomObjet(codeObjet), code)) {
                    infoBean.set("PERMISSION_ACTION_DISABLED_" + i + "_" + j, "1");
                }
                j++;
            }
            Enumeration<PermissionBean> enumPermissions = role.getVecteurPermissions().elements();
            if(enumPermissions != null) {
                while (enumPermissions.hasMoreElements()) {
                    indexPermission = 0;
                    final PermissionBean permission = (enumPermissions.nextElement());
                    if ((permission.getType().equals("FICHE")) && (permission.getObjet().equals(codeObjet))) {
                        for (String code : tmPermissionsValidation) {
                            if (permission.getAction().equals(code)) {
                                infoBean.set("PERMISSION_ACTION_SELECT_" + i + "_" + indexPermission, "1");
                            }
                            indexPermission++;
                        }
                    }
                }
            }
            infoBean.setInt("PERMISSIONS_NB_ACTIONS_" + i, j);
            k++;
            i++;
        }
        return i;
    }

    private int ajouterPermissionExtension(int i, final List<Permission> permissions) throws Exception {
        String idExtension = "";
        String section = null;
        for (final Permission permission : permissions) {
            final IExtension extension = ExtensionHelper.getExtension(permission.getIdExtension());
            if (!permission.getIdExtension().equals(idExtension)) {
                idExtension = permission.getIdExtension();
                // creation d'une nouvelle seection pour chaque extension
                section = extension.getLibelleAffichable(extension.getLibelle());
            }
            i = ajouterPermissionModule(i, permission.getId(), permission.getCode(), section, permission.getLibelleAffichable(), permission.getActions());
            section = null;
        }
        return i;
    }

    /**
     * Ajouter permission technique.
     *
     * @param _nbItems
     *            the _nb items
     * @param objet
     *            the objet
     * @param intituleObjet
     *            the intitule objet
     * @param titreSection
     *            the titre rupture
     * @param hDejaAffiche
     *            the h deja affiche
     * @param codesAction
     *            the codes action
     * @param titreActionParDefaut
     *            the titre action par defaut
     *
     * @return the int
     *
     * @throws Exception
     *             the exception
     */
    public int ajouterPermissionTechnique(final int _nbItems, String objet, final String intituleObjet, final String titreSection, final Hashtable<String, String> hDejaAffiche, final String codesAction[], final String titreActionParDefaut) throws Exception {
        int nbItems = _nbItems;
        if (hDejaAffiche.get(objet) == null) {
            infoBean.set("PERMISSION_TITRE_" + nbItems, titreSection);
            infoBean.set("PERMISSION_TYPE_" + nbItems, PermissionBean.TYPE_TECHNIQUE);
            infoBean.set("PERMISSION_CODE_OBJET_" + nbItems, objet);
            infoBean.set("PERMISSION_INTITULE_" + nbItems, intituleObjet);
            infoBean.set("PERMISSION_HORS_PERIMETRE_" + nbItems, "0");
            // Parcours du fichier contenant les actions
            int nbActions = 0;
            final Hashtable<String, String> listeActions = CodeLibelle.lireTable(infoBean.getNomExtension(), "permission_action_tech_" + objet, null);
            // codesAction permet de forcer l'ordre des actions
            if (codesAction == null) {
                if (listeActions.size() == 0) {
                    infoBean.set("PERMISSION_ACTION_" + nbItems + "_0", "");
                    nbActions = 1;
                } else {
                    final Enumeration<String> eActions = listeActions.keys();
                    while (eActions.hasMoreElements()) {
                        infoBean.set("PERMISSION_ACTION_" + nbItems + "_" + nbActions, eActions.nextElement());
                        nbActions++;
                    }
                }
            } else {
                for (final String element : codesAction) {
                    infoBean.set("PERMISSION_ACTION_" + nbItems + "_" + nbActions, element);
                    nbActions++;
                }
            }
            final String objetSave = objet;
            // Recherche de la permission pour chaque action
            int j;
            for (j = 0; j < nbActions; j++) {
                String codeAction = infoBean.getString("PERMISSION_ACTION_" + nbItems + "_" + j);
                final String codeActionPourLibelle = codeAction;
                objet = objetSave;
                if (objet.equals("rub") && codeAction.equals("P")) {
                    objet = "pub";
                    codeAction = "";
                }
                if (objet.equals("pho") && codeAction.equals("U")) {
                    objet = "acp";
                    codeAction = "";
                }
                // Recherche libellé associé pour afficher l'entete
                final String libelleAction = listeActions.get(codeActionPourLibelle);
                if (libelleAction == null) {
                    if (titreActionParDefaut != null) {
                        infoBean.set("PERMISSION_ACTION_LIBELLE_" + nbItems + "_" + j, titreActionParDefaut);
                    }
                } else {
                    infoBean.set("PERMISSION_ACTION_LIBELLE_" + nbItems + "_" + j, libelleAction);
                }
                final Enumeration<PermissionBean> enumPermissions = role.getVecteurPermissions().elements();
                while (enumPermissions.hasMoreElements()) {
                    final PermissionBean permission = (enumPermissions.nextElement());
                    // Recherche de la permission
                    if ((permission.getType().equals("TECH")) && (permission.getObjet().equals(objet)) && (permission.getAction().equals(codeAction))) {
                        infoBean.set("PERMISSION_ACTION_SELECT_" + nbItems + "_" + j, "1");
                    }
                }
            }
            infoBean.setInt("PERMISSIONS_NB_ACTIONS_" + nbItems, j);
            hDejaAffiche.put(objet, "");
            nbItems++;
        }
        return nbItems;
    }

    /**
     * Ajouter permission technique.
     *
     * @param _nbItems
     *            the _nb items
     * @param type
     *            the objet
     * @param objet
     *            the objet
     * @param section
     *            the objet
     * @param intituleObjet
     *            the intitule objet
     * @param actions
     *            the titre rupture
     * @return the int
     *
     * @throws Exception
     *             the exception
     */
    public int ajouterPermissionModule(final int _nbItems, final String type, final String objet, final String section, final String intituleObjet, final List<ActionPermission> actions) throws Exception {
        int nbItems = _nbItems;
        infoBean.set("PERMISSION_TYPE_" + nbItems, type.toUpperCase());
        infoBean.set("PERMISSION_CODE_OBJET_" + nbItems, objet);
        infoBean.set("PERMISSION_INTITULE_" + nbItems, intituleObjet);
        infoBean.set("PERMISSION_HORS_PERIMETRE_" + nbItems, "0");
        infoBean.set("PERMISSION_TITRE_" + nbItems, section);
        if (actions.size() == 0) {
            actions.add(new ActionPermission());
        }
        int nbActions = 0;
        for (final ActionPermission action : actions) {
            infoBean.set("PERMISSION_ACTION_" + nbItems + "_" + nbActions, action.getCode());
            if (action.getLibelle().length() > 0) {
                infoBean.set("PERMISSION_ACTION_LIBELLE_" + nbItems + "_" + nbActions, action.getLibelle());
            }
            final Enumeration<PermissionBean> enumPermissions = role.getVecteurPermissions().elements();
            while (enumPermissions.hasMoreElements()) {
                final PermissionBean permission = (enumPermissions.nextElement());
                // Recherche de la permission
                if ((permission.getType().equals(type.toUpperCase())) && (permission.getObjet().equals(objet)) && (permission.getAction().equals(action.getCode()))) {
                    infoBean.set("PERMISSION_ACTION_SELECT_" + nbItems + "_" + nbActions, "1");
                }
            }
            nbActions++;
        }
        infoBean.setInt("PERMISSIONS_NB_ACTIONS_" + nbItems, nbActions);
        nbItems++;
        return nbItems;
    }

    /**
     * Affichage de l'écran des critères de recherche d'un Role.
     *
     * @throws Exception
     *             the exception
     */
    private void preparerRECHERCHE() throws Exception {
        ecranLogique = ECRAN_RECHERCHE;
        infoBean.set("LISTE_ROLES", Role.getAllRolesWithoutCollab());
    }

    /**
     * Point d'entrée du processus.
     *
     * @return true, if traiter action
     *
     * @throws Exception
     *             the exception
     */
    @Override
    public boolean traiterAction() throws Exception {
        autorisations = (AutorisationBean) getGp().getSessionUtilisateur().getInfos().get(SessionUtilisateur.AUTORISATIONS);
        if (autorisations == null) {
            infoBean.setEcranRedirection(WebAppUtil.CONNEXION_BO);
            infoBean.setEcranLogique("LOGIN");
        } else {
            if (!ComposantRole.isAutoriseParActionProcessus(autorisations, null)) {
                throw new ErreurApplicative(MessageHelper.getCoreMessage("BO_OPERATION_INTERDITE"));
            }
            try {
                ecranLogique = infoBean.getEcranLogique();
                action = infoBean.getActionUtilisateur();
                if (etat == DEBUT) {
                    initialisation();
                } else if (ecranLogique.equals(ECRAN_RECHERCHE)) {
                    traiterRECHERCHE();
                } else if (ecranLogique.equals(ECRAN_LISTE)) {
                    traiterLISTE();
                } else if (ecranLogique.equals(ECRAN_PRINCIPAL)) {
                    traiterPRINCIPAL();
                }
                //placer l'état dans le composant d'infoBean
                infoBean.setEcranLogique(ecranLogique);
            } catch (final Exception e) {
                LOG.error("erreur de traitement sur le processus", e);
                infoBean.addMessageErreur(e.toString());
            }
        }
        // On continue si on n'est pas à la FIN !!!
        return (etat == FIN);
    }

    /**
     * Traitement de l'écran de sélection d'un Role.
     *
     * @throws Exception
     *             the exception
     */
    private void traiterLISTE() throws Exception {
        if (action.equals(InfoBean.ACTION_MODIFIER)) {
            final int indice = Integer.parseInt((String) infoBean.get("LISTE_INDICE"));
            role.setIdRole(listeIdentifiant[indice]);
            role.retrieve();
            preparerPRINCIPAL();
        }
        if (action.equals(InfoBean.ACTION_VALIDER)) {
            final int indice = Integer.parseInt((String) infoBean.get("LISTE_INDICE"));
            role.setIdRole(listeIdentifiant[indice]);
            role.retrieve();
            remplirDonneesRecherche();
        }
        if (action.equals(InfoBean.ACTION_ANNULER)) {
            infoBean.set("ID_ROLE", null);
            etat = FIN;
        }
    }

    /**
     * Gestion de l'écran de saisie d'un Role.
     *
     * @throws Exception
     *             the exception
     */
    private void traiterPRINCIPAL() throws Exception {
        if (action.equals(InfoBean.ACTION_VALIDER)) {
            // Vérifie que le code ne contient que des caractères alphanumériques ou '-' ou '_'
            Chaine.controlerCodeMetier((String) infoBean.get("CODE"));
            role.setCode((String) infoBean.get("CODE"));
            role.setLibelle((String) infoBean.get("LIBELLE"));
            String perimetreCodeEspaceCollaboratif = "";
            if ("1".equals(infoBean.getString("ESPACE_COLLABORATIF_COURANT_PERIMETRE"))) {
                perimetreCodeEspaceCollaboratif = "*";
            }
            // JSS 20051104 : gestion des périmètres
            String structurePerimetre = "";
            if (StringUtils.isNotBlank(infoBean.getString("STRUCTURE_PERIMETRE"))) {
                structurePerimetre = infoBean.getString("STRUCTURE_PERIMETRE");
            }
            String rubriquePerimetre = "";
            if (StringUtils.isNotBlank(infoBean.getString("RUBRIQUE_PERIMETRE"))) {
                rubriquePerimetre = infoBean.getString("RUBRIQUE_PERIMETRE");
            }
            String groupePerimetre = "";
            if (StringUtils.isNotBlank(infoBean.getString("GROUPE_PERIMETRE"))) {
                groupePerimetre = infoBean.getString("GROUPE_PERIMETRE");
            }
            role.setPerimetre(new Perimetre(structurePerimetre, rubriquePerimetre, "", groupePerimetre, perimetreCodeEspaceCollaboratif).getChaineSerialisee());
			/* Vecteur des permissions */
            final Vector<PermissionBean> v = new Vector<>();
            for (int j = 0; j < infoBean.getInt("PERMISSIONS_NB_ITEMS"); j++) {
                for (int k = 0; k < infoBean.getInt("PERMISSIONS_NB_ACTIONS_" + j); k++) {
                    final String permissionAction = infoBean.getString("PERMISSION_ACTION_" + j + "_" + k);
                    if (permissionAction != null) {
                        final String select = infoBean.getString("PERMISSION_ACTION_SELECT_" + j + "_" + k);
                        if ((select != null) && (select.equals("1"))) {
                            if (infoBean.getString("PERMISSION_CODE_OBJET_" + j).equals("rub") && permissionAction.equals("P")) {
                                v.add(new PermissionBean(infoBean.getString("PERMISSION_TYPE_" + j), "pub", ""));
                            } else if (infoBean.getString("PERMISSION_CODE_OBJET_" + j).equals("pho") && permissionAction.equals("U")) {
                                v.add(new PermissionBean(infoBean.getString("PERMISSION_TYPE_" + j), "acp", ""));
                            } else {
                                v.add(new PermissionBean(infoBean.getString("PERMISSION_TYPE_" + j), infoBean.getString("PERMISSION_CODE_OBJET_" + j), permissionAction));
                            }
                        }
                    }
                }
            }
            role.setVecteurPermissions(v);
            if (mode == MODE_AJOUT) {
				/* On teste si le code existe déjà */
                final Role role2 = new Role();
                role2.setCtx(this);
                role2.init();
                if (role2.select(role.getCode(), "", "") > 0) {
                    throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "ROLE.ERREUR.CODE_EXISTANT"));
                }
                role.add();
                final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_CREATION_ROLE"), role.getLibelle());
                infoBean.addMessageConfirmation(confirmation);
            }
            if (mode == MODE_MODIFICATION) {
                role.update();
                final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_MODIFICATION_ROLE"), role.getLibelle());
                infoBean.addMessageConfirmation(confirmation);
            }
            cache.flush();
            etat = FIN;
        } else if (action.equals(InfoBean.ACTION_SUPPRIMER)) {
            supprimerParId(Long.valueOf(infoBean.getString("ID_ROLE")));
        }
    }

    private void supprimerParId(final Long idRoleASupprimer) throws Exception {
        role.setIdRole(idRoleASupprimer);
        role.retrieve();
        role.delete();
        ecranLogique = ECRAN_LISTE;
        cache.flush();
        final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_SUPPRESSION_ROLE"), role.getLibelle());
        infoBean.addMessageConfirmation(confirmation);
        etat = FIN;
    }

    /**
     * Traitement associé à l'écran de saisie des critères.
     *
     * @throws Exception
     *             the exception
     */
    private void traiterRECHERCHE() throws Exception {
        if (action.equals(InfoBean.ACTION_MODIFIER)) {
            final ClauseWhere whereIdRole = new ClauseWhere(ConditionHelper.egalVarchar("ID_ROLE", infoBean.getString("ID_ROLE")));
            int count = role.select(whereIdRole.formaterSQL());
            if (count == 0) {
                infoBean.addMessageErreur("Aucun enregistrement sélectionné");
            }
            if (count == 1) {
                role.nextItem();
                if (mode == MODE_RECHERCHE) {
                    remplirDonneesRecherche();
                    etat = FIN;
                } else if (mode == MODE_MODIFICATION) {
                    preparerPRINCIPAL();
                } else if (mode == MODE_SUPPRESSION) {
                    preparerLISTE();
                }
            }
            if (count > 1) {
                preparerLISTE();
            }
        }
        if (action.equals(InfoBean.ACTION_SUPPRIMER)) {
            try {
                Long indice = new Long(infoBean.getString("ID_ROLE"));
                role.setIdRole(indice);
                role.retrieve();
                role.delete();
            } catch (final Exception e) {
            }
            cache.flush();
            etat = FIN;
        }
        if (action.equals(InfoBean.ACTION_ANNULER)) {
            if (mode == MODE_RECHERCHE) {
                infoBean.set("ID_ROLE", null);
                etat = FIN;
            }
        }
    }

    /**
     * Préparation des données à renvoyer pour une rechercher Role.
     *
     * @throws Exception
     *             the exception
     */
    private void remplirDonneesRecherche() throws Exception {
        infoBean.set("ID_ROLE", role.getIdRole());
        infoBean.set("CODE", role.getCode());
        infoBean.set("LIBELLE", role.getLibelle());
        infoBean.set("PERIMETRE", role.getPerimetre());
        infoBean.set("PERMISSIONS", role.getPermissions());
    }
}
