/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.processus;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.oro.text.regex.MalformedPatternException;
import org.apache.oro.text.regex.Perl5Compiler;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.ProcessusBean;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.jsbsoft.jtf.textsearch.sitesdistants.EtatsSitesIndexation;
import com.jsbsoft.jtf.textsearch.sitesdistants.IndexeurSitesDistants;
import com.jsbsoft.jtf.textsearch.sitesdistants.RechercheSitesDistants;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Site;

// TODO: Auto-generated Javadoc
/**
 * Processus de gestion des sites distants.
 */
public class SaisieSite extends ProcessusBean {

	/** The Constant ECRAN_PRINCIPAL. */
	private final static String ECRAN_PRINCIPAL = "PRINCIPAL";

	/** The Constant ECRAN_LISTE. */
	private final static String ECRAN_LISTE = "LISTE";

	/** The autorisations. */
	private AutorisationBean autorisations = null;

	/** The site. */
	private Site site = null;

	private static org.slf4j.Logger LOG = LoggerFactory.getLogger(SaisieSite.class);

	/**
	 * Constructeur.
	 *
	 * @param ciu
	 *            com.jsbsoft.jtf.core.InfoBean
	 */
	public SaisieSite(final InfoBean ciu) {
		super(ciu);
	}

	/**
	 * Point d'entrée du processus.
	 *
	 * @return true, if traiter action
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	public boolean traiterAction() throws Exception {
		autorisations = (AutorisationBean) getGp().getSessionUtilisateur().getInfos().get(SessionUtilisateur.AUTORISATIONS);
		if (autorisations == null) {
			infoBean.setEcranRedirection(WebAppUtil.CONNEXION_BO);
			infoBean.setEcranLogique("LOGIN");
		} else {
			try {
				ecranLogique = infoBean.getEcranLogique();
				action = infoBean.getActionUtilisateur();
				if (ecranLogique == null) {
					site = new Site();
					site.setCtx(this);
					site.init();
					etat = EN_COURS;
					if (action.equals("AJOUTER")) {
						infoBean.setEtatObjet(InfoBean.ETAT_OBJET_CREATION);
						preparerPRINCIPAL();
					} else if (action.equals("MODIFIER")) {
						infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
						preparerLISTE();
					} else if (action.equals("MODIFIERPARID")) {
						infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
						if (infoBean.get("ID_SITE") == null) {
							throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_SITE.ERREUR.SITE_A_MODIFIER_VIDE"));
						}
						site.setIdSite(Long.valueOf(infoBean.getString("ID_SITE")));
						site.retrieve();
						preparerPRINCIPAL();
					} else if (action.equals("SUPPRIMERPARID")) {
						if (infoBean.get("ID_SITE") == null) {
							throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_SITE.ERREUR.SITE_A_SUPPRIMER_VIDE"));
						}
						site.setIdSite(Long.valueOf(infoBean.getString("ID_SITE")));
						site.setCtx(this);
						site.retrieve();
						final File repertoireIndex = site.getRepertoireIndexation();
						site.delete();
						RechercheSitesDistants.init();
						if (repertoireIndex.exists()) {
							repertoireIndex.delete();
						}
						final String confirmation = String.format(MessageHelper.getCoreMessage(this.getLocale(), "CONFIRMATION_SUPPRESSION_SITEEXTERNE"), site.getLibelle());
						infoBean.addMessageConfirmation(confirmation);
						etat = FIN;
					} else if (action.equals("INDEXERPARID")) {
						if (infoBean.get("ID_SITE") == null) {
							throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_SITE.ERREUR.SITE_A_SUPPRIMER_VIDE"));
						}
						site.setIdSite(Long.valueOf(infoBean.getString("ID_SITE")));
						site.setCtx(this);
						site.retrieve();
						IndexeurSitesDistants.getInstance().indexeSite(site, false);
						preparerLISTE();
						etat = FIN;
					}
				} else {
					if (ecranLogique.equals(ECRAN_LISTE)) // modification ou suppression
					{
						traiterLISTE();
					} else if (ecranLogique.equals(ECRAN_PRINCIPAL)) // ajout ou modification
					{
						traiterPRINCIPAL();
					}
				}
				// placer l'état dans le composant d'infoBean
				infoBean.setEcranLogique(ecranLogique);
			} catch (final Exception e) {
				LOG.error("erreur de traitement sur le processus", e);
				infoBean.addMessageErreur(e.getMessage());
			}
		}
		// On continue si on n'est pas à la FIN !!!
		return (etat == FIN);
	}

	/**
	 * Affichage de l'écran des critères de recherche d'un Site.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerLISTE() throws Exception {
		ecranLogique = ECRAN_LISTE;
		final Map<Long, String> mapEtatSites = IndexeurSitesDistants.getInstance().getEtatsSites();
		final List<Site> toutLesSites = Site.getListeSites(this);
		final List<String> sitesEnCoursDIndexation = new ArrayList<String>();
		for (final Site siteCourant : toutLesSites) {
			if (StringUtils.isNotBlank(mapEtatSites.get(siteCourant.getIdSite()))) {
				siteCourant.setEtatIndexation(mapEtatSites.get(siteCourant.getIdSite()));
				if (EtatsSitesIndexation.ETAT_INDEXATION_EN_COURS.equals(siteCourant.getEtatIndexation())) {
					sitesEnCoursDIndexation.add(siteCourant.getLibelle());
				}
			} else {
				siteCourant.setEtatIndexation(EtatsSitesIndexation.ETAT_NON_LANCEE);
			}
		}
		infoBean.set("LISTE_SITES", toutLesSites);
		infoBean.set("SITES_EN_COURS_INDEXATION", sitesEnCoursDIndexation);
	}

	/**
	 * Affichage de l'écran de saisie d'un Site.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void preparerPRINCIPAL() throws Exception {
		ecranLogique = ECRAN_PRINCIPAL;
		infoBean.set("CODE", site.getCode());
		infoBean.set("LIBELLE", site.getLibelle());
		infoBean.set("URL", site.getUrl().equals("") ? "http://" : site.getUrl());
		infoBean.set("REG_EXP_ACCEPTE", site.getRegExpAccepte());
		infoBean.set("NIVEAU_PROFONDEUR", site.getNiveauProfondeur());
		infoBean.set("REG_EXP_REFUSE", site.getRegExpRefuse());
		infoBean.set("LISTE_LANGUES", LangueUtil.getListeLangues(getLocale()));
		infoBean.set("LANGUE", site.getLangue());
		if (StringUtils.isNotBlank(site.getLibelle())) {
			infoBean.setTitreEcran(site.getLibelle());
		}
	}

	/**
	 * Traitement associé à la sélection d'un site.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterLISTE() throws Exception {
		if (action.equals(InfoBean.ACTION_MODIFIER)) {
			if (infoBean.get("ID_SITE") == null) {
				throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_SITE.ERREUR.SITE_A_MODIFIER_VIDE"));
			}
			site.setIdSite(Long.valueOf(infoBean.getString("ID_SITE")));
			site.retrieve();
			preparerPRINCIPAL();
		} else if (action.equals(InfoBean.ACTION_SUPPRIMER)) {
			if (infoBean.get("ID_SITE") == null) {
				throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_SITE.ERREUR.SITE_A_SUPPRIMER_VIDE"));
			}
			site.setIdSite(Long.valueOf(infoBean.getString("ID_SITE")));
			site.setCtx(this);
			site.retrieve();
			final File repertoireIndex = site.getRepertoireIndexation();
			site.delete();
			RechercheSitesDistants.init();
			if (repertoireIndex.exists()) {
				repertoireIndex.delete();
			}
			etat = FIN;
		} else if (action.equals(InfoBean.ACTION_ANNULER) || action.equals(InfoBean.ACTION_ABANDON)) {
			etat = FIN;
		}
	}

	/**
	 * Gestion de l'écran de saisie d'un site distant.
	 *
	 * @throws Exception
	 *             the exception
	 */
	private void traiterPRINCIPAL() throws Exception {
		if (action.equals(InfoBean.ACTION_VALIDER)) {
			site.setLibelle((String) infoBean.get("LIBELLE"));
			String szUrl = (String) infoBean.get("URL");
			if (szUrl.indexOf("http://") == -1 || szUrl.equals("http://")) {
				throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_SITE.ERREUR.URL_INVALIDE"));
			}
			// on rajoute un / si besoin
			if (szUrl.charAt(szUrl.length() - 1) != '/') {
				// cas http://serveur
				if (szUrl.lastIndexOf('/') == "http://".lastIndexOf('/')) {
					szUrl += '/';
				}
			}
			site.setUrl(szUrl);
			// expression reguliere limitant le scope au site
			// On prend comme repère le dernier slash car si on demande d'indexer http://www.site.com/rubrique,
			// ce n'est pas pour que tout www.site.com soit indexé
			final String szRegExp = szUrl.substring(0, szUrl.lastIndexOf('/') + 1) + ".*";
			try {
				// verifications des expressions regulieres
				new Perl5Compiler().compile(szRegExp);
				site.setRegExpAccepte(szRegExp);
			} catch (final MalformedPatternException e) {
				throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_SITE.ERREUR.REGEXP_INVALIDE"));
			}
			final String szRegExpRefus = StringUtils.isNotBlank(infoBean.getString("REG_EXP_REFUSE")) ? infoBean.getString("REG_EXP_REFUSE") : StringUtils.EMPTY;
			try {
				new Perl5Compiler().compile(szRegExpRefus);
				site.setRegExpRefuse(szRegExpRefus);
			} catch (final MalformedPatternException e) {
				throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_SITE.ERREUR.REGEXP_INVALIDE"));
			}

			site.setNiveauProfondeur(Integer.valueOf(infoBean.getString("NIVEAU_PROFONDEUR")));
			if (infoBean.getEtatObjet().equals(InfoBean.ETAT_OBJET_CREATION)) {
				final String code = infoBean.getString("CODE");
				if (code.indexOf(';') != -1) {
					throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_SITE.ERREUR.CARACTERE_INTERDIT"));
				}
				site.setCode(code);
				site.add();
				final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_CREATION_SITEEXTERNE"), site.getLibelle());
				infoBean.addMessageConfirmation(confirmation);
			} else if (infoBean.getEtatObjet().equals(InfoBean.ETAT_OBJET_MODIF)) {
				site.update();
				final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_MODIFICATION_SITEEXTERNE"), site.getLibelle());
				infoBean.addMessageConfirmation(confirmation);
			}
			etat = FIN;
			RechercheSitesDistants.init();
		} else if (action.equals(InfoBean.ACTION_SUPPRIMER)) {
			if (infoBean.get("ID_SITE") == null) {
				throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "SAISIE_SITE.ERREUR.SITE_A_SUPPRIMER_VIDE"));
			}
			site.setIdSite(Long.valueOf(infoBean.getString("ID_SITE")));
			site.setCtx(this);
			site.retrieve();
			final File repertoireIndex = site.getRepertoireIndexation();
			site.delete();
			RechercheSitesDistants.init();
			if (repertoireIndex.exists()) {
				repertoireIndex.delete();
			}
			final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_SUPPRESSION_SITEEXTERNE"), site.getLibelle());
			infoBean.addMessageConfirmation(confirmation);
			etat = FIN;
		}
	}
}
