/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.processus;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.core.SynchroniseurUtilisateur;
import com.jsbsoft.jtf.database.ProcessusBean;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.ldap.ISynchroLdapUtilisateur;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.security.MySQLHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.kportal.extension.module.composant.ComposantUtilisateur;
import com.univ.collaboratif.om.Espacecollaboratif;
import com.univ.datagrid.processus.UtilisateurDatagrid;
import com.univ.datagrid.utils.DatagridUtils;
import com.univ.objetspartages.om.AnnuaireModele;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Groupedsi;
import com.univ.objetspartages.om.Groupeutilisateur;
import com.univ.objetspartages.om.InfosGroupeDsi;
import com.univ.objetspartages.om.InfosStructure;
import com.univ.objetspartages.om.Libelle;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.Profildsi;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Structure;
import com.univ.objetspartages.om.Utilisateur;
import com.univ.objetspartages.util.CritereRecherche;
import com.univ.objetspartages.util.CritereRechercheUtil;
import com.univ.utils.Chaine;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.RechercheFicheHelper;

/**
 * Processus de saisie d'un utilisateur.
 */
public class SaisieUtilisateur extends ProcessusBean {

    /** The Constant ECRAN_RECHERCHE. */
    private static final String ECRAN_RECHERCHE = "RECHERCHE";

    /** The Constant ECRAN_PRINCIPAL. */
    private static final String ECRAN_PRINCIPAL = "PRINCIPAL";

    /** The Constant ECRAN_LISTE. */
    private static final String ECRAN_LISTE = "LISTE";

    private static final Logger LOG = LoggerFactory.getLogger(SaisieUtilisateur.class);

    /** The utilisateur. */
    protected Utilisateur utilisateur = null;

    /** The liste identifiant. */
    private final Long listeIdentifiant[] = null;

    /**
     * processus saisie Utilisateur.
     *
     * @param ciu
     *            com.jsbsoft.jtf.core.InfoBean
     */
    public SaisieUtilisateur(final InfoBean ciu) {
        super(ciu);
    }

    /**
     * Affichage de l'écran de saisie d'un Utilisateur.
     *
     * @throws Exception
     *             the exception
     */
    public void preparerPRINCIPAL() throws Exception {
        ecranLogique = ECRAN_PRINCIPAL;
        infoBean.set("CODE", utilisateur.getCode());
        infoBean.set("CODE_LDAP", utilisateur.getCodeLdap());
        infoBean.set("CIVILITE", utilisateur.getCivilite());
        infoBean.set("NOM", utilisateur.getNom());
        infoBean.set("PRENOM", utilisateur.getPrenom());
        infoBean.set("CODE_RATTACHEMENT", utilisateur.getCodeRattachement());
        infoBean.set("LIBELLE_CODE_RATTACHEMENT", Structure.getLibelleAffichable(utilisateur.getCodeRattachement(), ""));
        infoBean.set("ADRESSE_MAIL", utilisateur.getAdresseMail());
        final String formatEnvoiNL = StringUtils.defaultIfBlank(utilisateur.getFormatEnvoi(), "0");
        infoBean.set("FORMAT_ENVOI", formatEnvoiNL);
        infoBean.set("RADIO0", "HTML");
        infoBean.set("RADIO1", "Texte");
        infoBean.set("TYPE_POPULATION", "");
        infoBean.set("LISTE_TYPES_POPULATION", Libelle.getListe("10", LangueUtil.getDefaultLocale()));
        infoBean.set("CORPS", "");
        infoBean.set("DISCIPLINE", "");
        infoBean.set("LISTE_DISCIPLINES", Libelle.getListe("02", LangueUtil.getDefaultLocale()));
        // Pas d'inclusion des groupes dynamiques
        final Vector<String> groupesDsi = Groupeutilisateur.getAllDatasourceGroupsCodesByUserCode(utilisateur.getCode());
        final Enumeration<String> en = groupesDsi.elements();
        String sGroupes = StringUtils.EMPTY;
        String libelles = StringUtils.EMPTY;
        while (en.hasMoreElements()) {
            final String currentCode = en.nextElement();
            final InfosGroupeDsi currentGroupe = Groupedsi.renvoyerItemGroupeDsi(currentCode);
            if (sGroupes.length() > 0) {
                sGroupes += ";";
                libelles += ";";
            }
            libelles = String.format("%s%s", libelles, StringUtils.defaultIfBlank(currentGroupe.getIntitule(), MessageHelper.getCoreMessage("BO.GROUP.LABEL.UNKNOWN")));
            sGroupes = String.format("%s%s%s", sGroupes, StringUtils.isNotBlank(currentGroupe.getRequete()) || !currentGroupe.isSelectionnable() ? "#AUTO#" : StringUtils.EMPTY, currentCode);
        }
        infoBean.set("GROUPE_DSI", sGroupes);
        infoBean.set("LIBELLE_GROUPE_DSI", libelles);
        final Collection<String> groupesDynamic = Groupeutilisateur.getAllDynamicGroupsCodesByUserCode(utilisateur.getCode());
        final Collection<InfosGroupeDsi> dynamicGroups = new ArrayList<>();
        for (String currentCode : groupesDynamic) {
            final InfosGroupeDsi currentGroupe = Groupedsi.renvoyerItemGroupeDsi(currentCode);
            if (currentGroupe != null) {
                dynamicGroups.add(currentGroupe);
            }
        }
        infoBean.set("GROUPE_DSI_DYN", dynamicGroups);
        infoBean.set("PROFIL_DEFAUT", utilisateur.getProfilDefaut());
        final Map<String, String> profiles = Profildsi.getListeProfilsDSIParGroupes(groupesDsi);
        profiles.putAll(Profildsi.getListeProfilsDSIParGroupes(new Vector<>(groupesDynamic)));
        infoBean.set("LISTE_PROFILS", profiles);
        /***********************/
        /*    Thématiques      */
        /***********************/
        final Vector<String> vThemesUtilisateurs = utilisateur.getVecteurCentresInteret();
        int iTheme = 0;
        /* tri des thèmes par libellés */
        final Libelle libelle = new Libelle();
        libelle.setCtx(this);
        if (libelle.select("WHERE TYPE='04' AND LANGUE='0' ORDER BY LIBELLE") > 0) {
            while (libelle.nextItem()) {
                /* Pour chaque theme, on regarde s'il est sélectionné */
                String valeurTheme = "0";
                if (vThemesUtilisateurs.indexOf(libelle.getCode()) != -1) {
                    valeurTheme = "1";
                }
                infoBean.set("CODE_THEME#" + iTheme, libelle.getCode());
                infoBean.set("LIBELLE_THEME#" + iTheme, libelle.getLibelle());
                infoBean.set("VALEUR_THEME#" + iTheme, valeurTheme);
                iTheme++;
            }
        }
        infoBean.set("THEMES_NB_ITEMS", iTheme);
        infoBean.set("ROLES", utilisateur.getRoles());
        // filtre les groupes dynamiques sur l'arbre
        infoBean.set("GRS_FILTRE_ARBRE_GROUPE_TMP_GROUPE_DSI", "1");
        infoBean.set("GRS_FILTRE_ARBRE_GROUPE_TYPE", "DYN");
        infoBean.set("GRS_FILTRE_ARBRE_GROUPE_OBJET", "dyn");
        infoBean.set("GRS_FILTRE_ARBRE_GROUPE_ACTION", "");
        // Si on modifie l'utilisateur et qu'il provient du ldap et que
        // l'authentification kportal est gérée par le ldap, cas ou cleartrust
        // on n'autorise pas la modification du mot de passe
        if (InfoBean.ETAT_OBJET_MODIF.equals(infoBean.getEtatObjet()) && utilisateur.isModificationMotDePasseInterditeParSource()) {
            infoBean.set("MODIFICATION_MOTDEPASSE_AUTORISEE", "0");
        } else {
            infoBean.set("MODIFICATION_MOTDEPASSE_AUTORISEE", "1");
        }
        if (InfoBean.ETAT_OBJET_MODIF.equals(infoBean.getEtatObjet()) && utilisateur.provientDuLdap()) {
            infoBean.set("MODIFICATION_AUTORISEE", "0");
        } else {
            infoBean.set("MODIFICATION_AUTORISEE", "1");
        }
        if (StringUtils.isNotBlank(utilisateur.getLibelle())) {
            infoBean.setTitreEcran(utilisateur.getLibelle());
        }
        ControleurAffectationRole.preparerPRINCIPAL(infoBean, this);
        // appel au traitement specifique (permet de surcharger le processus sans tout reecrire)
    }

    /**
     * Affichage de l'écran des critères de recherche d'un Utilisateur.
     *
     * @throws Exception
     *             the exception
     */
    private void preparerRECHERCHE() throws Exception {
        ecranLogique = ECRAN_RECHERCHE;
        infoBean.set("LISTE_PROFILS_DSI", Profildsi.getListeProfilsDSI());
    }

    /**
     * Point d'entrée du processus.
     *
     * @return true, if traiter action
     *
     * @throws Exception
     *             the exception
     */
    @Override
    public boolean traiterAction() throws Exception {
        try {
            ecranLogique = infoBean.getEcranLogique();
            action = infoBean.getActionUtilisateur();
            if (utilisateur == null) {
                utilisateur = new Utilisateur();
                utilisateur.setCtx(this);
                utilisateur.init();
            }
            final AutorisationBean autorisations = (AutorisationBean) getGp().getSessionUtilisateur().getInfos().get(SessionUtilisateur.AUTORISATIONS);
            String modeAffichage = infoBean.getString("MODE");
            if (modeAffichage == null) {
                modeAffichage = "GESTION";
            }
            infoBean.set("MODE", modeAffichage);
            // retour sur login si utilisateur non connecte
            if (autorisations == null) {
                infoBean.setEcranRedirection(WebAppUtil.CONNEXION_BO);
                ecranLogique = "LOGIN";
            } else if (!ComposantUtilisateur.isAutoriseParActionProcessusEtEcranLogique(autorisations, action, ecranLogique)) {
                throw new ErreurApplicative(MessageHelper.getCoreMessage("BO_OPERATION_INTERDITE"));
            } else if (ecranLogique == null) {
                if (action.equals("AJOUTER")) {
                    infoBean.setEtatObjet(InfoBean.ETAT_OBJET_CREATION);
                    preparerPRINCIPAL();
                } else if (action.equals("MODIFIER")) {
                    infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
                    preparerRECHERCHE();
                } else if (action.equals("RECHERCHER")) {
                    infoBean.setEtatObjet(InfoBean.ETAT_OBJET_RECHERCHE);
                    preparerRECHERCHE();
                } else if (action.equals("LISTE")) {
                    infoBean.setEtatObjet(InfoBean.ETAT_OBJET_RECHERCHE);
                    traiterRECHERCHE();
                } else if (action.equals("ACCUEIL")) {
                    action = InfoBean.ACTION_VALIDER;
                    infoBean.setEtatObjet(InfoBean.ETAT_OBJET_RECHERCHE);
                    traiterRECHERCHE();
                } else if (action.equals("LISTE_ROLES")) {
                    infoBean.setEtatObjet(InfoBean.ETAT_OBJET_RECHERCHE);
                    infoBean.set("LISTE", "ROLES");
                    preparerRECHERCHE();
                } else if (action.equals("MODIFIERPARID")) {
                    infoBean.setEtatObjet(InfoBean.ETAT_OBJET_MODIF);
                    final String id = infoBean.getString("ID");
                    if (StringUtils.isNotBlank(id) && StringUtils.isNumeric(id)) {
                        utilisateur.setIdUtilisateur(Long.valueOf(id));
                        utilisateur.retrieve();
                        preparerPRINCIPAL();
                    }
                } else if (action.equals("SUPPRIMERPARID")) {
                    infoBean.setEtatObjet(InfoBean.ETAT_OBJET_SUPPRESSION);
                    final String id = infoBean.getString("ID");
                    if (StringUtils.isNotBlank(id) && StringUtils.isNumeric(id)) {
                        utilisateur.setIdUtilisateur(Long.valueOf(id));
                        utilisateur.retrieve();
                        supprimerUtilisateur();
                        etat = FIN;
                    }
                }
            } else {
                if (ecranLogique.equals(ECRAN_RECHERCHE)) {
                    traiterRECHERCHE();
                } else if (ecranLogique.equals(ECRAN_LISTE)) {
                    traiterLISTE();
                } else if (ecranLogique.equals(ECRAN_PRINCIPAL)) {
                    traiterPRINCIPAL();
                }
            }
            // on place l'état dans le composant d'infoBean
            infoBean.setEcranLogique(ecranLogique);
        } catch (final Exception e) {
            LOG.error("erreur dans le traitement du processus", e);
            infoBean.addMessageErreur(e.toString());
        }
        // On continue si on n'est pas à la FIN !!!
        return (etat == FIN);
    }

    private void supprimerUtilisateur() throws Exception {
        if(Espacecollaboratif.isExtensionActivated()) {
            try {
                Espacecollaboratif.supprimerUtilisateur(utilisateur.getCode());
            } catch (final Exception e) {
                action = "LISTE";
                traiterRECHERCHE();
                throw new ErreurApplicative(e.toString());
            }
        }
        Groupeutilisateur.deleteParUtilisateur(this, utilisateur.getCode());
        utilisateur.delete();
        final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_SUPPRESSION_UTILISATEUR"), utilisateur.getLibelle());
        infoBean.addMessageConfirmation(confirmation);
    }

    /**
     * Traitement de l'écran de sélection d'un Utilisateur.
     *
     * @throws Exception
     *             the exception
     */
    private void traiterLISTE() throws Exception {
        if (action.equals(InfoBean.ACTION_MODIFIER)) {
            final int indice = Integer.parseInt((String) infoBean.get("LISTE_INDICE"));
            utilisateur.setIdUtilisateur(listeIdentifiant[indice]);
            utilisateur.retrieve();
            preparerPRINCIPAL();
        } else if (action.equals(InfoBean.ACTION_SUPPRIMER)) {
            final int indice = Integer.parseInt((String) infoBean.get("LISTE_INDICE"));
            utilisateur.setIdUtilisateur(listeIdentifiant[indice]);
            utilisateur.retrieve();
            supprimerUtilisateur();
            etat = FIN;
        } else if (action.equals("LISTE")) { // refresh
            traiterRECHERCHE();
        } else if (action.equals(InfoBean.ACTION_ANNULER)) {
            infoBean.set("ID_UTILISATEUR", null);
            etat = FIN;
        }
    }

    /**
     * Gestion de l'écran de saisie d'un Utilisateur.
     *
     * @throws Exception
     *             the exception
     */
    public void traiterPRINCIPAL() throws Exception {
        ControleurAffectationRole.traiterPRINCIPAL(infoBean, this);
        if (action.equals("SUPPRIMER")) {
            supprimerUtilisateur();
            etat = FIN;
        } else if (action.equals("ACTUALISER_PROFIL")) {
            final Vector<String> vGroupesDsi = new Vector<>();
            final StringTokenizer st = new StringTokenizer(infoBean.getString("GROUPE_DSI"), ";");
            while (st.hasMoreTokens()) {
                final String val = st.nextToken();
                vGroupesDsi.add(val);
            }
            final Hashtable<String, String> hProfil = Profildsi.getListeProfilsDSIParGroupes(vGroupesDsi);
            if (hProfil.containsKey(utilisateur.getProfilDefaut())) {
                infoBean.set("PROFIL_DEFAUT", utilisateur.getProfilDefaut());
            } else {
                infoBean.set("PROFIL_DEFAUT", "");
            }
            infoBean.set("LISTE_PROFILS", hProfil);
        } else if (action.equals(InfoBean.ACTION_VALIDER)) {
            if (infoBean.getEtatObjet().equals(InfoBean.ETAT_OBJET_CREATION)) {
                Chaine.controlerCodeMetier((String) infoBean.get("CODE"));
                if (Utilisateur.getUtilisateur(infoBean.getString("CODE")).getCode().equalsIgnoreCase(infoBean.getString("CODE"))) {
                    throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "UTILISATEUR.ERREUR.IDENTIFIANT_EXISTANT"));
                }
                utilisateur.setCode((String) infoBean.get("CODE"));
                final String codeLdap = (String) infoBean.get("CODE_LDAP");
                if ((codeLdap != null) && !codeLdap.equals("")) {
                    if (utilisateur.select("WHERE CODE_LDAP = '" + codeLdap + "'") > 0) {
                        throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "UTILISATEUR.ERREUR.LOGIN_EXISTANT"));
                    }
                    ISynchroLdapUtilisateur synchroLdap = SynchroniseurUtilisateur.getInstance().getSynchroLdap();
                    if (synchroLdap != null && !(synchroLdap.isUnique(codeLdap))) {
                        throw new ErreurApplicative(MessageHelper.getCoreMessage(this.getLocale(), "UTILISATEUR.ERREUR.LOGIN_LDAP_EXISTANT"));
                    }
                    utilisateur.setCodeLdap(codeLdap);
                }
            }
            if (StringUtils.isNotBlank(infoBean.getString("NEW_MOT_DE_PASSE"))) {
                String message;
                if (!infoBean.getString("NEW_MOT_DE_PASSE").equals(infoBean.getString("CONFIRM_MOT_DE_PASSE"))) {
                    message = "Vous avez saisi 2 mots de passe différents";
                } else {
                    message = Utilisateur.verifieMotDePasse(infoBean.getString("NEW_MOT_DE_PASSE"));
                }
                if (message.length() > 0) {
                    throw new ErreurApplicative(message);
                }
                utilisateur.setMotDePasse(MySQLHelper.encodePassword(infoBean.getString("NEW_MOT_DE_PASSE")));
            }
            if (infoBean.get("FORMAT_ENVOI") != null) {
                utilisateur.setFormatEnvoi((String) infoBean.get("FORMAT_ENVOI"));
            }
            utilisateur.setCivilite((String) infoBean.get("CIVILITE"));
            utilisateur.setNom((String) infoBean.get("NOM"));
            utilisateur.setPrenom((String) infoBean.get("PRENOM"));
            utilisateur.setCodeRattachement((String) infoBean.get("CODE_RATTACHEMENT"));
            utilisateur.setAdresseMail((String) infoBean.get("ADRESSE_MAIL"));
            String profilDSI = infoBean.getString("PROFIL_DEFAUT");
            if (profilDSI.equals("0000")) {
                profilDSI = "";
            }
            utilisateur.setProfilDefaut(profilDSI);
            final Vector<String> vGroupesDsi = new Vector<>();
            final StringTokenizer st = new StringTokenizer(infoBean.getString("GROUPE_DSI"), ";");
            while (st.hasMoreTokens()) {
                final String val = st.nextToken();
                vGroupesDsi.add(val);
            }
            // Gestion des centres d'interet
            final Vector<String> v = new Vector<>();
            for (int j = 0; j < infoBean.getInt("THEMES_NB_ITEMS"); j++) {
                if ("1".equals(infoBean.getString("VALEUR_THEME#" + j))) {
                    v.add(infoBean.getString("CODE_THEME#" + j));
                }
            }
            utilisateur.setVecteurCentresInteret(v);
            utilisateur.setRoles(infoBean.getString("ROLES"));
            if (infoBean.getEtatObjet().equals(InfoBean.ETAT_OBJET_CREATION)) {
                utilisateur.add();
                Groupeutilisateur.setVecteurGroupes(this, utilisateur.getCode(), vGroupesDsi);
				/* import de la fiche */
                if ("1".equals(infoBean.getString("CREER_FICHE"))) {
                    final String typeFiche = infoBean.getString("TYPE_FICHE");
                    if (StringUtils.isNotBlank(typeFiche)) {
                        final FicheUniv ficheUniv = ReferentielObjets.instancierFiche(typeFiche);
                        if (ficheUniv != null) {
                            ficheUniv.setCtx(this);
                            ficheUniv.init();
                            ficheUniv.setCode((String) infoBean.get("CODE"));
                            ficheUniv.setCodeRedacteur((String) infoBean.get("CODE"));
                            ficheUniv.setCodeRattachement((String) infoBean.get("CODE_RATTACHEMENT"));
                            ((AnnuaireModele) ficheUniv).setCivilite((String) infoBean.get("CIVILITE"));
                            ((AnnuaireModele) ficheUniv).setDateNaissance((java.sql.Date) infoBean.get("DATE_NAISSANCE"));
                            ((AnnuaireModele) ficheUniv).setNom((String) infoBean.get("NOM"));
                            ((AnnuaireModele) ficheUniv).setPrenom((String) infoBean.get("PRENOM"));
                            ((AnnuaireModele) ficheUniv).setAdresseMail((String) infoBean.get("ADRESSE_MAIL"));
                            ((AnnuaireModele) ficheUniv).setTypePopulation((String) infoBean.get("TYPE_POPULATION"));
                            ((AnnuaireModele) ficheUniv).setCorps((String) infoBean.get("CORPS"));
                            ((AnnuaireModele) ficheUniv).setDiscipline((String) infoBean.get("DISCIPLINE"));
                            if ("1".equals(infoBean.get("CREER_FICHE_EN_LIGNE"))) {
                                ficheUniv.setEtatObjet("0003");
                            }
                            ficheUniv.add();
                            final Metatag meta = FicheUnivMgr.lireMeta(ficheUniv);
                            meta.synchroniser(ficheUniv, false);
                            FicheUnivMgr.enregistrerMeta(meta);
                        }
                    }
                }
                final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_CREATION_UTILISATEUR"), utilisateur.getLibelle());
                infoBean.addMessageConfirmation(confirmation);
            }
            if (infoBean.getEtatObjet().equals(InfoBean.ETAT_OBJET_MODIF)) {
                utilisateur.update();
                Groupeutilisateur.setVecteurGroupes(this, utilisateur.getCode(), vGroupesDsi);
                final String confirmation = String.format(MessageHelper.getCoreMessage("CONFIRMATION_MODIFICATION_UTILISATEUR"), utilisateur.getLibelle());
                infoBean.addMessageConfirmation(confirmation);
            }
            // appel au traitement specifique (permet de surcharger le processus sans tout reecrire)
            if (!ecranLogique.equals("CONFIRMATION")) {
                etat = FIN;
            }
        }
    }

    /**
     * Traitement associé à l'écran de saisie des critères.
     *
     * @throws Exception
     *             the exception
     */
    private void traiterRECHERCHE() throws Exception {
        if (action.equals(InfoBean.ACTION_VALIDER) || action.equals("LISTE")) {
            final List<CritereRecherche> criteres = new ArrayList<>();
            CollectionUtils.addIgnoreNull(criteres, CritereRechercheUtil.getCritereTexteNonVide(infoBean, "CODE"));
            CollectionUtils.addIgnoreNull(criteres, CritereRechercheUtil.getCritereTexteNonVideFormater(infoBean, "NOM"));
            CollectionUtils.addIgnoreNull(criteres, CritereRechercheUtil.getCritereTexteNonVideFormater(infoBean, "PRENOM"));
            CollectionUtils.addIgnoreNull(criteres, CritereRechercheUtil.getCritereTexteNonVideFormater(infoBean, "ADRESSE_MAIL"));
            final String codeProfil = infoBean.getString("PROFIL_DSI");
            if (StringUtils.isNotBlank(codeProfil)) {
                final Hashtable<String, String> listeProfil = Profildsi.getListeProfilsDSI();
                if (listeProfil != null && listeProfil.get(codeProfil) != null) {
                    criteres.add(new CritereRecherche("PROFIL_DSI", codeProfil, listeProfil.get(codeProfil)));
                }
            }
            final String groupeDSI = infoBean.getString("GROUPE_DSI");
            if (StringUtils.isNotBlank(groupeDSI)) {
                final String libelleGroupeDSI = Groupedsi.getIntitule(groupeDSI);
                criteres.add(new CritereRecherche("GROUPE_DSI", groupeDSI, libelleGroupeDSI));
                criteres.add(new CritereRecherche("LIBELLE_GROUPE_DSI", libelleGroupeDSI));
            }
            final String codeStructure = infoBean.getString("CODE_RATTACHEMENT");
            if (StringUtils.isNotBlank(codeStructure)) {
                try {
                    final InfosStructure structure = Structure.renvoyerItemStructure(codeStructure, LangueUtil.getLangueLocale(this.getLocale()));
                    final String libelleStructure = structure.getLibelleLong();
                    criteres.add(new CritereRecherche("CODE_RATTACHEMENT", codeStructure, libelleStructure));
                    criteres.add(new CritereRecherche("LIBELLE_CODE_RATTACHEMENT", libelleStructure));
                } catch (final Exception e) {
                    LOG.debug("impossible de récuperer le libelle structure dans la recherche");
                }
            }
            criteres.add(new CritereRecherche(DatagridUtils.PARAM_BEAN_DATAGRID, UtilisateurDatagrid.ID_BEAN));
            infoBean.set(RechercheFicheHelper.ATTRIBUT_INFOBEAN_CRITERES, criteres);
            ecranLogique = ECRAN_LISTE;
        } else if (action.equals(InfoBean.ACTION_ANNULER)) {
            if (infoBean.getEtatObjet().equals(InfoBean.ETAT_OBJET_RECHERCHE)) {
                infoBean.set("ID_UTILISATEUR", null);
                etat = FIN;
            }
        }
    }
}
