/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.processus;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.database.ProcessusBean;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.core.config.PropertyHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.DiffusionSelective;
import com.univ.objetspartages.om.FicheRattachementsSecondaires;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.PermissionBean;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Rubrique;
import com.univ.objetspartages.om.Utilisateur;
import com.univ.utils.FicheUnivMgr;
// TODO: Auto-generated Javadoc

/**
 * processus saisie Structure AM 02.2003 Modif cf preparerListeAValider
 */
public class Validation extends ProcessusBean {

    private static org.slf4j.Logger LOG = LoggerFactory.getLogger(Validation.class);

    /** The Constant ECRAN_SELECTION_NIVEAU_APPROBATION. */
    private static final String ECRAN_SELECTION_NIVEAU_APPROBATION = "APPROBATION";

    /** The Constant ECRAN_SELECTION_VALIDATEUR. */
    private static final String ECRAN_SELECTION_VALIDATEUR = "VALIDATEUR";

    /** The Constant ECRAN_CONFIRMATION. */
    private static final String ECRAN_CONFIRMATION = "CONFIRMATION";

    /** The Constant ECRAN_LISTE. */
    private static final String ECRAN_LISTE = "LISTE";

    /** The autorisations. */
    AutorisationBean autorisations;

    /** The fiche univ. */
    FicheUniv ficheUniv;

    /**
     * processus saisie Structure.
     *
     * @param ciu
     *            com.jsbsoft.jtf.core.InfoBean
     */
    public Validation(final InfoBean ciu) {
        super(ciu);
    }

    /**
     * Point d'entrée du processus.
     *
     * @return true, if traiter action
     *
     * @throws Exception
     *             the exception
     */
    @Override
    public boolean traiterAction() throws Exception {
        final Object o = getGp().getSessionUtilisateur().getInfos().get(SessionUtilisateur.AUTORISATIONS);
        if (o == null) {
            infoBean.setEcranRedirection(WebAppUtil.CONNEXION_BO);
            infoBean.setEcranLogique("LOGIN");
        } else {
            try {
                autorisations = (AutorisationBean) o;
                ecranLogique = infoBean.getEcranLogique();
                action = infoBean.getActionUtilisateur();
                etat = EN_COURS;
                if (infoBean.get("CODE_OBJET") != null) {
                    ficheUniv = ReferentielObjets.instancierFiche(ReferentielObjets.getNomObjet(infoBean.getString("CODE_OBJET")));
                    ficheUniv.setCtx(this);
                    ficheUniv.init();
                    // on renseigne le périmètre saisi par l'utilisateur
                    if (infoBean.get("CODE_RATTACHEMENT") != null) {
                        ficheUniv.setCodeRattachement(infoBean.getString("CODE_RATTACHEMENT"));
                    }
                    // JSS 20051031 : structures secondaires
                    if (ficheUniv instanceof FicheRattachementsSecondaires && (infoBean.get("CODE_RATTACHEMENT_AUTRES") != null)) {
                        ((FicheRattachementsSecondaires) ficheUniv).setCodeRattachementAutres(infoBean.getString("CODE_RATTACHEMENT_AUTRES"));
                    }
                    if (infoBean.get("CODE_RUBRIQUE") != null) {
                        ficheUniv.setCodeRubrique(infoBean.getString("CODE_RUBRIQUE"));
                    }
                    if (ficheUniv instanceof DiffusionSelective && infoBean.get("PUBLIC_VISE_DSI") != null) {
                        ((DiffusionSelective) ficheUniv).setDiffusionPublicVise(infoBean.getString("PUBLIC_VISE_DSI"));
                    }
                }
                if (ecranLogique == null) {
                    if (action.equals("LISTE")) {
                        preparerListeAValider(autorisations.getCode(), autorisations);
                    } else if (action.equals("APPROBATION")) {
                        preparerListeApprobation(infoBean.getString("NIVEAU_APPROBATION"));
                    }
                } else {
                    if (action.equals(InfoBean.ACTION_ANNULER)) {
                        etat = FIN;
                    } else if (ecranLogique.equals(ECRAN_SELECTION_NIVEAU_APPROBATION)) {
                        preparerListeValidateur(infoBean.getString("NIVEAU_APPROBATION"));
                    } else if (ecranLogique.equals(ECRAN_SELECTION_VALIDATEUR)) {
                        traiterListeValidateur();
                    }
                }
                //placer l'état dans le composant d'infoBean
                infoBean.setEcranLogique(ecranLogique);
            } catch (final Exception e) {
                LOG.error("erreur de traitement sur le processus", e);
                infoBean.addMessageErreur(e.toString());
            }
        }
        infoBean.set("NOM_ONGLET", "valider");
        // On continue si on n'est pas à la FIN !!!
        return (etat == FIN);
    }

    /**
     * Affichage de la liste des objets à valider.
     *
     * @param ficheUniv
     *            the fiche univ
     * @param codeObjet
     *            the code objet
     * @param nbItems
     *            the nb items
     *
     * @return the int
     *
     * @throws Exception
     *             the exception
     */
    protected int insererElement(final FicheUniv ficheUniv, final String codeObjet, int nbItems) throws Exception {
        infoBean.set("NOM_PROCESSUS#" + nbItems, ReferentielObjets.getProcessus(codeObjet));
        infoBean.set("EXTENSION#" + nbItems, ReferentielObjets.getExtension(codeObjet));
        infoBean.set("ID_FICHE#" + nbItems, ficheUniv.getIdFiche().toString());
        infoBean.set("LIBELLE#" + nbItems, ficheUniv.getLibelleAffichable());
        infoBean.set("NOM_OBJET#" + nbItems, ReferentielObjets.getLibelleObjet(codeObjet));
        infoBean.set("CODE_RUBRIQUE#" + nbItems, Rubrique.getIntitule(ficheUniv.getCodeRubrique()));
        infoBean.set("DATE_PROPOSITION#" + nbItems, ficheUniv.getDateProposition());
        infoBean.set("LIBELLE_REDACTEUR#" + nbItems, Utilisateur.getLibelle(ficheUniv.getCodeRedacteur()));
        infoBean.set("LANGUE#" + nbItems, ficheUniv.getLangue());
        infoBean.set("ETAT_OBJET#" + nbItems, ficheUniv.getEtatObjet());
        nbItems++;
        return nbItems;
    }

    /**
     * Preparer liste approbation.
     *
     * @param _code
     *            the _code
     *
     * @throws Exception
     *             the exception
     */
    private void preparerListeApprobation(final String _code) throws Exception {
        final Hashtable<String, String> hPermissionsValidation = AutorisationBean.getListeNiveauxApprobation();
        final Iterator<String> it = AutorisationBean.getListePermissionsValidation().iterator();
        boolean possedePermissionApprobation = true;
        boolean possedeTouteLesPermissions = true;
        boolean debut = false;
        String codeApprobation = "";
        int i = 0;
        // on boucle sur les permissions d'approbation
        if (it.hasNext()) {
            while (it.hasNext()) {
                // si permission précedente ok on continue
                if (possedePermissionApprobation) {
                    codeApprobation = it.next();
                    // initialisation du code d'approbation demandé
                    if (_code.length() == 0) {
                        debut = true;
                    } else if (_code.equals(codeApprobation)) {
                        if (it.hasNext()) {
                            codeApprobation = it.next();
                            debut = true;
                        } else {
                            codeApprobation = "V";
                        }
                    }
                    if (debut) {
                        if (!autorisations.getAutorisationParFiche(ficheUniv, AutorisationBean.INDICE_APPROBATION, codeApprobation)) {
                            possedePermissionApprobation = false;
                            possedeTouteLesPermissions = false;
                            infoBean.set("CODE_NIVEAU_AUTOMATIQUE", codeApprobation);
                            infoBean.set("LIBELLE_NIVEAU_AUTOMATIQUE", "A valider : " + hPermissionsValidation.get(codeApprobation));
                        } else {
                            infoBean.set("CODE_NIVEAU_APPROBATION#" + i, codeApprobation);
                            infoBean.set("LIBELLE_NIVEAU_APPROBATION#" + i, hPermissionsValidation.get(codeApprobation));
                            i++;
                        }
                    }
                } else {
                    break;
                }
            }
            // validation classique
        } else {
            codeApprobation = "V";
        }
        // si il les a toutes on regarde la permission de validation classique
        if (possedeTouteLesPermissions) {
            if (autorisations.getAutorisationParFiche(ficheUniv, AutorisationBean.INDICE_MISE_EN_LIGNE_UNITAIRE, "")) {
                infoBean.set("CODE_NIVEAU_APPROBATION#" + i, "V");
                infoBean.set("LIBELLE_NIVEAU_APPROBATION#" + i, "Mise en ligne");
                infoBean.set("CODE_NIVEAU_AUTOMATIQUE", "M");
                infoBean.set("LIBELLE_NIVEAU_AUTOMATIQUE", "En ligne");
                i++;
            } else {
                infoBean.set("CODE_NIVEAU_AUTOMATIQUE", "V");
                infoBean.set("LIBELLE_NIVEAU_AUTOMATIQUE", "A valider : Mise en ligne");
            }
        }
        // Si il y a plus d'une approbation possible renvoi sur l'écran de sélection d'une approbation
        if (i > 0) {
            infoBean.setInt("NB_ITEMS_NIVEAU_APPROBATION", i);
            ecranLogique = ECRAN_SELECTION_NIVEAU_APPROBATION;
        }
        // sinon renvoi directement sur la selection des validateurs
        else {
            preparerListeValidateur(codeApprobation);
        }
    }

    /**
     * Preparer liste validateur.
     *
     * @param codeApprobation
     *            the code approbation
     *
     * @throws Exception
     *             the exception
     */
    private void preparerListeValidateur(final String codeApprobation) throws Exception {
        if (codeApprobation.equals("M") == false) {
            String libelleApprobation = "";
            final Hashtable<String, String> hPermissionsValidation = AutorisationBean.getListeNiveauxApprobation();
            if (codeApprobation.equals("V")) {
                libelleApprobation = "Mise en ligne";
            } else {
                libelleApprobation = hPermissionsValidation.get(codeApprobation);
            }
            String codeUtilisateur = "";
            final Set<String> mails = getUtilisateurMailParFicheUnivEtCodeApprobation(codeApprobation, ficheUniv);
            int i = 0;
            if (mails != null) {
                final Iterator<String> iter = mails.iterator();
                while (iter.hasNext()) {
                    codeUtilisateur = iter.next();
                    // on filtre le redacteur
                    if (!codeUtilisateur.equals(autorisations.getCode())) {
                        infoBean.set("CODE_VALIDATEUR_" + i, codeUtilisateur);
                        infoBean.set("NOM_VALIDATEUR_" + i, Utilisateur.getLibelle(codeUtilisateur));
                        i++;
                    }
                }
            }
            if (!infoBean.get("CODE_NIVEAU_AUTOMATIQUE").equals(codeApprobation)) {
                infoBean.set("NIVEAU_APPROBATION", codeApprobation);
            } else {
                infoBean.set("NIVEAU_APPROBATION", "");
            }
            infoBean.set("LIBELLE_APPROBATION", libelleApprobation);
            infoBean.setInt("NB_ITEMS_VALIDATEUR", i);
            ecranLogique = ECRAN_SELECTION_VALIDATEUR;
        } else {
            infoBean.set("NIVEAU_APPROBATION", "M");
            infoBean.set("LIBELLE_APPROBATION", "");
            infoBean.set("CHAINE_VALIDATEURS", "");
            infoBean.set("LISTE_VALIDATEURS", "");
            ecranLogique = ECRAN_CONFIRMATION;
        }
    }

    /**
     * Traiter liste validateur.
     *
     * @throws Exception
     *             the exception
     */
    private void traiterListeValidateur() throws Exception {
        String chaineValidateurs = "";
        String listeValidateurs = "";
        for (int i = 0; i < ((Integer) infoBean.get("NB_ITEMS_VALIDATEUR")).intValue(); i++) {
            if (infoBean.get("VALIDATEUR_" + i) != null) {
                if (chaineValidateurs.length() > 0) {
                    listeValidateurs += ";";
                    chaineValidateurs += ", ";
                }
                listeValidateurs += infoBean.getString("CODE_VALIDATEUR_" + i);
                chaineValidateurs += Utilisateur.getLibelle(infoBean.getString("CODE_VALIDATEUR_" + i));
            }
        }
        if (listeValidateurs.length() == 0) {
            listeValidateurs = "[AUCUNS]";
            chaineValidateurs = "[aucuns]";
        }
        infoBean.set("CHAINE_VALIDATEURS", chaineValidateurs);
        infoBean.set("LISTE_VALIDATEURS", listeValidateurs);
        ecranLogique = ECRAN_CONFIRMATION;
    }

    /**
     * Affichage de la liste des objets à valider AM 02.2003 Modif sur les périmètres de validation
     *
     * @param codeValidateur
     *            the code validateur
     * @param autorisations
     *            the autorisations
     *
     * @throws Exception
     *             the exception
     */
    private void preparerListeAValider(final String codeValidateur, final AutorisationBean autorisations) throws Exception {
        int nbItems = 0;
        String codeApprobation = "";
        /************************************************************
         * Traitement générique : droit de validations sur les objets
         *
         * Il peut exister une structure de restriction
         *************************************************************/
        for (final String codeObjet : ReferentielObjets.getListeCodesObjet()) {
            if (autorisations.getAutorisation(codeObjet, AutorisationBean.INDICE_VALIDATION)) {
                ficheUniv = ReferentielObjets.instancierFiche(codeObjet);
                if (ficheUniv == null) {
                    continue;
                }
                ficheUniv.setCtx(this);
                ficheUniv.init();
                final int count = ficheUniv.selectCodeLangueEtat("", "", "0002");
                if (count > 0) {
                    while (ficheUniv.nextItem()) {
                        // JSS-20051014 : pas d'affichage des fiches collaboratives
                        // dans le back
                        boolean ficheEspace = false;
                        if (ficheUniv instanceof DiffusionSelective) {
                            if (((DiffusionSelective) ficheUniv).getDiffusionModeRestriction().equals("4")) {
                                ficheEspace = true;
                            }
                        }
                        if (!ficheEspace) {
                            codeApprobation = "";
                            // RP 20050811 affinage selon les niveaux d'approbation
                            final Metatag meta = FicheUnivMgr.lireMeta(ficheUniv);
                            if (meta.getMetaNiveauApprobation().equals("") == false) {
                                codeApprobation = meta.getMetaNiveauApprobation();
                            }
                            // JSS 2004-04-19 : délégation (appel autorisations)
                            if (autorisations.getAutorisationParFiche(ficheUniv, AutorisationBean.INDICE_APPROBATION, codeApprobation)) {
                                nbItems = insererElement(ficheUniv, codeObjet, nbItems);
                            }
                        }
                    }
                }
            }
        }
        nbItems = traiterValidationsSpecifiques(nbItems, autorisations);
        infoBean.set("LISTE_NB_ITEMS", new Integer(nbItems));
        ecranLogique = ECRAN_LISTE;
    }

    /**
     * Traitement des validations spécifiques (A SOUS-CLASSER).
     *
     * @param _nbItems
     *            the _nb items
     * @param autorisations
     *            the autorisations
     *
     * @return the int
     *
     * @throws Exception
     *             the exception
     */
    protected int traiterValidationsSpecifiques(final int _nbItems, final AutorisationBean autorisations) throws Exception {
        return _nbItems;
    }

    /**
     * Vérifie si des utilisateurs ont le niveau d'approbation suivant.
     * Si ce n'est pas le cas le processus de validation est bloqué
     * @param codeApprobation
     * @param ficheUniv
     * @return
     */
    public static boolean hasUserAuthorizedNextLevel(final String codeApprobation, final FicheUniv ficheUniv) {
        if ("1".equals(PropertyHelper.getCoreProperty("validation_multi_niveaux.activation"))) {
            if (StringUtils.isNotEmpty(codeApprobation) && !StringUtils.equals(codeApprobation, "M") && !StringUtils.equals(codeApprobation, "V")) {
                return CollectionUtils.isNotEmpty(getUtilisateurMailParFicheUnivEtCodeApprobation(codeApprobation, ficheUniv));
            }
        }
        return true;
    }

    /**
     * Renvoie la liste des utilisateurs qui ont le niveau d'approbation requis pour une fiche donnée
     * @param codeApprobation
     * @param ficheUniv
     * @return
     */
    private static Set<String> getUtilisateurMailParFicheUnivEtCodeApprobation(final String codeApprobation, final FicheUniv ficheUniv) {
        final PermissionBean permissionBean = new PermissionBean("FICHE", ReferentielObjets.getCodeObjet(ficheUniv), codeApprobation);
        final String codeRubrique = ficheUniv.getCodeRubrique();
        final String publicsVises = StringUtils.EMPTY;
        String espace = StringUtils.EMPTY;
        if (ficheUniv instanceof DiffusionSelective) {
            // JSS 20051201 : suppression périmètres groupes sur les fiches
            //publicsVises = ((DiffusionSelective) ficheUniv).getDiffusionPublicVise();
            if (((DiffusionSelective) ficheUniv).getDiffusionModeRestriction().equals("4")) {
                espace = ((DiffusionSelective) ficheUniv).getDiffusionPublicViseRestriction();
            }
        }
        Set<String> mails = null;
        try {
            mails = Utilisateur.getListeUtilisateursPossedantPermission(permissionBean, AutorisationBean.getStructuresPerimetreFiche(ficheUniv), ficheUniv.getCodeRubrique(), publicsVises, espace, false);
        } catch (Exception e) {
            LOG.error("Erreur lors de la récupération des utilisateurs", e);
        }
        return mails;
    }

    /**
     * Renvoi le libelle du niveau d'approbation à partir de son code
     * @param codeApprobation
     * @return
     */
    public static String getLibelleAprobationParCodeNiveau(final String codeApprobation) {
        final Map<String, String> hPermissionsValidation = AutorisationBean.getListeNiveauxApprobation();
        if (hPermissionsValidation != null) {
            return StringUtils.defaultString(hPermissionsValidation.get(codeApprobation));
        }
        return StringUtils.EMPTY;
    }
}
