/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.util;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.Vector;

import com.univ.utils.ContexteDao;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.InfoBean;
import com.jsbsoft.jtf.database.OMContext;
import com.univ.objetspartages.bean.HistoriqueBean;
import com.univ.objetspartages.om.EtatFiche;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Utilisateur;
import com.univ.utils.FicheUnivMgr;

// TODO: Auto-generated Javadoc
/**
 * processus saisie Utilisateur.
 */
public class AffichageHistorique {

	private static final Logger LOG = LoggerFactory.getLogger(AffichageHistorique.class);

	public static List<HistoriqueBean> getListeHistorique(final OMContext ctx, final InfoBean infoBean) throws Exception {
		final String codeObjet = infoBean.getString("CODE_OBJET");
		final Long idFiche = Long.valueOf(infoBean.getString("ID_FICHE"));
		final FicheUniv ficheUniv = retrouverFiche(ctx, ReferentielObjets.getNomObjet(codeObjet), idFiche);
		return getAllHistorique(ficheUniv);
	}

	public static List<HistoriqueBean> getAllHistorique(final FicheUniv ficheUniv) throws Exception {
		List<HistoriqueBean> historiqueDispo = new ArrayList<>();
		if (ficheUniv != null) {
			Metatag meta = FicheUnivMgr.lireMeta(ficheUniv);
			historiqueDispo.addAll(instancierFicheHistoriqueDepuisInfoMeta(ficheUniv, meta.getVecteurHistorique()));
			String codeObjet = ReferentielObjets.getCodeObjet(ficheUniv);
			FicheUniv fichesAparentees = ReferentielObjets.instancierFiche(codeObjet);
			try (ContexteDao ctx = new ContexteDao()) {
				fichesAparentees.setCtx(ctx);
				fichesAparentees.selectCodeLangueEtat(ficheUniv.getCode(), ficheUniv.getLangue(), "");
				while (fichesAparentees.nextItem()) {
					if (!fichesAparentees.getEtatObjet().equals( ficheUniv.getEtatObjet()) || !fichesAparentees.getIdFiche().equals(ficheUniv.getIdFiche())) {
						meta = FicheUnivMgr.lireMeta(codeObjet, ficheUniv.getIdFiche());
						historiqueDispo.addAll(instancierFicheHistoriqueDepuisInfoMeta(ficheUniv, meta.getVecteurHistorique()));
					}
				}
			}
		}
		return historiqueDispo;
	}

	public static List<HistoriqueBean> getHistorique(final FicheUniv ficheUniv) throws Exception {
		List<HistoriqueBean> historiqueDispo = new ArrayList<>();
		if (ficheUniv != null) {
			Metatag meta = FicheUnivMgr.lireMeta(ficheUniv);
			historiqueDispo.addAll(instancierFicheHistoriqueDepuisInfoMeta(ficheUniv, meta.getVecteurHistorique()));
		}
		return historiqueDispo;
	}

	private static List<HistoriqueBean> instancierFicheHistoriqueDepuisInfoMeta(final FicheUniv fiche, final Vector<String> vecteurHistorique) {
		final List<HistoriqueBean> historiques = new ArrayList<>();
		for (String valeur : vecteurHistorique) {
			final HistoriqueBean historique = new HistoriqueBean();
			valeur = StringUtils.substringBetween(valeur, "[", "]");
			final String[] valeursSplittes = valeur.split("/");
			historique.setIdFiche(fiche.getIdFiche());
			historique.setAction(Metatag.getIntituleAction(valeursSplittes[0]));
			try {
				historique.setDateAction(DateUtils.parseDate(valeursSplittes[1], "yyyyMMddhhmmss"));
			} catch (final ParseException e) {
				LOG.info("impossible de parser la date de l'historique de la fiche.", e);
			}
			historique.setUtilisateur(valeursSplittes[2]);
			historique.setEtat(EtatFiche.getEtatParCode(valeursSplittes[3]));
			historiques.add(historique);
		}
		return historiques;
	}

	private static FicheUniv retrouverFiche(final OMContext ctx, final String codeObjet, final Long idFiche) {
		FicheUniv ficheUniv = ReferentielObjets.instancierFiche(codeObjet);
		ficheUniv.setCtx(ctx);
		ficheUniv.init();
		try {
			/* Lecture de la fiche */
			ficheUniv.setIdFiche(idFiche);
			ficheUniv.retrieve();
		} catch (final Exception ex) {
			ficheUniv = null;
		}
		return ficheUniv;
	}

	/**
	 * Affichage de la liste des utilisateurs.
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static void preparerLISTE(final InfoBean infoBean, final OMContext ctx) throws Exception {
		// Historique ordonné
		// Stockage sous la forme
		// { "DATE", "ID_FICHE[ACTION/DATE/UTILISATEUR/ETAT]" }
		final TreeMap<Long, String> historique = new TreeMap<>();
		Enumeration<String> e = null;
		/* Création d'un objet pour faire les requêtes */
		final String objet = ReferentielObjets.getNomObjet(infoBean.getString("CODE_OBJET"));
		final FicheUniv ficheUniv = ReferentielObjets.instancierFiche(objet);
		ficheUniv.setCtx(ctx);
		ficheUniv.init();
		infoBean.set("AVERTISSEMENT", "0");
		boolean historiqueVide = false;
		try {
			/* Lecture de la fiche */
			ficheUniv.setIdFiche(new Long(infoBean.getString("ID_FICHE")));
			ficheUniv.retrieve();
		} catch (final Exception ex) {
			historiqueVide = true;
		}
		/* Lecture des métatags de la fiche courante */
		if (!historiqueVide) {
			Metatag meta = FicheUnivMgr.lireMeta(infoBean.getString("CODE_OBJET"), new Long(infoBean.getString("ID_FICHE")));
			e = meta.getVecteurHistorique().elements();
			if (meta.getVecteurHistorique().size() == 0) {
				historiqueVide = true;
			}
		}
		if (!historiqueVide) {
			/* Stockage des métatags de la fiche courante */
			while (e.hasMoreElements()) {
				final String action = e.nextElement();
				final StringTokenizer st = new StringTokenizer(action, "[/]");
				// on retient le 2ème token pour la date*/
				st.nextToken();
				final Long dateInversee = 999999999999L - new Long(st.nextToken());
				historique.put(dateInversee, ficheUniv.getIdFiche() + action);
			}
			/* Lecture et stockage des métatags des autres états */
			final FicheUniv ficheUniv2 = ReferentielObjets.instancierFiche(objet);
			ficheUniv2.setCtx(ctx);
			ficheUniv2.init();
			if (ficheUniv2.selectCodeLangueEtat(ficheUniv.getCode(), ficheUniv.getLangue(), "") > 0) {
				while (ficheUniv2.nextItem()) {
					Metatag meta = FicheUnivMgr.lireMeta(infoBean.getString("CODE_OBJET"), ficheUniv2.getIdFiche());
					e = meta.getVecteurHistorique().elements();
					while (e.hasMoreElements()) {
						final String action = e.nextElement();
						final StringTokenizer st = new StringTokenizer(action, "[/]");
						// on retient le 2ème token pour la date*/
						st.nextToken();
						final Long dateInversee = 999999999999L - new Long(st.nextToken());
						// On vérifie que l'action n'a pas déjà été répertoriée
						if (historique.get(dateInversee) == null) {
							historique.put(dateInversee, ficheUniv2.getIdFiche() + action);
						}
					}
				}
			}
			/*
			 * Parcours de l'historique trié et remplissage du composant
			 * d'information
			 */
			int i = 0;
			final Iterator<String> iter = historique.values().iterator();
			// On affiche les 20 + récents
			while (iter.hasNext() && i < 20) {
				final String action = iter.next();
				final StringTokenizer st = new StringTokenizer(action, "[/]");
				int indToken = 0;
				while (st.hasMoreTokens()) {
					final String item = st.nextToken();
					if (indToken == 0) {
						infoBean.set("ID_FICHE#" + i, item);
						// Affichage avertissement
						if (!item.equals(ficheUniv.getIdFiche().toString())) {
							infoBean.set("AVERTISSEMENT", "1");
						}
					}
					if (indToken == 1) {
						infoBean.set("ACTION#" + i, Metatag.getIntituleAction(item));
					}
					if (indToken == 2) {
						infoBean.set("DATE#" + i, item.substring(6, 8) + "/" + item.substring(4, 6) + "/" + item.substring(0, 4));
						infoBean.set("HEURE#" + i, item.substring(8, 10) + ":" + item.substring(10, 12));
					}
					if (indToken == 3) {
						if (item.equalsIgnoreCase("nobody")) {
							infoBean.set("UTILISATEUR#" + i, "Script de mise à jour");
						} else {
							infoBean.set("UTILISATEUR#" + i, Utilisateur.getLibelle(item));
						}
					}
					if (indToken == 4) {
						infoBean.set("ETAT#" + i, item);
					}
					indToken++;
				}
				i++;
			}
		}
	}
}
