/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.objetspartages.util;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kportal.cms.objetspartages.annotation.FicheAnnotationHelper;
import com.univ.objetspartages.bean.MetatagBean;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.DiffusionSelective;
import com.univ.objetspartages.om.EtatFiche;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.utils.ContexteDao;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ContexteUtil;
import com.univ.utils.FicheUnivHelper;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseOrderBy;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.OrderByHelper;

public class FichesAValiderUtil {

	private static DateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");

	private static final Logger LOG = LoggerFactory.getLogger(FichesAValiderUtil.class);

	private static String computeHash(Metatag meta, FicheUniv ficheUniv, String codeObjet) {
		String dateString = dateFormat.format(meta.getMetaDateCreation());
		if (meta.getMetaDateProposition().getTime() > 0) {
			dateString = dateFormat.format(meta.getMetaDateProposition());
		} else if (meta.getMetaDateModification().getTime() > 0) {
			dateString = dateFormat.format(meta.getMetaDateModification());
		}
		return String.format("%s-%s-%s-%s-%s", dateString, codeObjet, ficheUniv.getLibelleAffichable(), ficheUniv.getCode(), ficheUniv.getLangue());
	}

	public static int compterObjetsAValider() throws Exception {
		return getListeObjetsAValider().size();
	}

	public static Collection<FicheUniv> getListeObjetsAValiderRecents(final int limit) throws Exception {
		final SortedMap<String, FicheUniv> fichesAValider = getListeObjetsAValider();
		if (limit <= 0 || limit >= fichesAValider.size()) {
			return fichesAValider.values();
		}
		final String[] keys = new String[fichesAValider.keySet().size()];
		return fichesAValider.subMap(fichesAValider.firstKey(), fichesAValider.keySet().toArray(keys)[limit]).values();
	}

	public static SortedMap<String, FicheUniv> getListeObjetsAValider() throws Exception {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final AutorisationBean autorisations = ctx.getAutorisation();
		final SortedMap<String, FicheUniv> liste = new TreeMap<>();
		for (final String codeObjet : ReferentielObjets.getListeCodesObjet()) {
			if (autorisations.getAutorisation(codeObjet, AutorisationBean.INDICE_VALIDATION)) {
				final FicheUniv ficheUniv = ReferentielObjets.instancierFiche(codeObjet);
				if (ficheUniv != null && FicheAnnotationHelper.isAccessibleBo(ficheUniv)) {
					ficheUniv.setCtx(ctx);
					ficheUniv.init();
					ficheUniv.selectCodeLangueEtat("", "", "0002");
					while (ficheUniv.nextItem()) {
						boolean ficheEspace = false;
						if (ficheUniv instanceof DiffusionSelective) {
							if ("4".equals(((DiffusionSelective) ficheUniv).getDiffusionModeRestriction())) {
								ficheEspace = true;
							}
						}
						if (!ficheEspace) {
							String codeApprobation = "";
							final Metatag meta = FicheUnivMgr.lireMeta(ficheUniv);
							if (!"".equals(meta.getMetaNiveauApprobation())) {
								codeApprobation = meta.getMetaNiveauApprobation();
							}
							if (autorisations.getAutorisationParFiche(ficheUniv, AutorisationBean.INDICE_APPROBATION, codeApprobation)) {
								final String hash = computeHash(meta, ficheUniv, codeObjet);
								liste.put(hash, FicheUnivHelper.getFicheParIdMeta(meta.getIdMetatag()));
							}
						}
					}
				}
			}
		}
		return liste;
	}


	/**
	 * Methode calculant les fiches à valider possible de l'utilisateur courant.
	 * @param autorisations les autorisations de l'utilisateur courant.
	 * @return La liste des méta à valider qui ne sont pas du collab et qui sont autorisées pour l'utilisateur courant.
	 */
	public static List<MetatagBean> getObjetsAValider(final AutorisationBean autorisations) {
		final Collection<String> codeObjets = new ArrayList<>();
		final List<MetatagBean> result = new ArrayList<>();
		for (final String codeObjet : ReferentielObjets.getListeCodesObjet()) {
			if (autorisations.getAutorisation(codeObjet, AutorisationBean.INDICE_VALIDATION)) {
				codeObjets.add(codeObjet);
			}
		}
		final Metatag meta = new Metatag();
		try (ContexteDao ctx = new ContexteDao()) {
			meta.setCtx(ctx);
			final ClauseWhere whereMetaCodeObjetEtEtat = new ClauseWhere(ConditionHelper.egalVarchar("META_ETAT_OBJET", EtatFiche.A_VALIDER.toString()));
			if (!codeObjets.isEmpty()) {
				whereMetaCodeObjetEtEtat.and(ConditionHelper.in("META_CODE_OBJET", codeObjets));
			}
			ClauseOrderBy orderBy = new ClauseOrderBy("META_DATE_MODIFICATION", ClauseOrderBy.SensDeTri.DESC);
			final RequeteSQL requete = new RequeteSQL();
			requete.where(whereMetaCodeObjetEtEtat).orderBy(orderBy);
			meta.select(requete.formaterRequete());
			while (meta.nextItem()) {
				if (!"4".equals(meta.getMetaDiffusionModeRestriction())) {
					final String codeApprobation = StringUtils.defaultString(meta.getMetaNiveauApprobation());
					if (autorisations.getAutorisation(meta.getMetaCodeObjet(), meta.getMetaCode(), meta.getMetaCodeRubrique(), meta.getMetaCodeRattachement(), meta.getMetaCodeRattachementAutres(),
							meta.getMetaDiffusionPublicVise(), meta.getMetaDiffusionModeRestriction(), meta.getMetaDiffusionPublicViseRestriction(), AutorisationBean.INDICE_APPROBATION, codeApprobation)) {
						final MetatagBean currentMeta = new MetatagBean();
						currentMeta.init(meta);
						result.add(currentMeta);
					}
				}
			}
		} catch (Exception e) {
			LOG.error("Impossible de récupérer les fiches à valider");
		}
		return result;
	}

}
