/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.portail;

import java.io.IOException;
import java.util.HashMap;
import java.util.Properties;

import javax.portlet.PortletMode;
import javax.portlet.WindowState;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.pluto.PortletContainer;
import org.apache.pluto.PortletContainerException;
import org.apache.pluto.om.window.PortletWindow;
import org.apache.pluto.portalImpl.core.PortletContainerEnvironment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.kportal.cms.objetspartages.ObjetPartageHelper;
import com.kportal.extension.ExtensionHelper;
import com.kportal.servlet.ExtensionServlet;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.portail.core.FactoryAccess;
import com.univ.portail.core.PortalControlParameter;
import com.univ.portail.core.PortalEnvironment;
import com.univ.portail.core.PortletContainerFactory;
import com.univ.portail.core.SessionStatefullParameter;
import com.univ.portail.om.window.impl.PortletWindowImpl;
import com.univ.portail.service.ServiceManager;
import com.univ.portail.service.config.Config;
import com.univ.portail.service.factorymanager.FactoryManager;
import com.univ.portail.service.log.LogService;
import com.univ.portail.service.registry.PortletOccurenceRegistry;
import com.univ.portail.servlet.ServletObjectAccess;
import com.univ.portail.util.PortletHelper;
import com.univ.services.CacheServiceViewManager;
import com.univ.url.FrontOfficeMgr;
import com.univ.url.UrlManager;
import com.univ.utils.ContexteUniv;

// TODO: Auto-generated Javadoc
/**
 * The Class Servlet.
 */
@WebServlet(name = "servlet", urlPatterns = { "/portal/*", "/privileged/*", "/secure/*" }, loadOnStartup = 1)
public class Servlet extends ExtensionServlet {

	/**
	 *
	 */
	private static final long serialVersionUID = -3967601594865921662L;

	private static final Logger LOGGER = LoggerFactory.getLogger(Servlet.class);

	/** The Constant KPORTAL_CTX. */
	private static final String KPORTAL_CTX = "kportal.ctx";

	/** The Constant CHEMIN_JSP_PORTLET_MAX. */
	private static final String CHEMIN_JSP_PORTLET_MAX = "/WEB-INF/jsp/portal/fo/portlet_max.jsp";

	/** The Constant AFFICHAGE_UNITAIRE. */
	private static final String AFFICHAGE_UNITAIRE = "unitaire";

	/** The CONTEN t_ type. */
	private static String CONTENT_TYPE = "text/html";

	/** Stores whether the portlet container used supports buffering. */
	private static final String CONTAINER_SUPPORTS_BUFFERING = "portletcontainer.supportsBuffering";

	/* (non-Javadoc)
	 * @see javax.servlet.GenericServlet#getServletInfo()
	 */
	@Override
	public String getServletInfo() {
		return "portalImpl - Pluto Driver";
	}

	/* (non-Javadoc)
	 * @see javax.servlet.GenericServlet#init(javax.servlet.ServletConfig)
	 */
	@Override
	public void init(final ServletConfig config) throws ServletException {
		super.init(config);
		final String charset = config.getInitParameter("charset");
		if (charset != null && charset.length() > 0) {
			CONTENT_TYPE = "text/html; charset=" + charset;
		}
		try {
			ServiceManager.init(config);
		} catch (final Exception exc) {
			LOGGER.error("Initialization failed!", exc);
			throw (new javax.servlet.UnavailableException("Initialization of one or more services failed."));
		}
		try {
			ServiceManager.postInit(config);
		} catch (final Exception expos) {
			LOGGER.error("Post initialization failed!", expos);
			throw (new javax.servlet.UnavailableException("Post initialization of one or more services failed."));
		}
		try {
			PortletContainer pc = PortletContainerFactory.getPortletContainer();
			if (!pc.isInitialized()) {
				String uniqueContainerName = Config.getParameters().getString("portletcontainer.uniquename", "pluto");
				// JSS 20060208 : permet d'avoir un nom unique sur clusters ayant même config
				uniqueContainerName += "_" + System.currentTimeMillis();
				LOGGER.info("Initializing PortletContainer [" + uniqueContainerName + "]...");
				final PortletContainerEnvironment environment = new PortletContainerEnvironment();
				environment.addContainerService(LogService.getLogService());
				environment.addContainerService(FactoryManager.getService());
				environment.addContainerService(FactoryAccess.getInformationProviderContainerService());
				environment.addContainerService(FactoryAccess.getDynamicTitleContainerService());
				final String allowSetBufferSizeString = Config.getParameters().getString(CONTAINER_SUPPORTS_BUFFERING);
				boolean allowSetBufferSize = false;
				if (allowSetBufferSizeString == null) {
					LOGGER.warn("org.apache.pluto.portalImpl.Servlet#init(): " + "Couldn't read property \"pluto.allowSetBufferSize\" " + "from config file ConfigService.properties");
				} else {
					allowSetBufferSize = allowSetBufferSizeString.equalsIgnoreCase("yes");
				}
				final Properties containerProperties = new Properties();
				containerProperties.put(CONTAINER_SUPPORTS_BUFFERING, allowSetBufferSize);
				try {
					pc.init(uniqueContainerName, config, environment, containerProperties);
				} catch (final PortletContainerException exc) {
					LOGGER.error("Initialization of the portlet container failed!", exc);
					throw (new javax.servlet.UnavailableException("Initialization of the portlet container failed."));
				}
			} else if (LOGGER.isInfoEnabled()) {
				LOGGER.info("PortletContainer already initialized");
			}
			LOGGER.debug("Ready to serve you.");
		} catch (final Exception e) {
			LOGGER.warn("Initialization of the portlet container failed! file /WEB-INF/classes/portlet.properties is missing");
		}
	}

	/* (non-Javadoc)
	 * @see javax.servlet.GenericServlet#destroy()
	 */
	@Override
	public void destroy() {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Shutting down portlet container. . .");
		}
		try {
			PortletContainerFactory.getPortletContainer().shutdown();
			// destroy all services
			ServiceManager.destroy(getServletConfig());
		} catch (final PortletContainerException t) {
			LOGGER.error("Destruction failed!", t);
		}
	}

	/* (non-Javadoc)
	 * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	public void doGet(final HttpServletRequest servletRequest, final HttpServletResponse servletResponse) throws IOException, ServletException {
		servletResponse.setContentType(CONTENT_TYPE);
		final PortalEnvironment env = new PortalEnvironment(servletRequest, servletResponse);
		final PortalControlParameter control = env.getPortalControlParameter();
		final PortletWindow actionWindow = control.getPortletWindowOfAction();
		ContexteUniv ctx = null;
		final CacheServiceViewManager cacheServiceMgr = (CacheServiceViewManager) ApplicationContextManager.getCoreContextBean(CacheServiceViewManager.ID_BEAN);
		try {
			ctx = ContexteUniv.initContexteUniv(servletRequest, servletResponse, getServletContext(), getServletConfig());
			if (actionWindow != null) {
				try {
					final HttpServletRequest request = ServletObjectAccess.getServletRequest(servletRequest, actionWindow);
					request.setAttribute(KPORTAL_CTX, ctx);
					final HttpServletResponse response = ServletObjectAccess.getServletResponse(servletResponse);
					PortletContainerFactory.getPortletContainer().processPortletAction(actionWindow, request, response);
					// invalidation du cache sur action
					cacheServiceMgr.flushView(ctx.getKsession() + "@" + actionWindow.getPortletEntity().getId() + "@" + actionWindow.getId());
					LOGGER.debug("invalidation cache portlet : " + actionWindow.getPortletEntity().getId().toString());
				}
				// This catch block is for compliance
				// of TCK's Portlet.ProcessActionIOExceptionTest
				catch (final Exception e) {
					LOGGER.error(e.getMessage(), e);
				}
				return; // we issued an redirect, so return directly and return to the controller, eg this class
			}
			// invalidation du cache sur changement d'état ou de mode
			String idPortlet = control.getPIDValue();
			if (idPortlet != null && idPortlet.length() > 0) {
				final PortletWindow portletWindow = new PortletWindowImpl(idPortlet);
				final WindowState stateCourant = control.getState(portletWindow);
				final PortletMode modeCourant = control.getMode(portletWindow);
				final PortletMode modePrecedent = control.getPrevMode(portletWindow);
				if ((!stateCourant.equals(WindowState.NORMAL)) || (modePrecedent != null && !modePrecedent.equals(modeCourant))) {
					if (idPortlet.contains(KportalPortletFragment.OCCURENCE_SUFFIXE)) {
						idPortlet = idPortlet.substring(0, idPortlet.indexOf(KportalPortletFragment.OCCURENCE_SUFFIXE));
					}
					cacheServiceMgr.flushView(ctx.getKsession() + "@" + idPortlet + "@" + portletWindow.getId());
					LOGGER.debug("Invalidation cache portlet : " + idPortlet);
				}
			}
			final String typeVueService = servletRequest.getParameter("AFFICHAGE");
			String code = "";
			final boolean isMaximized = control.isOnePortletWindowMaximized();
			if (AFFICHAGE_UNITAIRE.equals(typeVueService)) {
				code = servletRequest.getParameter("CODE_SERVICE");
			} else if (isMaximized) {
				code = control.getIdPortletMaximized();
			}
			/* Sauvegarde des paramétres statefull du portlet dans la session */
			SessionStatefullParameter.saveIntoSession(servletRequest, control.getPIDValue(), control);
			/*
			 * si AFFICHAGE=service est passé en paramètre, cela signifie que l'on
			 * souhaite exécuter le service en mode maximized directement
			 * et sans les autres modes (normal et minimized)
			 * si la portlet est maximisée on redirige directement vers cette portlet
			 */
			if (isMaximized || (AFFICHAGE_UNITAIRE.equals(typeVueService) && code != null && !"".equals(code))) {
				servletRequest.setAttribute("fragment", PortletOccurenceRegistry.getOccurence(code));
				final String forward = ExtensionHelper.getTemplateExtension(ApplicationContextManager.DEFAULT_CORE_CONTEXT, CHEMIN_JSP_PORTLET_MAX, true);
				final RequestDispatcher rd = getServletConfig().getServletContext().getRequestDispatcher(forward);
				rd.forward(servletRequest, servletResponse);
			} else {
				// variable qui permet de récupérer le portlet actif (pour positionner l'ancre à l'affichage de la page libre)
				servletRequest.setAttribute("PID", control.getPIDValue());
				final PortletWindow portletWindow = new PortletWindowImpl(control.getPIDValue());
				final String urlPageLibre = UrlManager.calculerUrlPage(control.getKportalParamValues(portletWindow, "ID_META")[0]);
				if (control.getKportalParamValues(portletWindow, "RH") != null) {
					ctx.setCodeRubriquePageCourante(control.getKportalParamValues(portletWindow, "RH")[0]);
				}
				final FicheUniv ficheCourante = FrontOfficeMgr.lireFiche(ctx, urlPageLibre, new HashMap<String, String[]>());
				final String forward = ObjetPartageHelper.getTemplateObjet(ObjetPartageHelper.TEMPLATE_FICHE, ReferentielObjets.getNomObjet(ficheCourante));
				final RequestDispatcher rd = getServletConfig().getServletContext().getRequestDispatcher(forward);
				rd.forward(servletRequest, servletResponse);
			}
		} catch (final Exception e) {
			LOGGER.error(e.getMessage(), e);
		} finally {
			if (ctx != null) {
				ctx.release();
			}
		}
	}

	/* (non-Javadoc)
	 * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	public void doPost(final HttpServletRequest request, final HttpServletResponse response) throws IOException, ServletException {
		doGet(request, response);
	}

	@Override
	public boolean isActive() {
		return PortletHelper.isPortletActivated();
	}
}
