/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* 

 */
package com.univ.portail.core;

import java.util.HashSet;
import java.util.Iterator;

import javax.portlet.PortletMode;
import javax.portlet.WindowState;
import javax.servlet.ServletConfig;
import javax.servlet.http.HttpServletRequest;

import org.apache.pluto.om.window.PortletWindow;
import org.apache.pluto.services.information.DynamicInformationProvider;
import org.apache.pluto.services.information.PortletActionProvider;
import org.apache.pluto.services.information.PortletURLProvider;
import org.apache.pluto.services.information.ResourceURLProvider;

import com.univ.portail.service.config.Config;

// TODO: Auto-generated Javadoc
/**
 * The Class DynamicInformationProviderImpl.
 */
public class DynamicInformationProviderImpl implements DynamicInformationProvider {

	/** The Constant NumberOfKnownMimetypes. */
	private final static int NumberOfKnownMimetypes = 15;

	/** The config. */
	private ServletConfig config;

	/** The env. */
	private PortalEnvironment env;

	/** The request. */
	HttpServletRequest request;

	/**
	 * Instantiates a new dynamic information provider impl.
	 * 
	 * @param request
	 *            the request
	 * @param config
	 *            the config
	 */
	public DynamicInformationProviderImpl(HttpServletRequest request, ServletConfig config) {
		this.request = request;
		this.config = config;
		env = PortalEnvironment.getPortalEnvironment(request);
	}

	// DynamicInformationProviderImpl implementation.
	/**
	 * Gets the portlet mode.
	 * 
	 * @param portletWindow
	 *            the portlet window
	 * 
	 * @return the portlet mode
	 */
	@Override
	public PortletMode getPortletMode(PortletWindow portletWindow) {
		return env.getPortalControlParameter().getMode(portletWindow);
	}

	/**
	 * Gets the portlet url provider.
	 * 
	 * @param portletWindow
	 *            the portlet window
	 * 
	 * @return the portlet url provider
	 */
	@Override
	public PortletURLProvider getPortletURLProvider(PortletWindow portletWindow) {
		return new PortletURLProviderImpl(this, portletWindow);
	}

	/**
	 * Gets the resource url provider.
	 * 
	 * @param portletWindow
	 *            the portlet window
	 * 
	 * @return the resource url provider
	 */
	@Override
	public ResourceURLProvider getResourceURLProvider(PortletWindow portletWindow) {
		return new ResourceURLProviderImpl(this, portletWindow);
	}

	/**
	 * Gets the portlet action provider.
	 * 
	 * @param portletWindow
	 *            the portlet window
	 * 
	 * @return the portlet action provider
	 */
	@Override
	public PortletActionProvider getPortletActionProvider(PortletWindow portletWindow) {
		return new PortletActionProviderImpl(request, config, portletWindow);
	}

	/**
	 * Gets the previous portlet mode.
	 * 
	 * @param portletWindow
	 *            the portlet window
	 * 
	 * @return the previous portlet mode
	 */
	public PortletMode getPreviousPortletMode(PortletWindow portletWindow) {
		return env.getPortalControlParameter().getPrevMode(portletWindow);
	}

	/**
	 * Gets the previous window state.
	 * 
	 * @param portletWindow
	 *            the portlet window
	 * 
	 * @return the previous window state
	 */
	public WindowState getPreviousWindowState(PortletWindow portletWindow) {
		return env.getPortalControlParameter().getPrevState(portletWindow);
	}

	/**
	 * Gets the response content type.
	 * 
	 * @return the response content type
	 */
	@Override
	public String getResponseContentType() {
		return "text/html";
	}

	/**
	 * Gets the response content types.
	 * 
	 * @return the response content types
	 */
	@Override
	public Iterator<String> getResponseContentTypes() {
		HashSet<String> responseMimeTypes = new HashSet<String>(NumberOfKnownMimetypes);
		responseMimeTypes.add("text/html");
		return responseMimeTypes.iterator();
	}

	/**
	 * Gets the window state.
	 * 
	 * @param portletWindow
	 *            the portlet window
	 * 
	 * @return the window state
	 */
	@Override
	public WindowState getWindowState(PortletWindow portletWindow) {
		return env.getPortalControlParameter().getState(portletWindow);
	}

	/**
	 * Checks if is portlet mode allowed.
	 * 
	 * @param mode
	 *            the mode
	 * 
	 * @return true, if is portlet mode allowed
	 */
	@Override
	public boolean isPortletModeAllowed(PortletMode mode) {
		// checks whether PortletMode is supported as example
		String[] supportedModes = Config.getParameters().getStrings("supported.portletmode");
		for (String supportedMode : supportedModes) {
			if (supportedMode.equalsIgnoreCase(mode.toString())) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Checks if is window state allowed.
	 * 
	 * @param state
	 *            the state
	 * 
	 * @return true, if is window state allowed
	 */
	@Override
	public boolean isWindowStateAllowed(WindowState state) {
		// checks whether WindowState is supported as example
		String[] supportedStates = Config.getParameters().getStrings("supported.windowstate");
		for (String supportedState : supportedStates) {
			if (supportedState.equalsIgnoreCase(state.toString())) {
				return true;
			}
		}
		return false;
	}
}
