/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.portail.core;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;

import javax.portlet.PortletMode;
import javax.portlet.WindowState;

import org.apache.pluto.om.window.PortletWindow;
import org.apache.pluto.util.StringUtils;

import com.univ.portail.KportalPortletFragment;
import com.univ.portail.service.registry.PortletOccurenceRegistry;

// TODO: Auto-generated Javadoc
/**
 * The Class PortalControlParameter.
 */
public class PortalControlParameter {

	/** The Constant ACTION. */
	static public final String ACTION = "ac";

	/** The Constant MODE. */
	static public final String MODE = "md";

	/** The Constant PORTLET_ID. */
	static public final String PORTLET_ID = "pid";

	/** The Constant PREFIX. */
	static public final String PREFIX = "_";

	/** The Constant PREV_MODE. */
	static public final String PREV_MODE = "pm";

	/** The Constant PREV_STATE. */
	static public final String PREV_STATE = "ps";

	/** The Constant RENDER_PARAM. */
	static public final String RENDER_PARAM = "rp";

	/** The Constant STATE. */
	static public final String STATE = "st";

	/** The Constant KPORTAL_PARAM. */
	static public final String KPORTAL_PARAM = "kp";

	/**
	 * Decode parameter name.
	 * 
	 * @param paramName
	 *            the param name
	 * 
	 * @return the string
	 */
	public static String decodeParameterName(final String paramName) {
		return paramName.substring(PREFIX.length());
	}

	/**
	 * Decode parameter value.
	 * 
	 * @param paramName
	 *            the param name
	 * @param paramValue
	 *            the param value
	 * 
	 * @return the string
	 */
	public static String decodeParameterValue(final String paramName, final String paramValue) {
		return paramValue;
	}

	/**
	 * Decode render param values.
	 * 
	 * @param encodedParamValues
	 *            the encoded param values
	 * 
	 * @return the string[]
	 */
	private static String[] decodeRenderParamValues(final String encodedParamValues) {
		final StringTokenizer tokenizer = new StringTokenizer(encodedParamValues, "_");
		if (!tokenizer.hasMoreTokens()) {
			return null;
		}
		final String _count = tokenizer.nextToken();
		final int count = Integer.valueOf(_count).intValue();
		final String[] values = new String[count];
		for (int i = 0; i < count; i++) {
			if (!tokenizer.hasMoreTokens()) {
				return null;
			}
			values[i] = decodeValue(tokenizer.nextToken());
		}
		return values;
	}

	/**
	 * Decode value.
	 * 
	 * @param value
	 *            the value
	 * 
	 * @return the string
	 */
	private static String decodeValue(String value) {
		value = StringUtils.replace(value, "0x1", "_");
		value = StringUtils.replace(value, "0x2", ".");
		value = StringUtils.replace(value, "0x3", "/");
		value = StringUtils.replace(value, "0x4", "\r");
		value = StringUtils.replace(value, "0x5", "\n");
		value = StringUtils.replace(value, "0x6", "<");
		value = StringUtils.replace(value, "0x7", ">");
		value = StringUtils.replace(value, "0x8", " ");
		value = StringUtils.replace(value, "0x9", "?");
		return value;
	}

	/**
	 * Encode parameter.
	 * 
	 * @param param
	 *            the param
	 * 
	 * @return the string
	 */
	public static String encodeParameter(final String param) {
		return PREFIX + param;
	}

	/**
	 * Encode render param name.
	 * 
	 * @param window
	 *            the window
	 * @param paramName
	 *            the param name
	 * 
	 * @return the string
	 */
	public static String encodeRenderParamName(final PortletWindow window, final String paramName) {
		final StringBuffer returnvalue = new StringBuffer(50);
		returnvalue.append(RENDER_PARAM);
		returnvalue.append("_");
		returnvalue.append(window.getId().toString());
		returnvalue.append("_");
		returnvalue.append(paramName);
		return returnvalue.toString();
	}

	/**
	 * Encode kportal param name.
	 * 
	 * @param window
	 *            the window
	 * @param paramName
	 *            the param name
	 * 
	 * @return the string
	 */
	public static String encodeKportalParamName(final PortletWindow window, final String paramName) {
		final StringBuffer returnvalue = new StringBuffer(50);
		returnvalue.append(KPORTAL_PARAM);
		returnvalue.append("_");
		returnvalue.append(window.getId().toString());
		returnvalue.append("_");
		returnvalue.append(paramName);
		return returnvalue.toString();
	}

	/**
	 * Encode render param values.
	 * 
	 * @param paramValues
	 *            the param values
	 * 
	 * @return the string
	 */
	public static String encodeRenderParamValues(final String[] paramValues) {
		final StringBuffer returnvalue = new StringBuffer(100);
		returnvalue.append(paramValues.length);
		for (final String paramValue : paramValues) {
			returnvalue.append("_");
			if (paramValue != null) {
				returnvalue.append(encodeValue(paramValue));
			}
		}
		return returnvalue.toString();
	}

	/**
	 * Encode value.
	 * 
	 * @param value
	 *            the value
	 * 
	 * @return the string
	 */
	private static String encodeValue(String value) {
		value = StringUtils.replace(value, "_", "0x1");
		value = StringUtils.replace(value, ".", "0x2");
		value = StringUtils.replace(value, "/", "0x3");
		value = StringUtils.replace(value, "\r", "0x4");
		value = StringUtils.replace(value, "\n", "0x5");
		value = StringUtils.replace(value, "<", "0x6");
		value = StringUtils.replace(value, ">", "0x7");
		value = StringUtils.replace(value, " ", "0x8");
		value = StringUtils.replace(value, "?", "0x9");
		return value;
	}

	/**
	 * Gets the render param key.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the render param key
	 */
	public static String getRenderParamKey(final PortletWindow window) {
		return RENDER_PARAM + "_" + window.getId().toString();
	}

	/**
	 * Gets the kportal param key.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the kportal param key
	 */
	public static String getKportalParamKey(final PortletWindow window) {
		return KPORTAL_PARAM + "_" + window.getId().toString();
	}

	/**
	 * Checks if is control parameter.
	 * 
	 * @param param
	 *            the param
	 * 
	 * @return true, if is control parameter
	 */
	public static boolean isControlParameter(final String param) {
		return param.startsWith(PREFIX);
	}

	/**
	 * Checks if is state full parameter.
	 * 
	 * @param param
	 *            the param
	 * 
	 * @return true, if is state full parameter
	 */
	public static boolean isStateFullParameter(final String param) {
		if (isControlParameter(param)) {
			if ((param.startsWith(PREFIX + MODE)) || (param.startsWith(PREFIX + PREV_MODE)) || (param.startsWith(PREFIX + STATE)) || (param.startsWith(PREFIX + PREV_STATE)) || (param.startsWith(PREFIX + RENDER_PARAM)) || (param.startsWith(PREFIX + KPORTAL_PARAM))) {
				return true;
			}
		}
		return false;
	}

	/** The request parameter. */
	private final Map<String, String[]> requestParameter = new HashMap<String, String[]>();

	/** The state full control parameter. */
	private Map<String, Object> stateFullControlParameter = null;

	/** The state less control parameter. */
	private Map<String, Object> stateLessControlParameter = null;

	/** The url. */
	private final PortalURL url;

	/**
	 * Instantiates a new portal control parameter.
	 * 
	 * @param url
	 *            the url
	 */
	public PortalControlParameter(final PortalURL url) {
		this.url = url;
		stateFullControlParameter = this.url.getClonedStateFullControlParameter();
		stateLessControlParameter = this.url.getClonedStateLessControlParameter();
		/**
		 * On stocke ici les paramètres que l'on souhaite conserver comme le code de l'objet KPortal, ou le code de l'objet service et le type d'affichage désiré du service
		 */
		if (url.getUrlParameter("AFFICHAGE") != null && url.getUrlParameter("CODE_SERVICE") != null) {
			requestParameter.put("AFFICHAGE", new String[] { url.getUrlParameter("AFFICHAGE") });
			requestParameter.put("CODE_SERVICE", new String[] { url.getUrlParameter("CODE_SERVICE") });
		}
	}

	/**
	 * Clear render parameters.
	 * 
	 * @param portletWindow
	 *            the portlet window
	 */
	public void clearRenderParameters(final PortletWindow portletWindow) {
		final String prefix = getRenderParamKey(portletWindow);
		final Iterator<String> keyIterator = stateFullControlParameter.keySet().iterator();
		while (keyIterator.hasNext()) {
			final String name = keyIterator.next();
			if (name.startsWith(prefix)) {
				keyIterator.remove();
			}
		}
	}

	/**
	 * Gets the action key.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the action key
	 */
	private String getActionKey(final PortletWindow window) {
		return ACTION + "_" + window.getId().toString();
	}

	/**
	 * Gets the action parameter.
	 * 
	 * @param window
	 *            the window
	 * @param paramName
	 *            the param name
	 * 
	 * @return the action parameter
	 */
	public String[] getActionParameter(final PortletWindow window, final String paramName) {
		final String encodedValues = (String) stateFullControlParameter.get(encodeRenderParamName(window, paramName));
		final String[] values = decodeRenderParamValues(encodedValues);
		return values;
	}

	/**
	 * Gets the mode.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the mode
	 */
	public PortletMode getMode(final PortletWindow window) {
		final String mode = (String) stateFullControlParameter.get(getModeKey(window));
		PortletMode modeRetour = PortletMode.VIEW;
		if (mode != null) {
			modeRetour = new PortletMode(mode);
		}
		return modeRetour;
	}

	/**
	 * Gets the mode key.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the mode key
	 */
	private String getModeKey(final PortletWindow window) {
		return MODE + "_" + window.getId().toString();
	}

	/**
	 * Gets the pID value.
	 * 
	 * @return the pID value
	 */
	public String getPIDValue() {
		final String value = (String) stateLessControlParameter.get(getPortletIdKey());
		return value == null ? "" : value;
	}

	/**
	 * Gets the portlet id key.
	 * 
	 * @return the portlet id key
	 */
	private String getPortletIdKey() {
		return PORTLET_ID;
	}

	/**
	 * Gets the portlet window of action.
	 * 
	 * @return the portlet window of action
	 */
	public PortletWindow getPortletWindowOfAction() {
		final Iterator<String> iterator = getStateLessControlParameter().keySet().iterator();
		while (iterator.hasNext()) {
			final String name = iterator.next();
			if (name.startsWith(ACTION)) {
				final String id = name.substring(ACTION.length() + 1);
				final KportalPortletFragment fragment = PortletOccurenceRegistry.getOccurence(id);
				PortletWindow window = null;
				if (fragment != null) {
					window = fragment.getPortletWindow();
				}
				return window;
			}
		}
		return null;
	}

	/**
	 * Gets the prev mode.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the prev mode
	 */
	public PortletMode getPrevMode(final PortletWindow window) {
		final String mode = (String) stateFullControlParameter.get(getPrevModeKey(window));
		if (mode != null) {
			return new PortletMode(mode);
		} else {
			return null;
		}
	}

	/**
	 * Gets the prev mode key.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the prev mode key
	 */
	private String getPrevModeKey(final PortletWindow window) {
		return PREV_MODE + "_" + window.getId().toString();
	}

	/**
	 * Gets the prev state.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the prev state
	 */
	public WindowState getPrevState(final PortletWindow window) {
		final String state = (String) stateFullControlParameter.get(getPrevStateKey(window));
		if (state != null) {
			return new WindowState(state);
		} else {
			return null;
		}
	}

	/**
	 * Gets the prev state key.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the prev state key
	 */
	private String getPrevStateKey(final PortletWindow window) {
		return PREV_STATE + "_" + window.getId().toString();
	}

	/**
	 * Gets the render param names.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the render param names
	 */
	public Iterator<String> getRenderParamNames(final PortletWindow window) {
		//        if( window.getId().toString().indexOf("TCK") != -1)	{
		//            Log.debug("getRenderParamNames() - testId = " + stateFullControlParameter.get("rp_TCK_occ_1_testId"));
		//        }
		//        
		//        
		//        Log.debug("getRenderParamNames() - stateFullControlParameter = " + stateFullControlParameter);
		final ArrayList<String> returnvalue = new ArrayList<String>();
		final String prefix = getRenderParamKey(window);
		final Iterator<String> keyIterator = stateFullControlParameter.keySet().iterator();
		while (keyIterator.hasNext()) {
			final String name = keyIterator.next();
			if (name.startsWith(prefix)) {
				returnvalue.add(name.substring(prefix.length() + 1));
			}
		}
		return returnvalue.iterator();
	}

	/**
	 * Gets the kportal param values.
	 * 
	 * @param window
	 *            the window
	 * @param paramName
	 *            the param name
	 * 
	 * @return the kportal param values
	 */
	public String[] getKportalParamValues(final PortletWindow window, final String paramName) {
		final String encodedValues = (String) stateFullControlParameter.get(encodeKportalParamName(window, paramName));
		if (encodedValues == null) {
			return null;
		}
		final String[] values = decodeRenderParamValues(encodedValues);
		return values;
	}

	/**
	 * Gets the kportal param names.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the kportal param names
	 */
	public Iterator<String> getKportalParamNames(final PortletWindow window) {
		final ArrayList<String> returnvalue = new ArrayList<String>();
		final String prefix = getKportalParamKey(window);
		final Iterator<String> keyIterator = stateFullControlParameter.keySet().iterator();
		while (keyIterator.hasNext()) {
			final String name = keyIterator.next();
			if (name.startsWith(prefix)) {
				returnvalue.add(name.substring(prefix.length() + 1));
			}
		}
		return returnvalue.iterator();
	}

	/**
	 * Gets the render param values.
	 * 
	 * @param window
	 *            the window
	 * @param paramName
	 *            the param name
	 * 
	 * @return the render param values
	 */
	public String[] getRenderParamValues(final PortletWindow window, final String paramName) {
		final String encodedValues = (String) stateFullControlParameter.get(encodeRenderParamName(window, paramName));
		final String[] values = decodeRenderParamValues(encodedValues);
		return values;
	}

	/**
	 * Gets the request parameter.
	 * 
	 * @return the request parameter
	 */
	public Map<String, String[]> getRequestParameter() {
		return requestParameter;
	}

	/**
	 * Gets the state.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the state
	 */
	public WindowState getState(final PortletWindow window) {
		final String state = (String) stateFullControlParameter.get(getStateKey(window));
		if (state != null) {
			return new WindowState(state);
		} else {
			return WindowState.NORMAL;
		}
	}

	/**
	 * Gets the state full control parameter.
	 * 
	 * @return the state full control parameter
	 */
	public Map<String, Object> getStateFullControlParameter() {
		return stateFullControlParameter;
	}

	/**
	 * Gets the state key.
	 * 
	 * @param window
	 *            the window
	 * 
	 * @return the state key
	 */
	private String getStateKey(final PortletWindow window) {
		return STATE + "_" + window.getId().toString();
	}

	/**
	 * Gets the state less control parameter.
	 * 
	 * @return the state less control parameter
	 */
	public Map<String, Object> getStateLessControlParameter() {
		return stateLessControlParameter;
	}

	/**
	 * Checks if is one portlet window maximized.
	 * 
	 * @return true, if is one portlet window maximized
	 */
	public boolean isOnePortletWindowMaximized() {
		final Iterator<String> iterator = stateFullControlParameter.keySet().iterator();
		while (iterator.hasNext()) {
			final String name = iterator.next();
			if (name.startsWith(STATE)) {
				if (stateFullControlParameter.get(name).equals(WindowState.MAXIMIZED.toString())) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * Sets the action.
	 * 
	 * @param window
	 *            the new action
	 */
	public void setAction(final PortletWindow window) {
		getStateFullControlParameter().put(getActionKey(window), ACTION.toUpperCase());
	}

	/**
	 * Sets the mode.
	 * 
	 * @param window
	 *            the window
	 * @param mode
	 *            the mode
	 */
	public void setMode(final PortletWindow window, final PortletMode mode) {
		final Object prevMode = stateFullControlParameter.get(getModeKey(window));
		if (prevMode != null) {
			stateFullControlParameter.put(getPrevModeKey(window), prevMode);
		}
		// set current mode
		stateFullControlParameter.put(getModeKey(window), mode.toString());
	}

	/**
	 * Sets the portlet id.
	 * 
	 * @param window
	 *            the new portlet id
	 */
	public void setPortletId(final PortletWindow window) {
		getStateFullControlParameter().put(getPortletIdKey(), window.getId().toString());
		//getStateLessControlParameter().put(getPortletIdKey(),window.getId().toString());
	}

	/**
	 * Sets the render param.
	 * 
	 * @param window
	 *            the window
	 * @param name
	 *            the name
	 * @param values
	 *            the values
	 */
	public void setRenderParam(final PortletWindow window, final String name, final String[] values) {
		stateFullControlParameter.put(encodeRenderParamName(window, name), encodeRenderParamValues(values));
	}

	/**
	 * Sets the kportal param.
	 * 
	 * @param window
	 *            the window
	 * @param name
	 *            the name
	 * @param values
	 *            the values
	 */
	public void setKportalParam(final PortletWindow window, final String name, final String[] values) {
		stateFullControlParameter.put(encodeKportalParamName(window, name), encodeRenderParamValues(values));
	}

	/*
	    public void setRequestParam(String name, String value)
	    {
	        requestParameter.put(name, value );
	    }
	*/
	/**
	 * Sets the request param.
	 * 
	 * @param name
	 *            the name
	 * @param values
	 *            the values
	 */
	public void setRequestParam(final String name, final String[] values) {
		requestParameter.put(name, values);
	}

	/**
	 * Sets the state.
	 * 
	 * @param window
	 *            the window
	 * @param state
	 *            the state
	 */
	public void setState(final PortletWindow window, final WindowState state) {
		final Object prevState = stateFullControlParameter.get(getStateKey(window));
		if (prevState != null) {
			stateFullControlParameter.put(getPrevStateKey(window), prevState);
		}
		stateFullControlParameter.put(getStateKey(window), state.toString());
	}

	/**
	 * Méthode permettant de retourner l'identifiant de la portlet maximisée.
	 * 
	 * @return identifiant de la portlet maximisée
	 */
	public String getIdPortletMaximized() {
		String code = "";
		final Iterator<String> iterator = this.getStateFullControlParameter().keySet().iterator();
		while (iterator.hasNext()) {
			final String name = iterator.next();
			if (name.startsWith(STATE)) {
				if (this.getStateFullControlParameter().get(name).equals(WindowState.MAXIMIZED.toString())) {
					code = name.substring((STATE + PREFIX).length());
				}
			}
		}
		return code;
	}
}
