/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.portail.core;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import javax.servlet.http.HttpServletRequest;

import org.apache.pluto.om.window.PortletWindow;

import com.univ.portail.KportalPortletFragment;
import com.univ.portail.service.config.Config;

// TODO: Auto-generated Javadoc
/**
 * The Class PortalURL.
 */
public class PortalURL {

	/** The Constant insecureServlet. */
	private static final String insecureServlet;

	/** The Constant secureServlet. */
	private static final String secureServlet;
	static {
		insecureServlet = Config.getParameters().getString("servlet.insecure");
		secureServlet = Config.getParameters().getString("servlet.secure");
	}

	/**
	 * Creates and URL pointing to the home of the portal.
	 * 
	 * @param request
	 *            the servlet request
	 * 
	 * @return the portal URL
	 */
	public String getBasePortalURL(final HttpServletRequest request) {
		return getBasePortalURL(PortalEnvironment.getPortalEnvironment(request));
	}

	/**
	 * Creates and URL pointing to the home of the portal.
	 * 
	 * @param env
	 *            the portal environment
	 * 
	 * @return the portal URL
	 */
	public String getBasePortalURL(final PortalEnvironment env) {
		final StringBuffer result = new StringBuffer(256);
		/*
		        result.append(secure?hostNameHTTPS:hostNameHTTP);
		*/
		result.append(env.getRequest().getContextPath());
		result.append(env.getRequest().getServletPath());
		return result.toString();
	}

	/** The start global navigation. */
	private List<String> startGlobalNavigation = new ArrayList<String>();

	/** The start local navigation. */
	private List<String> startLocalNavigation = new ArrayList<String>();

	/** The start state full control parameter. */
	private HashMap<String, Object> startStateFullControlParameter = new HashMap<String, Object>();

	/** The start state less control parameter. */
	private HashMap<String, Object> startStateLessControlParameter = new HashMap<String, Object>();

	/** The analyzed. */
	private boolean analyzed = false;

	/** The environment. */
	private PortalEnvironment environment;

	/**
	 * Creates and URL pointing to the home of the portal.
	 * 
	 * @param env
	 *            the portal environment
	 */
	public PortalURL(final PortalEnvironment env) {
		environment = env;
	}

	/**
	 * Creates and URL pointing to the home of the portal.
	 * 
	 * @param env
	 *            the portal environment
	 */
	public void setEnv(final PortalEnvironment env) {
		environment = env;
	}

	/**
	 * Permet de retourner un chaine passée dans l'url via la forme http://wwww/portal...?cle=valeur
	 * 
	 * @param cle
	 *            clé indentifiant la valeur à retourner
	 * 
	 * @return valeur trouvée sous forme d'une chaine de caractères
	 */
	public String getUrlParameter(final String cle) {
		return environment.getRequest().getParameter(cle);
	}

	/**
	 * Creates and URL pointing to the home of the portal.
	 * 
	 * @param request
	 *            the servlet request
	 */
	public PortalURL(final HttpServletRequest request) {
		this(PortalEnvironment.getPortalEnvironment(request));
	}

	/**
	 * Creates and URL pointing to the given fragment of the portal.
	 * 
	 * @param request
	 *            the servlet request
	 * @param pointTo
	 *            the fragment to point to
	 */
	public PortalURL(final HttpServletRequest request, final KportalPortletFragment pointTo) {
		this(request);
		pointTo.createURL(this);
	}

	/**
	 * Creates and URL pointing to the given fragment of the portal.
	 * 
	 * @param env
	 *            the portal environment
	 * @param pointTo
	 *            the fragment to point to
	 */
	public PortalURL(final PortalEnvironment env, final KportalPortletFragment pointTo) {
		this(env);
		pointTo.createURL(this);
	}

	/**
	 * Adds a navigational information pointing to a portal part, e.g. PageGroups or Pages
	 * 
	 * @param nav
	 *            the string pointing to a portal part
	 */
	public void addGlobalNavigation(final String nav) {
		startGlobalNavigation.add(nav);
	}

	/**
	 * Sets the local navigation. Because the local navigation is always handled by the Browser, therefore the local navigation cleared.
	 */
	public void setLocalNavigation() {
		startLocalNavigation = new ArrayList<String>();
	}

	/**
	 * Adds a navigational information pointing to a local portal part inside of a global portal part, for example, a portlet on a page.
	 * 
	 * @param nav
	 *            the string pointing to a local portal part
	 */
	public void addLocalNavigation(final String nav) {
		startLocalNavigation.add(nav);
	}

	/**
	 * Returns true if the given string is part of the global navigation of this URL.
	 * 
	 * @param nav
	 *            the string to check
	 * 
	 * @return true, if the string is part of the navigation
	 */
	public boolean isPartOfGlobalNavigation(final String nav) {
		return startGlobalNavigation.contains(nav);
	}

	/**
	 * Returns true if the given string is part of the local navigation of this URL.
	 * 
	 * @param nav
	 *            the string to check
	 * 
	 * @return true, if the string is part of the navigation
	 */
	public boolean isPartOfLocalNavigation(final String nav) {
		return startLocalNavigation.contains(nav);
	}

	/**
	 * Gets the global navigation as string.
	 * 
	 * @return the global navigation as string
	 */
	public String getGlobalNavigationAsString() {
		final StringBuffer result = new StringBuffer(200);
		final Iterator<String> iterator = startGlobalNavigation.iterator();
		if (iterator.hasNext()) {
			result.append(iterator.next());
			while (iterator.hasNext()) {
				result.append("/");
				final String st = iterator.next();
				result.append(st);
			}
		}
		return result.toString();
	}

	/**
	 * Gets the local navigation as string.
	 * 
	 * @return the local navigation as string
	 */
	public String getLocalNavigationAsString() {
		final StringBuffer result = new StringBuffer(30);
		final Iterator<String> iterator = startLocalNavigation.iterator();
		if (iterator.hasNext()) {
			result.append(iterator.next());
			while (iterator.hasNext()) {
				result.append(".");
				result.append(iterator.next());
			}
		}
		return result.toString();
	}

	/**
	 * Gets the control parameter as string.
	 * 
	 * @param controlParam
	 *            the control param
	 * 
	 * @return the control parameter as string
	 */
	public String getControlParameterAsString(final PortalControlParameter controlParam) {
		Map<String, Object> stateFullParams = startStateFullControlParameter;
		if (controlParam != null) {
			stateFullParams = controlParam.getStateFullControlParameter();
		}
		final StringBuffer result = new StringBuffer(100);
		final Iterator<String> iterator = stateFullParams.keySet().iterator();
		while (iterator.hasNext()) {
			if (iterator.hasNext()) {
				result.append("/");
			}
			final String name = iterator.next();
			final String value = (String) stateFullParams.get(name);
			if (value != null) {
				result.append(PortalControlParameter.encodeParameter(name));
				result.append("/");
				result.append(value);
			}
		}
		return result.toString();
	}

	/**
	 * Gets the request parameter as string.
	 * 
	 * @param controlParam
	 *            the control param
	 * 
	 * @return the request parameter as string
	 */
	public String getRequestParameterAsString(final PortalControlParameter controlParam) {
		if (controlParam != null) {
			final Map<String, String[]> requestParams = controlParam.getRequestParameter();
			final StringBuffer result = new StringBuffer(100);
			final Iterator<String> iterator = requestParams.keySet().iterator();
			boolean hasNext = iterator.hasNext();
			if (hasNext) {
				result.append("?");
			}
			while (hasNext) {
				final String name = iterator.next();
				final Object value = requestParams.get(name);
				final String[] values = value instanceof String ? new String[] { (String) value } : (String[]) value;
				result.append(name);
				result.append("=");
				result.append(values[0]);
				for (int i = 1; i < values.length; i++) {
					result.append("&");
					result.append(name);
					result.append("=");
					result.append(values[i]);
				}
				hasNext = iterator.hasNext();
				if (hasNext) {
					result.append("&");
				}
			}
			//            String id = (String) controlParam.getStateFullControlParameter().get(PortalControlParameter.PORTLET_ID); 
			//            if (id != null && !"".equals(id)) 
			//                result.append("#" + id);
			return result.toString();
		}
		return "";
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return toString(null, null);
	}

	/**
	 * To string.
	 * 
	 * @param controlParam
	 *            the control param
	 * @param p_secure
	 *            the p_secure
	 * 
	 * @return the string
	 */
	public String toString(final PortalControlParameter controlParam, final Boolean p_secure) {
		final StringBuffer urlBase = new StringBuffer(256);
		boolean secure = false;
		if (p_secure != null) {
			secure = p_secure.booleanValue();
		} else {
			secure = environment.getRequest().isSecure();
		}
		urlBase.append(environment.getRequest().getContextPath());
		urlBase.append(secure ? secureServlet : insecureServlet);
		String url = urlBase.toString();
		final String global = getGlobalNavigationAsString();
		if (global.length() > 0) {
			url += "/";
			url += global;
		}
		String control = getControlParameterAsString(controlParam);
		if (control.length() > 0) {
			// parameter string should be x-www-form-urlencoded here! See jira 119
			// parameter string will be decoded by the tomcat container
			// control = URLEncoder.encode(control);
			// java.net.URLEncoder encodes space (' ') as a plus sign ('+'),
			// instead of %20 thus it will not be decoded properly by tomcat when the
			// request is parsed. Therefore replace all '+' by '%20'.
			// If there would have been any plus signs in the original string, they would
			// have been encoded by URLEncoder.encode()
			// control = control.replace("+", "%20");//only works with JDK 1.5
			control = control.replaceAll("\\+", "%20");
			url += control;
		}
		final String requestParam = getRequestParameterAsString(controlParam);
		if (requestParam.length() > 0) {
			url += requestParam;
		}
		final String local = getLocalNavigationAsString();
		if (local.length() > 0) {
			url += "#";
			url += local;
		}
		//return environment.getResponse().encodeURL(url);
		//Log.debug( "ToString() : "+url);
		return url;
	}

	/**
	 * Gets the cloned state full control parameter.
	 * 
	 * @return the cloned state full control parameter
	 */
	Map<String, Object> getClonedStateFullControlParameter() {
		analyzeRequestInformation();
		return (Map<String, Object>) startStateFullControlParameter.clone();
	}

	/**
	 * Gets the cloned state less control parameter.
	 * 
	 * @return the cloned state less control parameter
	 */
	Map<String, Object> getClonedStateLessControlParameter() {
		analyzeRequestInformation();
		return (Map<String, Object>) startStateLessControlParameter.clone();
	}

	/**
	 * Analyze control information.
	 * 
	 * @param control
	 *            the control
	 */
	void analyzeControlInformation(final PortalControlParameter control) {
		startStateFullControlParameter = (HashMap<String, Object>) control.getStateFullControlParameter();
		startStateLessControlParameter = (HashMap<String, Object>) control.getStateLessControlParameter();
	}

	/**
	 * Analyze request information.
	 */
	void analyzeRequestInformation() {
		if (analyzed) {
			return;
		}
		startGlobalNavigation = new ArrayList<String>();
		startLocalNavigation = new ArrayList<String>();
		startStateFullControlParameter = new HashMap<String, Object>();
		startStateLessControlParameter = new HashMap<String, Object>();
		// check the complete pathInfo for
		// * navigational information
		// * control information
		if (environment.getRequest().getPathInfo() != null) {
			final String pathInfo = environment.getRequest().getPathInfo();
			final StringTokenizer tokenizer = new StringTokenizer(pathInfo, "/");
			int mode = 0; // 0=navigation, 1=control information
			String name = null;
			while (tokenizer.hasMoreTokens()) {
				final String token = tokenizer.nextToken();
				if (PortalControlParameter.isControlParameter(token)) {
					mode = 1;
					name = token;
				} else if (mode == 0) {
					startGlobalNavigation.add(token);
				} else if (mode == 1) {
					if ((PortalControlParameter.isStateFullParameter(name))) {
						startStateFullControlParameter.put(PortalControlParameter.decodeParameterName(name), PortalControlParameter.decodeParameterValue(name, token));
					} else {
						startStateLessControlParameter.put(PortalControlParameter.decodeParameterName(name), PortalControlParameter.decodeParameterValue(name, token));
					}
					mode = 0;
				}
			}
		}
		//      String id = (String) this.getStateFullControlParameter().get(PortalControlParameter.PORTLET_ID); 
		//      if (id != null && !"".equals(id)) 
		/* Si infos présentes dans la session, on les prend */
		if (sessionStateFullsParameters != null) {
			startStateFullControlParameter.clear();
			startStateFullControlParameter.putAll(sessionStateFullsParameters);
		}
		//Log.debug("analyzeRequestInformation startStateFullControlParameter = "+ startStateFullControlParameter);
		analyzed = true;
	}

	/**
	 * Sets the render parameter.
	 * 
	 * @param portletWindow
	 *            the portlet window
	 * @param name
	 *            the name
	 * @param values
	 *            the values
	 */
	public void setRenderParameter(final PortletWindow portletWindow, final String name, final String[] values) {
		startStateFullControlParameter.put(PortalControlParameter.encodeRenderParamName(portletWindow, name), PortalControlParameter.encodeRenderParamValues(values));
	}

	/**
	 * Clear render parameters.
	 * 
	 * @param portletWindow
	 *            the portlet window
	 */
	public void clearRenderParameters(final PortletWindow portletWindow) {
		final String prefix = PortalControlParameter.getRenderParamKey(portletWindow);
		final Iterator<String> keyIterator = startStateFullControlParameter.keySet().iterator();
		while (keyIterator.hasNext()) {
			final String name = keyIterator.next();
			if (name.startsWith(prefix)) {
				keyIterator.remove();
			}
		}
	}

	/** The session state fulls parameters. */
	private Map<String, ?> sessionStateFullsParameters = null;

	/**
	 * Sets the session statefull parameter.
	 * 
	 * @param _sessionStatefullParam
	 *            the new session statefull parameter
	 */
	public void setSessionStatefullParameter(final Map<String, ?> _sessionStatefullParam) {
		sessionStateFullsParameters = _sessionStatefullParam;
	}
}
