/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*

 */
package com.univ.portail.core.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import javax.portlet.PortletPreferences;
import javax.portlet.PreferencesValidator;
import javax.portlet.ReadOnlyException;
import javax.portlet.ValidatorException;

import org.apache.pluto.Constants;
import org.apache.pluto.om.ControllerObjectAccess;
import org.apache.pluto.om.common.Preference;
import org.apache.pluto.om.common.PreferenceCtrl;
import org.apache.pluto.om.common.PreferenceSet;
import org.apache.pluto.om.common.PreferenceSetCtrl;
import org.apache.pluto.om.entity.PortletEntity;
import org.apache.pluto.om.portlet.PortletDefinition;
import org.apache.pluto.util.Enumerator;
import org.apache.pluto.util.StringUtils;

import com.univ.portail.om.user.UserPreference;
import com.univ.portail.util.PreferenceSetImpl;

// TODO: Auto-generated Javadoc
/**
 * The Class PortletPreferencesImpl.
 *
 * @author EBO
 */
public class PortletPreferencesImpl implements PortletPreferences {

	/** The preference set list. */
	private final List<PreferenceSet> preferenceSetList = new LinkedList<PreferenceSet>();

	// private List preferenceSetList = new MaLinkList();
	/** The changed preferences. */
	private final HashMap<String, String[]> changedPreferences = new HashMap<String, String[]>();

	/** The removed preferences. */
	private final HashSet<String> removedPreferences = new HashSet<String>();

	// current method used for managing these preferences
	/** The method id. */
	private Integer methodId = null;

	/** The portlet entity. */
	private PortletEntity portletEntity = null;

	/** The portlet definition. */
	private PortletDefinition portletDefinition = null;

	/** The user preference. */
	private UserPreference userPreference = null;

	/**
	 * Instantiates a new portlet preferences impl.
	 *
	 * @param methodId
	 *            the method id
	 * @param portletEntity
	 *            the portlet entity
	 */
	public PortletPreferencesImpl(final Integer methodId, final PortletEntity portletEntity) {
		this.methodId = methodId;
		this.portletEntity = portletEntity;
		this.portletDefinition = portletEntity.getPortletDefinition();
		// fill list of preference sets
		//ajout des préférences utilisateur
		preferenceSetList.add(new PreferenceSetImpl());
		preferenceSetList.add(portletEntity.getPreferenceSet());
		preferenceSetList.add(portletDefinition.getPreferenceSet());
	}

	/**
	 * Instantiates a new portlet preferences impl.
	 *
	 * @param methodId
	 *            the method id
	 * @param portletDefinition
	 *            the portlet definition
	 */
	public PortletPreferencesImpl(final Integer methodId, final PortletDefinition portletDefinition) {
		this.methodId = methodId;
		this.portletDefinition = portletDefinition;
		//ajout des préférences utilisateur
		preferenceSetList.add(new PreferenceSetImpl());
		// fill list of preference sets
		preferenceSetList.add(portletDefinition.getPreferenceSet());
	}

	// javax.portlet.PortletPreferences implementation
	// --------------------------------------------
	/**
	 * Checks if is read only.
	 *
	 * @param key
	 *            the key
	 *
	 * @return true, if is read only
	 */
	@Override
	public boolean isReadOnly(final String key) {
		if (key == null) {
			throw new IllegalArgumentException("key == null");
		}
		// default is false
		boolean isReadOnly = false;
		// if managing the first layer of the preferences
		// modifiable returns always true (for administration purposes)
		if (preferenceSetList.size() != 2) { // otherwise
			// iterate through all preferences
			Preference preference = null;
			final ListIterator<PreferenceSet> iter = preferenceSetList.listIterator();
			while ((preference == null) && (iter.hasNext())) {
				preference = iter.next().get(key);
			}
			if (preference != null) {
				isReadOnly = preference.isReadOnly();
			}
		}
		return isReadOnly;
	}

	/**
	 * Gets the value.
	 *
	 * @param key
	 *            the key
	 * @param def
	 *            the def
	 *
	 * @return the value
	 */
	@Override
	public String getValue(final String key, final String def) {
		if (key == null) {
			throw new IllegalArgumentException("key == null");
		}
		final String[] defStr = new String[1];
		defStr[0] = def;
		final String[] values = this.getValues(key, defStr);
		// null values are allowed
		if ((values == null) || (values.length == 0)) {
			return null;
		}
		return values[0];
	}

	/**
	 * Gets the values.
	 *
	 * @param key
	 *            the key
	 * @param def
	 *            the def
	 *
	 * @return the values
	 */
	@Override
	public String[] getValues(final String key, final String[] def) {
		if (key == null) {
			throw new IllegalArgumentException("key == null");
		}
		// get modified preferences
		if (changedPreferences.containsKey(key)) {
			return StringUtils.copy(changedPreferences.get(key));
		}
		// get all preference sets
		final ListIterator<PreferenceSet> iter = preferenceSetList.listIterator();
		// if removed preference do not look in first layer and User layer TODO
		if (removedPreferences.contains(key)) {
			iter.next();// on ne prend pas en compte les préférences issues des userPreferences..
		}
		// iterate through all preference sets
		Preference preference = null;
		while ((preference == null) && (iter.hasNext())) {
			preference = iter.next().get(key);
		}
		if (preference == null || !preference.isValueSet()) {
			// if preference not exists
			return def;
		} else { // if preference exists
			String[] result = this.getValuesFromPreference(preference);
			if (result != null) {
				result = StringUtils.copy(result);
			}
			return result;
		}
	}

	/**
	 * Sets the value.
	 *
	 * @param key
	 *            the key
	 * @param value
	 *            the value
	 *
	 * @throws ReadOnlyException
	 *             the read only exception
	 */
	@Override
	public void setValue(final String key, final String value) throws ReadOnlyException {
		if (key == null) {
			throw new IllegalArgumentException("key == null");
		}
		final String[] values = new String[1];
		values[0] = value;
		setValues(key, values);
	}

	/**
	 * Sets the values.
	 *
	 * @param key
	 *            the key
	 * @param values
	 *            the values
	 *
	 * @throws ReadOnlyException
	 *             the read only exception
	 */
	@Override
	public void setValues(final String key, final String[] values) throws ReadOnlyException {
		if (key == null) {
			throw new IllegalArgumentException("key == null");
		}
		if (isReadOnly(key)) {
			throw new ReadOnlyException("Preference attribute called " + key + " may not be modified");
		}
		changedPreferences.put(key, StringUtils.copy(values));
		removedPreferences.remove(key);
	}

	/**
	 * Gets the names.
	 *
	 * @return the names
	 */
	@Override
	public Enumeration<String> getNames() {
		final HashSet<String> keyset = new HashSet<String>();
		final ListIterator<PreferenceSet> listIter = preferenceSetList.listIterator();
		final Iterator<String> changedIter = changedPreferences.keySet().iterator();
		final Iterator<String> removedIter = removedPreferences.iterator();
		// iterate through all modified preferences of first layer
		while (changedIter.hasNext()) {
			keyset.add(changedIter.next());
		}
		// iterate through all existing preferences of first layer
		Iterator<Preference> preferencesIter = listIter.next().iterator();
		while (preferencesIter.hasNext()) {
			final String name = preferencesIter.next().getName();
			keyset.add(name);
		}
		// iterate through all removed preferences of first layer
		while (removedIter.hasNext()) {
			keyset.remove(removedIter.next());
		}
		// iterate through all other preference sets
		while (listIter.hasNext()) {
			preferencesIter = listIter.next().iterator();
			// iterate through all preferences
			while (preferencesIter.hasNext()) {
				final String name = preferencesIter.next().getName();
				keyset.add(name);
			}
		}
		return new Enumerator(keyset.iterator());
	}

	/**
	 * Gets the map.
	 *
	 * @return the map
	 */
	@Override
	public Map<String, String[]> getMap() {
		final Map<String, String[]> map = new HashMap<String, String[]>();
		final Enumeration<String> enumerator = this.getNames();
		while (enumerator.hasMoreElements()) {
			final String name = enumerator.nextElement();
			map.put(name, getValues(name, null));
		}
		return map;
	}

	/**
	 * Reset.
	 *
	 * @param key
	 *            the key
	 *
	 * @throws ReadOnlyException
	 *             the read only exception
	 */
	@Override
	public void reset(final String key) throws ReadOnlyException {
		if (key == null) {
			throw new IllegalArgumentException("key == null");
		}
		if (isReadOnly(key)) {
			throw new ReadOnlyException("preference attribute called " + key + " may not be modified");
		}
		changedPreferences.remove(key);
		removedPreferences.add(key);
	}

	/**
	 * Sets the user preferences.
	 *
	 * @param userPref
	 *            the new user preferences
	 */
	public void setUserPreferences(final UserPreference userPref) {
		this.userPreference = userPref;
	}

	/**
	 * Store.
	 *
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 * @throws ValidatorException
	 *             the validator exception
	 */
	@Override
	public void store() throws java.io.IOException, ValidatorException {
		// not allowed when not called in action
		if (!this.methodId.equals(Constants.METHOD_ACTION)) {
			throw new java.lang.IllegalStateException("store is only allowed inside a processAction call");
		}
		if (userPreference != null) {
			// validate preferences
			final PreferencesValidator validator = portletDefinition.getPreferenceSet().getPreferencesValidator();
			if (validator != null) {
				validator.validate(this);
			}
			// transfer changes to the top preference set
			final PreferenceSet preferences = preferenceSetList.get(0);
			final PreferenceSetCtrl preferencesCtrl = (PreferenceSetCtrl) ControllerObjectAccess.get(preferences);
			// modified preferences
			Iterator<String> iter = changedPreferences.keySet().iterator();
			while (iter.hasNext()) {
				final String key = iter.next();
				final String[] values = changedPreferences.get(key);
				// null values are allowed
				List<String> newValues = null;
				if (values != null) {
					// convert values from string[] to collection
					newValues = new ArrayList<String>(values.length);
					for (final String value : values) {
						newValues.add(value);
					}
				}
				// transfer changings
				final Preference preference = preferences.get(key);
				if (preference != null) {
					// change preference
					final PreferenceCtrl preferenceCtrl = (PreferenceCtrl) ControllerObjectAccess.get(preference);
					preferenceCtrl.setValues(newValues);
				} else {
					// add new preference
					preferencesCtrl.add(key, newValues);
				}
			}
			changedPreferences.clear();
			// removed preferences
			iter = removedPreferences.iterator();
			while (iter.hasNext()) {
				final String key = iter.next();
				preferencesCtrl.remove(key);
			}
			removedPreferences.clear();
			userPreference.store(preferences, portletEntity.getId().toString());
			//        // store changes to the top preference set
			//        if (portletEntity != null) {
			//            PortletEntityCtrl portletEntityCtrl = (PortletEntityCtrl) ControllerObjectAccess.get(portletEntity);
			//            portletEntityCtrl.store();
			//        } else {
			//            PortletDefinitionCtrl portletDefinitionCtrl = (PortletDefinitionCtrl) ControllerObjectAccess
			//                    .get(portletDefinition);
			//            portletDefinitionCtrl.store();
			//        }
		}
	}

	/**
	 * Méthode permettant d'attribuer les préférences utilisateur à la création.
	 *
	 * @param key
	 *            nom de la préférence
	 * @param values
	 *            liste des valeurs associées
	 */
	public void setUserValues(final String key, final List values) {
		final PreferenceSetImpl userPrefSet = (PreferenceSetImpl) preferenceSetList.get(0);
		userPrefSet.addUserPreferencesValues(key, values);
	}

	// --------------------------------------------------------------------------------------------
	// internal methods
	// ---------------------------------------------------------------------------
	/**
	 * Gets the values from preference.
	 *
	 * @param preference
	 *            the preference
	 *
	 * @return the values from preference
	 */
	private String[] getValuesFromPreference(final Preference preference) {
		if (preference == null) {
			return null;
		}
		final Iterator<String> values = preference.getValues();
		// null values are allowed
		if (values == null) {
			return null;
		}
		if (!values.hasNext()) {
			return new String[0];
		}
		// convert values from Iterator to string[]
		final List<String> newValues = new ArrayList<>();
		while (values.hasNext()) {
			newValues.add(values.next());
		}
		return newValues.toArray(new String[newValues.size()]);
	}
	// --------------------------------------------------------------------------------------------
}
