/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.portail.deploiment;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.PortletDiskFileUpload;
import org.apache.commons.fileupload.PortletFileUploadBase;
import org.apache.pluto.descriptors.common.SecurityRoleRefDD;
import org.apache.pluto.descriptors.portlet.PortletDD;
import org.apache.pluto.portlet.admin.BaseAdminObject;
import org.apache.pluto.portlet.admin.PlutoAdminConstants;
import org.apache.pluto.portlet.admin.PlutoAdminException;
import org.apache.pluto.portlet.admin.bean.PortletMessage;
import org.apache.pluto.portlet.admin.bean.PortletMessageType;
import org.apache.pluto.portlet.admin.services.PortletConfigService;
import org.apache.pluto.portlet.admin.services.PortletNameFinder;

import com.jsbsoft.jtf.exception.ErreurTechnique;
import com.univ.portail.deploiment.util.PlutoAdminContext;
import com.univ.portail.service.registry.PortletEntityRegistry;

// TODO: Auto-generated Javadoc
/**
 * This is the service that does is called by DeployWarPortlet to do the work of deploying a portlet war.
 *
 * @author Ken Atherton
 * @author Craig Doremus
 */
public class DeployWarService extends BaseAdminObject {

	/** The Constant ERROR_NO_FILE. */
	public static final String ERROR_NO_FILE = "ERROR_NO_FILE";

	/** The Constant CLASS_NAME. */
	public static final String CLASS_NAME = "DeployWarService";

	/**
	 * Default constructor.
	 */
	public DeployWarService() {
		super(CLASS_NAME);
	}

	/**
	 * Constructor taking a String used to identify a logging record.
	 *
	 * @param logId
	 *            the log id
	 */
	public DeployWarService(final String logId) {
		super(CLASS_NAME, logId);
	}

	/**
	 * Does the work of this service to deploy a portlet war file.
	 *
	 * @param request
	 *            DeployWarService request object.
	 * @param response
	 *            DeployWarService response object.
	 *
	 * @return the string
	 */
	public String processFileUpload(final ActionRequest request, final ActionResponse response) {
		final String METHOD_NAME = "processFileUpload(request,response)";
		String fileName = null;
		String serverFileName = null;
		boolean modifyWebXml = true;
		request.getPortletSession().setAttribute(PlutoAdminConstants.MESSAGE_ATTR, new PortletMessage("Deployment unsuccessful", PortletMessageType.ERROR));
		// Check the request content type to see if it starts with multipart/
		if (PortletFileUploadBase.isMultipartContent(request)) {
			final PortletDiskFileUpload dfu = new PortletDiskFileUpload();
			//maximum allowed file upload size (10 MB)
			dfu.setSizeMax(10 * 1000 * 1000);
			//maximum size in memory (vs disk) (100 KB)
			dfu.setSizeThreshold(100 * 1000);
			try {
				//get the FileItems
				@SuppressWarnings("unchecked")
				final List<FileItem> fileItems = dfu.parseRequest(request);
				final Iterator<FileItem> iter = fileItems.iterator();
				while (iter.hasNext()) {
					final FileItem item = iter.next();
					if (item.isFormField()) {
						//pass along to render request
						final String fieldName = item.getFieldName();
						final String value = item.getString();
						response.setRenderParameter(fieldName, value);
						if (fieldName.equalsIgnoreCase("NoWebXmlModification")) {
							final String noWebXmlModification = item.getString();
							logWarn(METHOD_NAME, "Don't modify web.xml? " + noWebXmlModification);
							if (noWebXmlModification != null) {
								modifyWebXml = false;
							}
						}
					} else {
						//write the uploaded file to a new location
						fileName = item.getName();
						final String contentType = item.getContentType();
						final long size = item.getSize();
						response.setRenderParameter("size", Long.toString(size));
						response.setRenderParameter("contentType", contentType);
						final String tempDir = System.getProperty("java.io.tmpdir");
						serverFileName = getRootFilename(File.separatorChar, fileName);
						final File serverFile = new File(tempDir, serverFileName);
						item.write(serverFile);
						response.setRenderParameter("serverFileName", serverFileName);
						//Add to portletentityregistry.xml
						final int index = serverFileName.indexOf(".war");
						String context = "";
						if (index != -1) {
							context = serverFileName.substring(0, index);
						} else {
							context = serverFileName;
						}
						//Check to see if a record exists
						final boolean appExists = (PortletEntityRegistry.getPortletEntity(context) != null);
						final ArrayList<String> argList = createDeploymentArgs(serverFileName, tempDir, request, appExists, context);
						logDebug(METHOD_NAME, "Arguments for Deploy.main():");
						final String[] args = arrayListToStringArray(argList);
						for (int i = 0; i < args.length; i++) {
							logDebug(METHOD_NAME, "args[" + i + "]=" + args[i]);
						}
						Deploy.main(args);
						//NEW: Update web.xml with new servlet elements
						if (modifyWebXml) {
							updateWebXml(context);
						}
						if (appExists) {
							request.getPortletSession().setAttribute(
								PlutoAdminConstants.MESSAGE_ATTR,
								new PortletMessage(
									"Deployment of the new portlet app has been successful, but the portlet app record '" + context + "' already exists in portletentityregistry.xml. " + "This may have occurred if the portlet was previously partially deployed. If that is the case, continue with this screen and the next to register the portlet in pageregistry.xml. " + "If you are deploying a previously deployed portlet app, click on the 'Deploy War home' link below and then the 'Hot deploy . . .' link on the resulting page to see your redeployed portlet. " + "However, caching of the old app may require that you restart Pluto to see the new changes.",
									PortletMessageType.INFO));
						} else {
							request.getPortletSession().setAttribute(PlutoAdminConstants.MESSAGE_ATTR,
								new PortletMessage("Deployment and addition to portletentityregistry.xml successful.", PortletMessageType.SUCCESS));
						}
					}
				}
			} catch (final FileUploadException e) {
				final String msg = "File Upload Exception: " + e.getMessage();
				logError(METHOD_NAME, msg, e);
				throw new PlutoAdminException(e);
			} catch (final Exception e) {
				final String msg = "Exception: " + e.getMessage();
				logError(METHOD_NAME, msg, e);
				throw new PlutoAdminException(e);
			}
		} else {
			//set an error message
			request.getPortletSession().setAttribute(PlutoAdminConstants.MESSAGE_ATTR, new PortletMessage("No file appears to have been selected.", PortletMessageType.ERROR));
		}
		logMethodEnd(METHOD_NAME, serverFileName);
		return serverFileName;
	}

	/**
	 * Gets the root filename.
	 *
	 * @param delimiter
	 *            the delimiter
	 * @param pathName
	 *            the path name
	 *
	 * @return the root filename
	 */
	private String getRootFilename(final char delimiter, final String pathName) {
		final int startFilenameIndex = pathName.lastIndexOf(delimiter) + 1;
		final String filename = pathName.substring(startFilenameIndex);
		return filename;
	}

	/**
	 * Array list to string array.
	 *
	 * @param argStringArrayList
	 *            the arg string array list
	 *
	 * @return the string[]
	 */
	private static String[] arrayListToStringArray(final ArrayList<String> argStringArrayList) {
		return argStringArrayList.toArray(new String[argStringArrayList.size()]);
	}

	/**
	 * Extract file.
	 *
	 * @param zipfilename
	 *            the zipfilename
	 * @param filename
	 *            the filename
	 *
	 * @return the input stream
	 */
	private InputStream extractFile(final String zipfilename, final String filename) {
		final String METHOD_NAME = "extractFile(zipfilename,filename)";
		InputStream ins = null;
		try (final ZipFile zf = new ZipFile(zipfilename)) {
			if (null != zf) {
				final ZipEntry ze = zf.getEntry(filename);
				if (null != ze) {
					ins = zf.getInputStream(ze);
				}
			}
		} catch (final Exception e) {
			logError(CLASS_NAME, METHOD_NAME, e);
			throw new PlutoAdminException(e);
		}
		return ins;
	}

	/**
	 * Creates arguments (parameters) for Deploy class that does the deployment.
	 *
	 * @param serverFileName
	 *            The name of the war file to be deployed
	 * @param tempDir
	 *            Full path to temp dir that holds the war file to be deployed
	 * @param request
	 *            ActionRequest of the portlet.
	 * @param appExists
	 *            True if this is a re-deployment, else false
	 * @param context
	 *            the context
	 *
	 * @return ArrayList of arguments
	 *
	 * @throws Exception
	 *             the exception
	 *
	 * @see com.univ.portal.Deploy#main
	 */
	private ArrayList<String> createDeploymentArgs(final String serverFileName, final String tempDir, final ActionRequest request, final boolean appExists, final String context)
		throws Exception {
		final String METHOD_NAME = "createDeploymentArgs(serverFileName,tempDir,request)";
		final Properties props = PlutoAdminContext.getProperties();
		//	    final String CONTAINER_HOME =  PlutoAdminContext.getContainerHome();
		final String PORTLET_DEPLOY_DIR = props.getProperty("portlet-deploy-dir");
		final ArrayList<String> args = new ArrayList<String>();
		args.add(PlutoAdminContext.getDeploymentPath());
		args.add(PlutoAdminContext.getPlutoWebContext());
		args.add(tempDir + PlutoAdminConstants.FS + serverFileName);
		//This is probably not used???, but left here to as to not change
		//	args indexing used by Deploy class.
		//	    args.add(CONTAINER_HOME + PlutoAdminConstants.FS + PORTLET_DEPLOY_DIR);
		args.add(PORTLET_DEPLOY_DIR);
		String appId = String.valueOf(Math.random());
		//check if a record in portletentityregistry exists
		if (!appExists) {
			args.add("-addToEntityReg");
			args.add(appId);
		} else {
			//Don't add it to portletentityregistry.xml,
			//and retreive id using webapp context and XAO
			//			PortletEntityRegistryXao xao = new PortletEntityRegistryXao();
			//			PortletApplicationEntityImpl app = xao.getApplication(context);
			//		    appId = app.getCastorId();
			appId = PortletEntityRegistry.getPortletEntity(context).getId().toString();
		}
		//Add Map of portlet name/values to session
		// to be used in drop downs on page layout page
		final Map<String, String> pmap = new HashMap<String, String>();
		final InputStream ins = extractFile(tempDir + PlutoAdminConstants.FS + serverFileName, "WEB-INF/portlet.xml");
		if (null != ins) {
			@SuppressWarnings("unchecked")
			final ArrayList<String> names = PortletNameFinder.getPortletNames(ins);
			for (int i = 0; i < names.size(); i++) {
				//check if a record in portletentityregistry exists
				if (!appExists) {
					args.add(i + ":" + names.get(i));
				}
				pmap.put(names.get(i), appId + "." + i);
			}
			ins.close();
		} else {
			final String msg = "Input stream is null";
			final PlutoAdminException e = new PlutoAdminException(msg);
			logError(METHOD_NAME, e);
			throw e;
		}
		request.getPortletSession().setAttribute(PlutoAdminConstants.PORTLET_MAP_ATTR, pmap);
		return args;
	}

	/**
	 * Puts the contents of a file into a String. This only works with text files.
	 *
	 * @param file
	 *            The File to read
	 *
	 * @return A String containing the contents of the file.
	 */
	private String readFileToString(final File file) {
		final String METHOD_NAME = "readFileToString(path)";
		String contents = null;
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(file);
			int c;
			char b;
			final StringBuffer sb = new StringBuffer();
			while ((c = fis.read()) != -1) {
				b = (char) c;
				sb.append(b);
			}
			contents = sb.toString().trim();
		} catch (final FileNotFoundException e) {
			logError(METHOD_NAME, e);
			throw new PlutoAdminException(e);
		} catch (final IOException e) {
			logError(METHOD_NAME, e);
			throw new PlutoAdminException(e);
		} finally {
			if (fis != null) {
				try {
					fis.close();
				} catch (final IOException e) {
					logError(METHOD_NAME, e);
					throw new PlutoAdminException(e);
				}
			}
		}
		return contents;
	}

	/**
	 * Writes the contents of a file into a String.
	 *
	 * @param file
	 *            Te File to write.
	 * @param contents
	 *            The String to add to the file.
	 */
	private void writeStringToFile(final File file, final String contents) {
		final String METHOD_NAME = "addFileToStringToFile(contents)";
		FileOutputStream fos = null;
		try {
			fos = new FileOutputStream(file);
			final byte[] bytes = contents.getBytes();
			fos.write(bytes);
		} catch (final FileNotFoundException e) {
			logError(METHOD_NAME, e);
			throw new PlutoAdminException(e);
		} catch (final IOException e) {
			logError(METHOD_NAME, e);
			throw new PlutoAdminException(e);
		} finally {
			if (fos != null) {
				try {
					fos.close();
				} catch (final IOException e) {
					logError(METHOD_NAME, e);
					throw new PlutoAdminException(e);
				}
			}
		}
	}

	/**
	 * Updates web.xml with servlet and servlet-mapping records related to PortletServlet.
	 *
	 * @param context
	 *            the context
	 *
	 * @throws ErreurTechnique
	 *             the erreur technique
	 */
	@SuppressWarnings("unchecked")
	public void updateWebXml(final String context) throws ErreurTechnique {
		final String METHOD_NAME = "updateWebXml(context)";
		//These constants are used to place the
		//new record in web.xml in the proper place
		//elements to check prior to servlet (if not found)
		final String[] PRIOR_ELEMENTS_SERVLET = { "servlet", "listener", "filter-mapping", "filter", "context-param", "distributable", "description", "display-name", "icon", "web-app" };
		//elements to check prior to servlet-mapping  (if not found)
		final String[] PRIOR_ELEMENTS_SERVLET_MAPPING = { "servlet-mapping", "servlet" };
		final String webapps = PlutoAdminContext.getDeploymentPath();
		final File webXml = new File(webapps + PlutoAdminConstants.FS + context + PlutoAdminConstants.FS + "WEB-INF" + PlutoAdminConstants.FS + "web.xml");
		final DeployWarService svc = new DeployWarService();
		String contents = svc.readFileToString(webXml);
		final File portletXml = new File(webapps + PlutoAdminConstants.FS + context + PlutoAdminConstants.FS + "WEB-INF" + PlutoAdminConstants.FS + "portlet.xml");
		List<PortletDD> plist = null;
		try {
			final InputStream ins = new FileInputStream(portletXml);
			final PortletConfigService pcsvc = new PortletConfigService(ins);
			plist = pcsvc.getPortletDDList();
		} catch (final FileNotFoundException e) {
			logError(METHOD_NAME, e);
			throw new PlutoAdminException(e);
		}
		String newWebXml = svc.addRecordsToWebXml(context, contents, PRIOR_ELEMENTS_SERVLET, plist);
		contents = newWebXml;
		newWebXml = svc.addRecordsToWebXml(context, contents, PRIOR_ELEMENTS_SERVLET_MAPPING, plist);
		writeStringToFile(webXml, newWebXml);
	}

	/**
	 * Adds ServletPortlet servlet or servlet-mapping records to web.xml for new portlet deployment.
	 *
	 * @param context
	 *            Web context or directory under webapps
	 * @param contents
	 *            Current contents of the web.xml file
	 * @param elements
	 *            Elements in web.xml to search for. If found, new elements will be inserted to the contents String. NOTE: First element (elements[0] signals the kind of record to
	 *            add (servlet or servlet-mapping).
	 * @param portletData
	 *            A List of PortletDD items containing the data of portlets to be deployed.
	 *
	 * @return the string
	 */
	String addRecordsToWebXml(final String context, final String contents, final String[] elements, final List<PortletDD> portletData) {
		final String METHOD_NAME = "addRecordsToWebXml(context.contents,elements,portletData)";
		StringBuffer results = new StringBuffer(contents);
		int index = -1;//index indicating the start of the insertion point
		final int len = portletData.size();
		final int lenElements = elements.length;
		//string before a found element
		String before = null;
		//The new record to be added
		String newRecord = null;
		//The remainder of the string after the found element (tag)
		String remainder = null;
		//Rest of the string after the opening of the element
		String rest = null;
		//go through the list of portlets in ArrayLists
		for (int i = 0; i < len; i++) {
			//check each element in web.xml contents
			for (int j = 0; j < lenElements; j++) {
				if ((index = results.lastIndexOf("</" + elements[j] + ">")) != -1) {
					//get the length to the end of the element (>)
					rest = results.substring(index);
					int elementLen = rest.indexOf('>') + 1;
					//First portlet could have to deal with web-app element
					//so the new records will be put after the web-app start element.
					//Also account for web-app's attributes in calculating the element length
					if (i == 0 & elements[j].equals("web-app")) {
						index = results.indexOf("<web-app");
						//Get the rest of the results String after <web-app
						rest = results.substring(index);//get string starting with <web-app
						elementLen = rest.indexOf('>') + 1;//end of starting web-app element
					}
					logDebug(METHOD_NAME, "Length of '" + elements[j] + "' tag = " + elementLen);
					//get everything before the found element including the tag
					before = results.substring(0, index + elementLen);
					//get everything after the found element starting at the end of the tag
					remainder = results.substring(index + elementLen);
					//add the new element with child elements
					//create the new element using the first item in the elements array
					if (elements[0].equals("servlet")) {
						newRecord = getServletRecord(context, portletData.get(i));
					} else if (elements[0].equals("servlet-mapping")) {
						newRecord = getServletMappingRecord(portletData.get(i));
					}
					results = new StringBuffer();
					results.append(before);
					//start the new content with a newline
					results.append(PlutoAdminConstants.LS);
					results.append(newRecord);
					results.append(remainder);
					break;
				}
			}
		}
		return results.toString();
	}

	/**
	 * Gets the web.xml servlet record for PortletServlet from portlet.xml data
	 *
	 * @param context
	 *            Context name
	 * @param portletData
	 *            Data from portlet.xml
	 *
	 * @return the servlet record
	 */
	private String getServletRecord(final String context, final PortletDD portletData) {
		final StringBuffer record = new StringBuffer();
		record.append("    <servlet>" + PlutoAdminConstants.LS);
		record.append("      <servlet-name>" + portletData.getPortletName() + "</servlet-name>" + PlutoAdminConstants.LS);
		record.append("      <display-name>" + portletData.getPortletName() + " Wrapper</display-name>" + PlutoAdminConstants.LS);
		record.append("      <description>Automated generated Portlet Wrapper</description>" + PlutoAdminConstants.LS);
		record.append("      <servlet-class>org.apache.pluto.core.PortletServlet</servlet-class>" + PlutoAdminConstants.LS);
		record.append("      <init-param>" + PlutoAdminConstants.LS);
		record.append("        <param-name>portlet-guid</param-name>" + PlutoAdminConstants.LS);
		record.append("        <param-value>" + context + "." + portletData.getPortletName() + "</param-value>" + PlutoAdminConstants.LS);
		record.append("      </init-param>" + PlutoAdminConstants.LS);
		record.append("      <init-param>" + PlutoAdminConstants.LS);
		record.append("        <param-name>portlet-class</param-name>" + PlutoAdminConstants.LS);
		record.append("        <param-value>" + portletData.getPortletClass() + "</param-value>" + PlutoAdminConstants.LS);
		record.append("      </init-param>" + PlutoAdminConstants.LS);
		//Add list of security-role-ref elements with
		// corresponding role-name and role-link if there is one in portlet.xml
		final String securityRef = getSecurityRoleRefRecord(context, portletData);
		if (securityRef != null && !securityRef.equals("")) {
			record.append(securityRef);
		}
		record.append("    </servlet>" + PlutoAdminConstants.LS);
		return record.toString();
	}

	/**
	 * Gets the web.xml security-role-ref record for PortletServlet from portlet.xml data
	 *
	 * @param context
	 *            Context name
	 * @param portletData
	 *            Data from portlet.xml
	 *
	 * @return the security role ref record
	 */
	private String getSecurityRoleRefRecord(final String context, final PortletDD portletData) {
		final StringBuffer record = new StringBuffer("");
		@SuppressWarnings("unchecked")
		final List<SecurityRoleRefDD> refs = portletData.getSecurityRoleRefs();
		String link = null;
		for (final Object element : refs) {
			final PortletConfigService.RoleRef ref = (PortletConfigService.RoleRef) element;
			record.append("      <security-role-ref>" + PlutoAdminConstants.LS);
			record.append("        <role-name>" + ref.roleName + "</role-name>" + PlutoAdminConstants.LS);
			//role-link is optional
			link = ref.roleLink;
			if (link != null && !link.equals("")) {
				record.append("        <role-link>" + link + "</role-link>" + PlutoAdminConstants.LS);
			}
			record.append("      </security-role-ref>" + PlutoAdminConstants.LS);
		}
		return record.toString();
	}

	/**
	 * Gets the web.xml servlet-mapping record for PortletServlet from portlet.xml data
	 *
	 * @param portletData
	 *            Data from portlet.xml
	 *
	 * @return the servlet mapping record
	 */
	private String getServletMappingRecord(final PortletDD portletData) {
		final StringBuffer record = new StringBuffer();
		record.append("    <servlet-mapping>" + PlutoAdminConstants.LS);
		record.append("      <servlet-name>" + portletData.getPortletName() + "</servlet-name>" + PlutoAdminConstants.LS);
		//pattern = /PortletName/*
		record.append("      <url-pattern>/" + portletData.getPortletName() + "/*</url-pattern>" + PlutoAdminConstants.LS);
		record.append("    </servlet-mapping>" + PlutoAdminConstants.LS);
		return record.toString();
	}
}