/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.portail.om.user.impl;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.pluto.om.common.Preference;
import org.apache.pluto.om.common.PreferenceSet;
import org.apache.pluto.portalImpl.om.common.impl.PreferenceSetImpl;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.webutils.ContextePage;
import com.univ.objetspartages.om.InfosPreferences;
import com.univ.objetspartages.om.Preferences;
import com.univ.portail.om.user.UserPreference;

// TODO: Auto-generated Javadoc
/**
 * The Class UserPreferenceImpl.
 */
public class UserPreferenceImpl implements UserPreference {

	private static org.slf4j.Logger LOG = LoggerFactory.getLogger(UserPreferenceImpl.class);

	//    /**
	//     * Liste des infos préférences
	//     * Correspond à la liste des infos préférences issue de la base de données 
	//     */
	//    private List listeInfosPreferences = null;
	/**
	 * Table des infos préférences stockées en fonction du service auquel elles sont rattachées. Correspond à un tri des infos préférences pour un accès plus rapide vers les infos
	 * préférences qui nous concerne.
	 */
	private Map<String, List<InfosPreferences>> tableInfosPreferences = null;

	/** Code de l'utilisateur connecté. */
	private String userCode = null;

	/**
	 * Mettre a jour preference.
	 * 
	 * @param infosPreferences
	 *            the infos preferences
	 */
	private void mettreAJourPreference(final InfosPreferences infosPreferences) {
		final ContextePage ctx = new ContextePage("");
		final Preferences pref = new Preferences();
		pref.init();
		pref.setCtx(ctx);
		try {
			pref.selectUserPreferences(userCode, infosPreferences.getService(), infosPreferences.getNom());
			if (pref.nextItem()) {
				pref.setValeur(infosPreferences.getValeursAsString());
				pref.update();
			}
		} catch (final Exception ex) {
			LOG.debug("Mise à jour de la préférence impossible: '" + pref.getNom() + "'.");
		} finally {
			ctx.release();
		}
	}

	/**
	 * Permet d'ajout une préférence utilisateur.
	 * 
	 * @param service
	 *            the service
	 * @param nom
	 *            the nom
	 * @param values
	 *            the values
	 * 
	 * @return the preferences
	 * 
	 * @throws Exception
	 */
	private Preferences ajouterPreference(final String service, final String nom, final Iterator<String> values) {
		final ContextePage ctx = new ContextePage("");
		Preferences pref = new Preferences();
		pref.init();
		pref.setCtx(ctx);
		pref.setNom(nom);
		pref.setType(Preferences.TYPE_SERVICE);
		pref.setService(service);
		pref.setValues(values);
		pref.setCodeUtilisateur(userCode);
		try {
			pref.add();
		} catch (final Exception ex) {
			LOG.error("Impossible d'enregistrer une préférence utilisateur pour l'utilisateur: '" + userCode + "'.");
			pref = null;
			//throws Ex;
		} finally {
			ctx.release();
		}
		return pref;
	}

	/**
	 * Permet de retourner une infopréférence en fonction du code service et du nom de la préférence.
	 * 
	 * @param codeService
	 *            code du service
	 * @param name
	 *            nom de la préférence
	 * 
	 * @return InfoPreference
	 */
	private InfosPreferences getInfoPreference(final String codeService, final String name) {
		if (name == null || codeService == null) {
			return null;
		}
		final Iterator<InfosPreferences> it = getUserPreferencesFromService(codeService).iterator();
		InfosPreferences infoPref = null;
		while (infoPref == null && it.hasNext()) {
			final InfosPreferences ip = it.next();
			if (name.equals(ip.getNom())) {
				infoPref = ip;
			}
		}
		return infoPref;
	}

	/**
	 * constructeur.
	 * 
	 * @param tableInfosPref
	 *            the table infos pref
	 * @param userCode
	 *            the user code
	 */
	public UserPreferenceImpl(final Map<String, List<InfosPreferences>> tableInfosPref, final String userCode) {
		this.tableInfosPreferences = tableInfosPref;
		this.userCode = userCode;
	}

	/**
	 * On enregistre les préférences utilisateurs en base. l'accès est protégé par un synchronized pour empêcher plx accès simultanés. (multi threading..)
	 * 
	 * @param prefSet
	 *            the pref set
	 * @param codeService
	 *            the code service
	 * 
	 * @see UserPreference#store()
	 */
	@Override
	public synchronized void store(final PreferenceSet prefSet, final String codeService) {
		ContextePage ctx = null;
		try {
			ctx = new ContextePage("");
			//on récupère la liste des
			final List<InfosPreferences> listePrefOrig = getUserPreferencesFromService(codeService);
			final List<InfosPreferences> newListePref = new ArrayList<InfosPreferences>();
			{
				Preference pref = null;
				InfosPreferences infoPref = null;
				//on parcourt le set de préférences reçu pour mettre à jour les préférences originales
				for (final Iterator<Preference> it = prefSet.iterator(); it.hasNext();) {
					pref = it.next();
					if (pref != null) {
						//on récupère l'info préférence dans la liste orginale
						infoPref = getInfoPreference(codeService, pref.getName());
						if (infoPref != null) {
							/*
							 * l'info pref existe dans la liste originale, 
							 * on modifie sa valeur, on l'enregistre en base et on le stocke dans la nouvelle liste
							 */
							infoPref.setValeur(pref.getValues());
							mettreAJourPreference(infoPref);
							//on retire de la liste originale l'info préférence ainsi crée.
							listePrefOrig.remove(infoPref);
						} else {
							//l'info préférence n'existe pas, il s'agit d'un nouvel info préférence à créer
							infoPref = new InfosPreferences(ajouterPreference(codeService, pref.getName(), pref.getValues()));
						}
						newListePref.add(infoPref);
					}
				}
			}
			//à ce stade, il ne reste dans la liste originale que les préférences qui ont été supprimées.$
			//on met à jour la liste des préférences pour la rendre accessible
			tableInfosPreferences.remove(codeService);
			tableInfosPreferences.put(codeService, newListePref);
			//=> on supprime les préférences utilisateur
			{
				final Iterator<InfosPreferences> itOrig = listePrefOrig.iterator();
				InfosPreferences infosPref = null;
				final Preferences pref = new Preferences();
				pref.setCtx(ctx);
				while (itOrig.hasNext()) {
					infosPref = itOrig.next();
					try {
						pref.selectUserPreferences(userCode, infosPref.getService(), infosPref.getNom());
						if (pref.nextItem()) {
							pref.delete();
						}
					} catch (final Exception ex) {
						final String messageErreur = "Exception lors de la suppression de la préférence '" + infosPref.getNom() + "' de l'utilisateur '" + userCode + "'.";
						LOG.error(messageErreur);
					}
				}
			}
		} finally {
			ctx.release();
		}
	}

	/**
	 * Gets the preference set.
	 * 
	 * @param codeService
	 *            the code service
	 * 
	 * @return the preference set
	 * 
	 * @see UserPreference#getPreferenceSet(String)
	 */
	public PreferenceSet getPreferenceSet(final String codeService) {
		/*
		 * On récupère la liste des infos préférences et on l'ajoute au préférenceSet ainsi créé.
		 */
		final PreferenceSetImpl prefSet = new PreferenceSetImpl();
		final List<InfosPreferences> listeInfosPreferences = getUserPreferencesFromService(codeService);
		for (final InfosPreferences infosPreferences : listeInfosPreferences) {
			prefSet.add(infosPreferences);
		}
		return prefSet;
	}

	/* (non-Javadoc)
	 * @see com.univ.portail.om.user.UserPreference#getUserPreferencesFromService(java.lang.String)
	 */
	@Override
	public List<InfosPreferences> getUserPreferencesFromService(final String service) {
		if (tableInfosPreferences.containsKey(service)) {
			return tableInfosPreferences.get(service);
		} else {
			return new ArrayList<InfosPreferences>();
		}
	}
}
