/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*

 */
package com.univ.portail.service;

import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;

import org.apache.pluto.util.StringUtils;

import com.kportal.core.config.PropertyHelper;
import com.univ.portail.util.Properties;

// TODO: Auto-generated Javadoc
/**
 * Manages the life-time of services registered during servlet startup. There are two types of services: Services required by the container, and Services used for the portal driver
 * configuration and execution. All services required by the container have to derive from {@link org.apache.pluto.services.ContainerService} and implement the <CODE>init()</CODE>
 * and <CODE>destroy()</CODE> methods as appropriate. All services must derive from the {@link org.apache.pluto.portalImpl.services.ServiceInit} interface if they are to be used
 * with the portal driver.
 * <P>
 * By registering the service and its implementation in the file <CODE>/config/services.properties</CODE>, the service will become available to the portal engine. The format of the
 * file is simple:
 *
 * <PRE>
 *
 * org.apache.pluto.portalImpl.services.log.Logger = org.apache.pluto.portalImpl.services.log.LogServicesImpl
 *
 * </PRE>
 *
 * Each entry represents one service. The left-hand side is the abstract service class, the right-hand side is the implementation of this service. The services are initialized in
 * the order of appearance.
 *
 * <P>
 * Each service can have its own configuration file, located in <CODE>/config/services</CODE>. It has to have the name of either implementation or abstract class of the service,
 * without the leading package name. For example, the service manager looks for <CODE>LoggerImpl .properties</CODE>. This allows a special implementation to provide different
 * configuration than the general (abstract) service requires.
 *
 * <P>
 * If present, one of the services configuration files is loaded and passed to the service as {@link org.apache.pluto.portalImpl.util.Properties} object. Not providing a service
 * configuration file is okay too, in that case the properties are emptyemptySessionPath .
 *
 * @see org.apache.pluto.services.ContainerService
 */
public class ServiceManager {

	/** Préfixe permettant d'identifier un service devant être initialisé. */
	private final static String CLE_SERVICE_MANAGER = "ServiceManager.";

	/**
	 * Récupère les propriétés des classes à initialiser dans le jtf...
	 *
	 * @param prefixe
	 *            préfixe permettant de reconnaître les paramètres...
	 *
	 * @return the properties
	 */
	private static Map<String, Object> getProperties(String prefixe) {
		if (!prefixe.endsWith(".")) {
			prefixe = prefixe.concat(".");
		}
		final Map<String, Object> props = new TreeMap<String, Object>();
		final Enumeration<Object> enumProps = PropertyHelper.getProperties("core").keys();
		while (enumProps.hasMoreElements()) {
			final String cleProp = (String) enumProps.nextElement();
			if (cleProp.startsWith(prefixe)) {
				final String cle = cleProp.substring(prefixe.length());
				final String propriete = PropertyHelper.getCoreProperty(cleProp);
				if (propriete.indexOf(",") != -1) {
					props.put(cle, Arrays.asList(propriete.split(",")));
				} else {
					props.put(cle, propriete);
				}
			}
		}
		return props;
	}

	/**
	 * * Initializes all services specified in <CODE>services.properties</CODE>. * By specifying a different implementation of the service the behaviour * of the portal can be
	 * modified. * *
	 *
	 * @param aConfig
	 *            * the servlet configuration *
	 *
	 * @throws Exception
	 *             the exception
	 *
	 * @exception Exception
	 *                * if loading <CODE>services.properties</CODE> * or initializing any of its contained services fails
	 */
	public static void init(final ServletConfig aConfig) throws Exception {
		// avoid duplicate initialization of services
		if (!cInitialized) {
			synchronized (ServiceManager.class) {
				if (!cInitialized) {
					cInitialized = true;
				} else {
					return;
				}
			}
		} else {
			return;
		}
		ServletContext context = null;
		if (aConfig != null) {
			context = aConfig.getServletContext();
		}
		if (context != null) {
			context.log("ServiceManager: Loading services...");
		}
		int numAll = 0;
		int numSuccessful = 0;
		//récupération des clés du service manager dans le jtf
		final Map<String, Object> props = getProperties(CLE_SERVICE_MANAGER);
		for (final String cleService : props.keySet()) {
			final String serviceImplName = (String) props.get(cleService);
			numAll++;
			// ty to get hold of the base service
			Class<?> serviceBase;
			try {
				serviceBase = Class.forName(cleService);
			} catch (final ClassNotFoundException exc) {
				if (context != null) {
					context.log("ServiceManager: A service with name " + cleService + " cannot be found.");
				}
				continue;
			}
			Class<?> serviceImpl = null;
			ServiceInit service = null;
			try {
				serviceImpl = Class.forName(serviceImplName);
				service = (ServiceInit) serviceImpl.newInstance();
				final Properties serviceProps = new Properties();
				//on récupère les propriétés de la classe dans le jtf
				final Map<String, Object> propsService = getProperties(StringUtils.nameOf(serviceBase));
				if (propsService != null && propsService.size() != 0) {
					serviceProps.load(propsService);
				}
				if (context != null) {
					context.log(StringUtils.nameOf(serviceBase) + " initializing...");
				}
				service.init(aConfig, serviceProps);
				if (context != null) {
					context.log(StringUtils.nameOf(serviceBase) + " done.");
				}
			} catch (final ClassNotFoundException exc) {
				if (context != null) {
					context.log("ServiceManager: A service implementation with name " + serviceImplName + " cannot be found.", exc);
				}
				continue;
			} catch (final ClassCastException exc) {
				if (context != null) {
					context.log("ServiceManager: Service implementation " + serviceImplName + " is not a service of the required type.", exc);
				}
				continue;
			} catch (final InstantiationException exc) {
				if (context != null) {
					context.log("ServiceManager: Service implementation " + serviceImplName + " cannot be instantiated.", exc);
				}
				continue;
			} catch (final Exception exc) {
				if (context != null) {
					context.log("ServiceManager: An unidentified error occurred", exc);
				}
				service = null;
			}
			if (service != null) {
				cServicesMap.put(serviceBase, service);
				// build up list in reverse order for later destruction
				cServicesList.add(0, service);
				numSuccessful++;
			}
		}
		if (context != null) {
			context.log("ServiceManager: Services initialized (" + numSuccessful + "/" + numAll + " successful).");
		}
		if (numSuccessful != numAll) {
			throw new Exception("ServiceManager: Services initialized (" + numSuccessful + "/" + numAll + " successful).");
		}
	}

	/**
	 * Calls post init for all services.
	 *
	 * @param aConfig
	 *            the servlet configuration
	 */
	public static void postInit(final ServletConfig aConfig) {
		// avoid duplicate destruction of services
		if (cInitialized) {
			synchronized (ServiceManager.class) {
				if (cInitialized) {
					cInitialized = false;
				} else {
					return;
				}
			}
		} else {
			return;
		}
		ServletContext context = null;
		if (aConfig != null) {
			context = aConfig.getServletContext();
		}
		// post init all services
		for (final ServiceInit service : cServicesList) {
			try {
				service.postInit(aConfig);
			} catch (final Exception exc) {
				if (context != null) {
					context.log("ServiceManager: Service couldn't be started (postInit) after init..", exc);
				}
			}
		}
	}

	/**
	 * * Destroys all services. * *
	 *
	 * @param aConfig
	 *            * the servlet configuration
	 */
	public static void destroy(final ServletConfig aConfig) {
		// avoid duplicate destruction of services
		if (cInitialized) {
			synchronized (ServiceManager.class) {
				if (cInitialized) {
					cInitialized = false;
				} else {
					return;
				}
			}
		} else {
			return;
		}
		ServletContext context = null;
		if (aConfig != null) {
			context = aConfig.getServletContext();
		}
		// destroy the services in reverse order
		for (final ServiceInit service : cServicesList) {
			try {
				service.destroy(aConfig);
			} catch (final Exception exc) {
				if (context != null) {
					context.log("ServiceManager: Service couldn't be destroyed.", exc);
				}
			}
		}
		cServicesList.clear();
		cServicesMap.clear();
	}

	/**
	 * * Returns the service implementation for the given service class, or * <CODE>null</CODE> if no such service is registered. * *
	 *
	 * @param aClass
	 *            * the service class * *
	 *
	 * @return the service implementation
	 */
	public static ServiceInit getService(final Class<?> aClass) {
		// at this state the services map is read-only,
		// therefore we can go without synchronization
		return cServicesMap.get(aClass);
	}

	// --- PRIVATE MEMBERS --- //
	/** The c initialized. */
	private static volatile boolean cInitialized = false;

	/** The c services map. */
	private static Map<Class<?>, ServiceInit> cServicesMap = new HashMap<Class<?>, ServiceInit>();

	/** The c services list. */
	private static List<ServiceInit> cServicesList = new LinkedList<ServiceInit>();
}
