/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* 

 */
package com.univ.portail.service.factorymanager;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;

import org.apache.pluto.factory.Factory;
import org.apache.pluto.util.StringUtils;

import com.univ.portail.util.Properties;

// TODO: Auto-generated Javadoc
/**
 * Manages the life-time of factories registered during container startup. A service has to derive from {@link Factory} and implement the <CODE>init()</CODE> and
 * <CODE>destroy()</CODE> methods as appropriate.
 * 
 * @see Factory
 */
public class FactoryManagerServiceImpl extends FactoryManagerService {

	/** The Constant CONFIG_FACTORY_PRE. */
	private final static String CONFIG_FACTORY_PRE = "factory.";

	/**
	 * * Initializes all factories specified in the configuration beginning with 'factory.'. * By specifying a different implementation of the factory the behaviour * of the
	 * portlet container can be modified. * * @param config * the servlet configuration * * @exception Exception * if initializing any of the factories fails
	 * 
	 * @param config
	 *            the config
	 * @param aProperties
	 *            the a properties
	 * 
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected void init(final ServletConfig config, final Properties aProperties) throws Exception {
		ServletContext context = null;
		if (config != null) {
			context = config.getServletContext();
		}
		if (context != null) {
			context.log("FactoryManager: Loading factories...");
		}
		final Map<String, String> factoryImpls = new HashMap<String, String>();
		final Map<String, HashMap<String, String>> factoryProps = new HashMap<String, HashMap<String, String>>();
		final Iterator<String> configNames = aProperties.keys();
		String lastFactoryInterfaceName = null;
		while (configNames.hasNext()) {
			final String configName = configNames.next();
			if (configName.startsWith(CONFIG_FACTORY_PRE)) {
				final String name = configName.substring(CONFIG_FACTORY_PRE.length());
				if ((lastFactoryInterfaceName != null) && (name.startsWith(lastFactoryInterfaceName))) {
					final String propertyName = name.substring(lastFactoryInterfaceName.length() + 1);
					final String propertyValue = aProperties.getString(configName);
					final Map<String, String> properties = factoryProps.get(lastFactoryInterfaceName);
					properties.put(propertyName, propertyValue);
				} else {
					final String factoryInterfaceName = name;
					final String factoryImplName = aProperties.getString(configName);
					factoryImpls.put(factoryInterfaceName, factoryImplName);
					factoryProps.put(factoryInterfaceName, new HashMap<String, String>());
					// remember interface name to get all properties
					lastFactoryInterfaceName = factoryInterfaceName;
				}
			}
		}
		int numAll = 0;
		for (final String factoryInterfaceName : factoryImpls.keySet()) {
			numAll++;
			// try to get hold of the factory
			Class<?> factoryInterface;
			try {
				factoryInterface = Class.forName(factoryInterfaceName);
			} catch (final ClassNotFoundException exc) {
				if (context != null) {
					context.log("FactoryManager: A factory with name " + factoryInterfaceName + " cannot be found.");
				}
				continue;
			}
			final String factoryImplName = factoryImpls.get(factoryInterfaceName);
			Class<?> factoryImpl = null;
			Factory factory = null;
			try {
				factoryImpl = Class.forName(factoryImplName);
				factory = (Factory) factoryImpl.newInstance();
				final HashMap<String, String> props = factoryProps.get(factoryInterfaceName);
				if (context != null) {
					context.log(StringUtils.nameOf(factoryInterface) + " initializing...");
				}
				factory.init(config, props);
				if (context != null) {
					context.log(StringUtils.nameOf(factoryInterface) + " done.");
				}
			} catch (final ClassNotFoundException exc) {
				if (context != null) {
					context.log("FactoryManager: A factory implementation with name " + factoryImplName + " cannot be found.", exc);
				}
				throw exc;
			} catch (final ClassCastException exc) {
				if (context != null) {
					context.log("FactoryManager: Factory implementation " + factoryImplName + " is not a factory of the required type.", exc);
				}
				throw exc;
			} catch (final InstantiationException exc) {
				if (context != null) {
					context.log("FactoryManager: Factory implementation " + factoryImplName + " cannot be instantiated.", exc);
				}
				throw exc;
			} catch (final Exception exc) {
				if (context != null) {
					context.log("FactoryManager: An unidentified error occurred", exc);
				}
				throw exc;
			}
			if (factory != null) {
				factoryMap.put(factoryInterface, factory);
				// build up list in reverse order for later destruction
				factoryList.add(0, factory);
			}
		}
		if (context != null) {
			context.log("FactoryManager: Factories initialized (" + numAll + " successful).");
		}
	}

	/**
	 * * Destroys all services. * * @param config * the servlet configuration
	 * 
	 * @param config
	 *            the config
	 */
	@Override
	protected void destroy(final ServletConfig config) {
		ServletContext context = null;
		if (config != null) {
			context = config.getServletContext();
		}
		// destroy the services in reverse order
		for (final Factory factory : factoryList) {
			try {
				factory.destroy();
			} catch (final Exception exc) {
				if (context != null) {
					context.log("FactoryManager: Factory couldn't be destroyed.", exc);
				}
			}
		}
		factoryList.clear();
		factoryMap.clear();
	}

	/**
	 * * Returns the service implementation for the given service class, or * <CODE>null</CODE> if no such service is registered. * * @param theClass * the service class * * @return
	 * the service implementation
	 * 
	 * @param theClass
	 *            the the class
	 * 
	 * @return the factory
	 */
	@Override
	public Factory getFactory(final Class theClass) {
		// at this state the services map is read-only,
		// therefore we can go without synchronization
		return factoryMap.get(theClass);
	}

	// --- PRIVATE MEMBERS --- //
	/** The factory map. */
	private final Map<Class<?>, Factory> factoryMap = new HashMap<Class<?>, Factory>();

	/** The factory list. */
	private final List<Factory> factoryList = new LinkedList<Factory>();
}
