/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.portail.service.registry;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.UnavailableException;

import org.apache.pluto.om.common.ObjectID;
import org.apache.pluto.om.portlet.PortletApplicationDefinition;
import org.apache.pluto.om.portlet.PortletApplicationDefinitionList;
import org.apache.pluto.om.portlet.PortletDefinition;
import org.apache.pluto.om.portlet.PortletDefinitionList;
import org.exolab.castor.mapping.Mapping;
import org.exolab.castor.mapping.MappingException;
import org.exolab.castor.xml.MarshalException;
import org.exolab.castor.xml.Unmarshaller;
import org.exolab.castor.xml.ValidationException;
import org.slf4j.LoggerFactory;
import org.xml.sax.InputSource;

import com.kportal.core.config.PropertyHelper;
import com.univ.portail.om.definition.impl.PortletApplicationDefinitionImpl;
import com.univ.portail.om.definition.impl.PortletApplicationDefinitionListImpl;
import com.univ.portail.om.definition.impl.PortletDefinitionListImpl;
import com.univ.portail.om.servlet.impl.WebApplicationDefinitionImpl;
import com.univ.portail.service.ServiceInit;
import com.univ.portail.util.PortletHelper;
import com.univ.portail.util.Properties;
import com.univ.portail.util.xml.Constants;
import com.univ.portail.util.xml.XmlParser.EntityResolver;

// TODO: Auto-generated Javadoc
/**
 * The Class ServiceChargementPorltetsDefinition.
 */
public class ServiceChargementPorltetsDefinition extends ServiceInit {

	/** The Constant CONFIG_MAPPING_PORTLETXML. */
	private static final String CONFIG_MAPPING_PORTLETXML = "mapping.portletxml.configfile";

	/** The Constant CONFIG_MAPPING_WEBXML. */
	private static final String CONFIG_MAPPING_WEBXML = "mapping.webxml.configfile";

	/** The Constant DEFAULT_MAPPING_PORTLETXML. */
	private final static String DEFAULT_MAPPING_PORTLETXML = "portletdefinitionmapping.xml";

	/** The Constant DEFAULT_MAPPING_WEBXML. */
	private final static String DEFAULT_MAPPING_WEBXML = "servletdefinitionmapping.xml";

	/** The liste application definition. */
	private PortletApplicationDefinitionList listeApplicationDefinition = null;

	/** The liste definition. */
	private PortletDefinitionList listeDefinition = null;

	/** The web xml mapping. */
	private Mapping webXmlMapping = null;

	/** The portlet xml mapping. */
	private Mapping portletXmlMapping = null;

	/** The definitions. */
	private final Map<ObjectID, PortletDefinition> definitions = new Hashtable<ObjectID, PortletDefinition>();

	/** The liste application chargees. */
	private List<String> listeApplicationChargees = null;

	private static org.slf4j.Logger LOG = LoggerFactory.getLogger(ServiceChargementPorltetsDefinition.class);

	/* (non-Javadoc)
	 * @see com.univ.portail.service.ServiceInit#init(javax.servlet.ServletConfig, com.univ.portail.util.Properties)
	 */
	@Override
	protected void init(final ServletConfig config, final Properties props) throws Exception {
		final ServletContext servletContext = config.getServletContext();
		final String cheminFichierConf = PropertyHelper.getCoreProperty(PortletHelper.PARAM_CHEMIN_CONFIGURATION_DIR);
		portletXmlMapping = loadMapping(servletContext, props.getString(CONFIG_MAPPING_PORTLETXML, cheminFichierConf + DEFAULT_MAPPING_PORTLETXML));
		webXmlMapping = loadMapping(servletContext, props.getString(CONFIG_MAPPING_WEBXML, cheminFichierConf + DEFAULT_MAPPING_WEBXML));
		if (listeApplicationDefinition == null) {
			listeApplicationDefinition = new PortletApplicationDefinitionListImpl();
		}
		if (listeDefinition == null) {
			listeDefinition = new PortletDefinitionListImpl();
		}
		if (listeApplicationChargees == null) {
			listeApplicationChargees = new ArrayList<String>();
			//initialisation(config);
		}
	}

	/**
	 * Force le rechargement de toutes les applications.
	 * 
	 * @throws Exception
	 */
	public void forcerRelectureApplication() {
		listeApplicationDefinition = new PortletApplicationDefinitionListImpl();
		listeDefinition = new PortletDefinitionListImpl();
		listeApplicationChargees = new ArrayList<String>();
	}

	/**
	 * Charger portlet definition.
	 * 
	 * @param definitionId
	 *            the definition id
	 * @param servletContext
	 *            the servlet context
	 */
	public synchronized void chargerPortletDefinition(final String definitionId, final ServletContext servletContext) {
		PortletApplicationDefinition application = null;
		String ajout = definitionId;
		if (definitionId.contains(".")) {
			ajout = definitionId.substring(0, definitionId.indexOf("."));
		}
		final String applicationContexteName = "/".concat(ajout);
		try {
			//on vérifie tout d'abord que l'application n'a pas déjà été chargée
			if (!listeApplicationChargees.contains(applicationContexteName)) {
				//l'application n'a pas été chargée, on ne le refait pas.
				application = loadApplicationDefinition(servletContext, applicationContexteName);
				if (application != null) {
					listeApplicationChargees.add(applicationContexteName);
					((PortletApplicationDefinitionListImpl) listeApplicationDefinition).add(application);
					for (final Iterator<PortletDefinition> j = application.getPortletDefinitionList().iterator(); j.hasNext();) {
						final PortletDefinition portlet = j.next();
						definitions.put(portlet.getId(), portlet);
						((PortletDefinitionListImpl) listeDefinition).add(portlet);
					}
				} else {
					final String messageErreur = "Impossible de charger le contexte suivant: " + applicationContexteName;
					throw new UnavailableException(messageErreur);
				}
			} else {
				//l'application a déjà été chargée, on ne fait rien.
			}
		} catch (final UnavailableException e) {
			// TODO Gérer le cas où le contexte ne peut être chargé... cf norme JSR 168
			LOG.error("Impossible de charger le contexte suivant: " + applicationContexteName, e);
		}
	}

	/**
	 * Gets the application definition list.
	 * 
	 * @return the application definition list
	 */
	public PortletApplicationDefinitionList getApplicationDefinitionList() {
		return listeApplicationDefinition;
	}

	/**
	 * Gets the definition list.
	 * 
	 * @return the definition list
	 */
	public PortletDefinitionList getDefinitionList() {
		return listeDefinition;
	}

	/**
	 * Gets the.
	 * 
	 * @param objectId
	 *            the object id
	 * 
	 * @return the portlet definition
	 */
	public PortletDefinition get(final ObjectID objectId) {
		return definitions.get(objectId);
	}

	/**
	 * Load mapping.
	 * 
	 * @param context
	 *            the context
	 * @param path
	 *            the path
	 * 
	 * @return the mapping
	 * 
	 * @throws UnavailableException
	 *             the unavailable exception
	 */
	private Mapping loadMapping(final ServletContext context, final String path) throws UnavailableException {
		final InputSource source = new InputSource(context.getResourceAsStream(path));
		final Mapping mapping = new Mapping();
		try {
			mapping.loadMapping(source);
		} catch (final IOException e) {
			throw (UnavailableException) new UnavailableException("Error reading mapping " + path).initCause(e);
		} catch (final MappingException e) {
			throw (UnavailableException) new UnavailableException("Invalid mapping " + path).initCause(e);
		}
		return mapping;
	}

	/**
	 * Load application definition.
	 * 
	 * @param context
	 *            the context
	 * @param path
	 *            the path
	 * 
	 * @return the portlet application definition
	 * 
	 * @throws UnavailableException
	 *             the unavailable exception
	 */
	private PortletApplicationDefinition loadApplicationDefinition(final ServletContext context, final String path) throws UnavailableException {
		// locate the portlet application's context
		final ServletContext appContext = context.getContext(path);
		if (appContext == null) {
			throw new UnavailableException("Unable to access context for " + path);
		}
		// load its portlet.xml
		InputStream stream = appContext.getResourceAsStream("/WEB-INF/portlet.xml");
		if (stream == null) {
			throw new UnavailableException("No portlet.xml found in context " + appContext.getServletContextName());
		}
		InputSource source = new InputSource(stream);
		Unmarshaller unmarshaller;
		try {
			unmarshaller = new Unmarshaller(portletXmlMapping);
			unmarshaller.setEntityResolver(new EntityResolver(context, Constants.RES_PORTLET_DTDS, Constants.RES_PORTLET_DTD_NAMES));
		} catch (final MappingException e) {
			throw (UnavailableException) new UnavailableException("Unable to construct unmarshaller for portlet.xml").initCause(e);
		}
		unmarshaller.setIgnoreExtraElements(true);
		PortletApplicationDefinitionImpl portletApplicationDefinition;
		try {
			portletApplicationDefinition = (PortletApplicationDefinitionImpl) unmarshaller.unmarshal(source);
		} catch (final MarshalException e) {
			throw (UnavailableException) new UnavailableException("Unable to unmarshal portlet.xml from context " + appContext.getServletContextName()).initCause(e);
		} catch (final ValidationException e) {
			throw (UnavailableException) new UnavailableException("Unable to validate portlet.xml from context " + appContext.getServletContextName()).initCause(e);
		}
		// load its web.xml
		stream = appContext.getResourceAsStream("/WEB-INF/web.xml");
		if (stream == null) {
			throw new UnavailableException("No web.xml found in context " + appContext.getServletContextName());
		}
		source = new InputSource(stream);
		try {
			unmarshaller = new Unmarshaller(webXmlMapping);
			unmarshaller.setIgnoreExtraAttributes(true);
			unmarshaller.setIgnoreExtraElements(true);
			unmarshaller.setEntityResolver(new EntityResolver(context, Constants.RES_WEB_PUBLIC_ID, Constants.RES_WEB_DTD, Constants.RES_WEB_DTD_NAME));
		} catch (final MappingException e) {
			throw (UnavailableException) new UnavailableException("Unable to construct unmarshaller for web.xml").initCause(e);
		}
		unmarshaller.setIgnoreExtraElements(true);
		WebApplicationDefinitionImpl webApplicationDefinition = null;
		try {
			webApplicationDefinition = (WebApplicationDefinitionImpl) unmarshaller.unmarshal(source);
		} catch (final MarshalException e) {
			throw (UnavailableException) new UnavailableException("Unable to unmarshal web.xml from context " + appContext.getServletContextName()).initCause(e);
		} catch (final ValidationException e) {
			throw (UnavailableException) new UnavailableException("Unable to validate web.xml from context " + appContext.getServletContextName()).initCause(e);
		}
		final Vector<Serializable> structure = new Vector<Serializable>();
		structure.add(portletApplicationDefinition);
		structure.add(path);
		try {
			webApplicationDefinition.postLoad(structure);
			webApplicationDefinition.preBuild(structure);
			webApplicationDefinition.postBuild(structure);
		} catch (final Exception e) {
			throw (UnavailableException) new UnavailableException(e.getMessage()).initCause(e);
		}
		return portletApplicationDefinition;
	}
}
