/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.rss;

import java.lang.reflect.Method;
import java.text.SimpleDateFormat;

import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.kportal.core.config.PropertyHelper;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Libelle;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Ressource;
import com.univ.objetspartages.om.Structure;
import com.univ.utils.ContexteUniv;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.RequeteUtil;
import com.univ.utils.URLResolver;
import com.univ.utils.UnivWebFmt;

// TODO: Auto-generated Javadoc
/**
 * Classe réalisant un export RSS sur une actualité.
 *
 * @author fred
 */
public class ExportRssDefaut extends AbstractExportRss {

	/**
	 * Instantiates a new export rss defaut.
	 *
	 * @param query
	 *            the query
	 * @param objectName
	 *            the object name
	 * @param feedType
	 *            the feed type
	 */
	public ExportRssDefaut(final String query, final String objectName, final String feedType) {
		super(query, objectName, feedType);
	}

	/**
	 * Renvoie la description du channel correspondant à cet export rss.
	 *
	 * @return La description du channel
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected String getDescription(final OMContext ctx) throws Exception {
		// on recupere la description dans la requete
		String description = super.getDescription(ctx);
		if (description.length() == 0) // sinon
		{
			final String thematique = RequeteUtil.renvoyerParametre(getQuery(), "THEMATIQUE");
			if (thematique.length() > 0) {
				description += "\"" + Libelle.getLibelle("04", thematique, ctx.getLocale()) + "\" ";
			}
			final String codeRattachement = RequeteUtil.renvoyerParametre(getQuery(), "CODE_RATTACHEMENT");
			if (codeRattachement.length() > 0) {
				description += "[" + Structure.getLibelleRattachement(ctx, codeRattachement, "0", true).replaceAll("<br />", " / ") + "]";
			}
			if (description.length() == 0) {
				final String objet = RequeteUtil.renvoyerParametre(getQuery(), "OBJET");
				if (objet.length() > 0) {
					description = "Les ";
					final String nombre = RequeteUtil.renvoyerParametre(getQuery(), "NOMBRE");
					if (nombre.length() > 0) {
						description += nombre + " ";
					}
					description += "dernières fiches " + ReferentielObjets.getLibelleObjet(ReferentielObjets.getCodeObjet(objet));
				}
			} else {
				description = "Sélection " + description;
			}
		}
		return description;
	}

	/**
	 * Renvoie le titre de l'actu.
	 *
	 * @param fiche
	 *            L'actu à exporter
	 *
	 * @return Le titre
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected String getItemTitle(final FicheUniv fiche) throws Exception {
		return fiche.getLibelleAffichable();
	}

	/**
	 * Renvoie la description de l'actu.
	 *
	 * @param fiche
	 *            L'actu à exporter
	 *
	 * @return La description
	 *
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected String getItemDescription(final ContexteUniv ctx, final FicheUniv fiche) throws Exception {
		String description = "";
		final Ressource ressource = Ressource.getFichier(fiche);
		if (ressource != null) {
			final String urlVignette = URLResolver.getAbsoluteUrl(ressource.getUrlVignette(), ctx);
			description = "<img src=\"" + urlVignette + "\" width=\"150px\" /><br />";
		}
		final String codeObjet = ReferentielObjets.getCodeObjet(fiche);
		final String nomObjet = ReferentielObjets.getNomObjet(codeObjet);
		final String libelleObjet = ReferentielObjets.getLibelleObjet(codeObjet);
		final String optionExport = PropertyHelper.getCoreProperty("export_rss." + nomObjet + ".description");
		// par defaut on affiche le type d'objet et la date
		if (optionExport == null || optionExport.length() == 0) {
			final Metatag meta = FicheUnivMgr.lireMeta(fiche);
			description += "Fiche " + libelleObjet + " - " + new SimpleDateFormat("dd/MM/yyyy").format(meta.getMetaDateMiseEnLigne());
		} else {
			try {
				// Récupération de la liste de valeurs
				final String nomAccesseurGET = "get" + optionExport.substring(0, 1).toUpperCase() + optionExport.substring(1);
				// ex méthode getParcourss() sur l'objet Formation
				final Method accesseurGET = fiche.getClass().getMethod(nomAccesseurGET, new Class[0]);
				final Object o[] = new Object[0];
				description += (String) accesseurGET.invoke(fiche, o);
			} catch (final Exception e) {
				throw new ErreurApplicative("Attribut " + optionExport + " pour la fiche " + libelleObjet);
			}
		}
		//	on a besoin de cette info pour le formatage HTML
		ctx.setFicheCourante(fiche);
		return UnivWebFmt.formaterEnHTML(ctx, description);
	}
}
