/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.tree.processus;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import com.univ.objetspartages.om.AutorisationBean;
import com.univ.tree.bean.JsTreeModel;
import com.univ.tree.bean.JsTreeNodeModel;

/**
 * Classe abstraite à implementer pour réaliser les traitements d'un JsTree
 * 
 * @author fabien.leconte
 * 
 */
public abstract class GestionJsTree<T> {

	/**
	 * Génère une arborescence JsTree en fonction des paramètres passés dans la requête <code>req</code>
	 * 
	 * @param req
	 * @return
	 */
	public abstract JsTreeModel traiterDepuisRequete(HttpServletRequest req);

	/**
	 * Permet de traiter les critères de filtre reçu depuis un JsTree
	 * 
	 * @param req
	 * @return
	 */
	public abstract JsTreeModel traiterFiltreDepuisRequete(HttpServletRequest req);

	/**
	 * Permet de traiter les critères de recherche reçu depuis un JsTree
	 * 
	 * @param req
	 * @return
	 */
	public abstract JsTreeModel traiterRechercheDepuisRequete(HttpServletRequest req);

	/**
	 * Assure la consistance des paramètres passés dans la requête
	 * 
	 * @param req
	 * @return
	 */
	public abstract void assertParametersConsistency(final Map<String, String> parameters) throws Exception;

	/**
	 * Permet de traiter des actions d'un JsTree (suppression de masse, archivage...) Par défaut il n'y a pas d'action de masse.
	 * 
	 * @param parametresDeLaRequete
	 */
	public String traiterAction(AutorisationBean autorisations, Map<String, String[]> parametresDeLaRequete) {
		return "";
	}

	/**
	 * Permet d'ouvrir les branches de l'arbre avant même qu'il soit affiché
	 * 
	 * @param nodes
	 *            : liste noeuds à partir desquels les branches doivent être ouverte
	 */
	protected void openAllNodes(final List<JsTreeNodeModel> nodes) {
		for (JsTreeNodeModel currentNode : nodes) {
			if (currentNode != null && !currentNode.getChildren().isEmpty()) {
				currentNode.setState("open");
				openAllNodes(currentNode.getChildren());
			}
		}
	}

	protected abstract JsTreeModel filterTree(AutorisationBean autorisations, T object, HashMap<String, String> parameters);

	public abstract String getSelectedIds(String string);
}
