/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.tree.processus;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.exception.ErreurAsyncException;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.core.config.MessageHelper;
import com.univ.objetspartages.cache.CacheGroupeDsiManager;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Groupedsi;
import com.univ.objetspartages.om.Groupeutilisateur;
import com.univ.objetspartages.om.InfosGroupeDsi;
import com.univ.objetspartages.om.Libelle;
import com.univ.objetspartages.om.Structure;
import com.univ.tree.bean.JsTreeDataModel;
import com.univ.tree.bean.JsTreeModel;
import com.univ.tree.bean.JsTreeNodeModel;
import com.univ.tree.bean.JsTreePath;
import com.univ.tree.utils.JsTreeUtils;
import com.univ.utils.ContexteUtil;
import com.univ.utils.SessionUtil;

public class GroupsJsTree extends GestionJsTree<Groupedsi> {

	private final CacheGroupeDsiManager cache = (CacheGroupeDsiManager) ApplicationContextManager.getCoreContextBean(CacheGroupeDsiManager.ID_BEAN);

	/** l'id Spring du bean. */
	public static final String ID_BEAN = "groupsJsTree";

	private static final Logger LOG = LoggerFactory.getLogger(GroupsJsTree.class);

	@Override
	public JsTreeModel traiterDepuisRequete(HttpServletRequest req) {
		AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		HashMap<String, String> parameters;
		JsTreeModel jsTree = new JsTreeModel();
		try {
			parameters = JsTreeUtils.getParameters(req);
			assertParametersConsistency(parameters);
			String permissions = parameters.get("PERMISSION");
			CacheGroupeDsiManager cache = (CacheGroupeDsiManager) ApplicationContextManager.getCoreContextBean(CacheGroupeDsiManager.ID_BEAN);
			InfosGroupeDsi groupe = cache.getListeGroupesDsi().get(parameters.get("RACINE"));
			int niveau = Integer.parseInt(parameters.get("NIVEAU"));
			final String[] selection = StringUtils.defaultString(parameters.get("SELECTED")).split(";");
			boolean locked_dyn = Boolean.parseBoolean(StringUtils.defaultString(parameters.get("LOCK_DYN"), "false"));
			jsTree.getNodes().add(buildGroupeNode(autorisations, permissions, groupe, niveau, null, locked_dyn));
			if (selection.length > 0) {
				loadToPath(autorisations, permissions, jsTree, selection, locked_dyn);
			}
		} catch (Exception e) {
			LOG.error("impossible de generer l'arborescence des groupes", e);
			jsTree = new JsTreeModel();
		}
		return jsTree;
	}

	private void loadToPath(AutorisationBean autorisations, String permissions, final JsTreeModel jsTree, String[] selection, boolean locked_dyn) {
		List<String> loadedNodes = new ArrayList<>();
		for (String code : selection) {
			try {
				final JsTreePath path = getPathToCode(code);
				if(path != null) {
					final Iterator<String> codeIt = path.getChildPath().iterator();
					final Set<String> nodes = new HashSet<>();
					String topCode = codeIt.next();
					nodes.addAll(path.getChildPath());
					if (!loadedNodes.isEmpty()) {
						while (loadedNodes.contains(topCode) && codeIt.hasNext()) {
							topCode = codeIt.next();
						}
					}
					if (StringUtils.isNotBlank(topCode)) {
						JsTreeNodeModel parentNode = JsTreeUtils.getNodeWithCode(jsTree.getNodes().get(0), topCode);
						if(parentNode != null && CollectionUtils.isEmpty(parentNode.getChildren())) {
							final InfosGroupeDsi parent = Groupedsi.renvoyerItemGroupeDsi(topCode);
							parentNode.getChildren().addAll(buildGroupeNode(autorisations, permissions, parent, -2, nodes, locked_dyn).getChildren());
							loadedNodes.addAll(nodes);
						}
					}
				}
			} catch (Exception e) {
				LOG.error("An error occured trying to get OpenPath for group tree", e);
			}
		}
	}

	private JsTreePath getPathToCode(String code) throws Exception {
		final JsTreePath path = new JsTreePath();
		final InfosGroupeDsi child = Groupedsi.renvoyerItemGroupeDsi(code);
		InfosGroupeDsi parent = child.getGroupePere();
		path.addChild(code);
		while (parent != null && !parent.getCode().equals("00")) {
			path.addChild(parent.getCode());
			parent = parent.getGroupePere();
		}
		return path;
	}

	private static JsTreeNodeModel buildGroupeNode(AutorisationBean autorisations, String permissions, InfosGroupeDsi groupe, int niveau, Set<String> ids,
		boolean locked_dyn) throws Exception {
		JsTreeNodeModel node = null;
		// Ajoute les éléments à l'arbre
		Iterator<InfosGroupeDsi> listSousGroupeIt = groupe.getListeSousGroupesSortedByLibelle().iterator();
		boolean selectable = JsTreeUtils.isGroupeSelectable(permissions, autorisations, groupe, ids);
		boolean isDynamicGroup = StringUtils.isNotBlank(groupe.getRequete());
		if (JsTreeUtils.isGroupeVisible(permissions, autorisations, groupe, selectable, ids)) {
			node = new JsTreeNodeModel();
			// on ajoute l'élément
			JsTreeDataModel datas = new JsTreeDataModel();
			if (groupe.getCode().equals(JsTreeUtils.CODE_ROOT)) {
				datas.setTitle(" ");
				node.getAttr().put("rel", "root");
				node.getAttr().put("class", "groupe_root");
				if(groupe.getListeSousGroupes().size() > 0) {
					node.setState("open");
				}
			} else {
				if (!selectable) {
					node.getAttr().put("rel", "not_selectable");
				}
				if (isDynamicGroup && locked_dyn) {
					node.getAttr().put("rel", "locked");
				}
				datas.setTitle(groupe.getIntitule());
				node.getAttr().put("class", "groupe_" + groupe.getType());
				node.getAttr().put("title", groupe.getIntitule());
				node.getMetadata().put("libelle", groupe.getIntitule());
				node.getMetadata().put("sCode", groupe.getCode());
				node.getMetadata().put("idGroupeDsi", groupe.getId().toString());
				node.getMetadata().put("type", Libelle.getLibelle("11", groupe.getType(), LangueUtil.getDefaultLocale()));
				node.getMetadata().put("structure", Structure.getLibelleAffichable(groupe.getCodeStructure(), "0"));
				if (groupe.getListeSousGroupes().size() > 0 && hasVisibleChildren(permissions, autorisations, groupe, ids)) {
					node.setState("closed");
				}
			}
			node.getMetadata().put("numchildren", Integer.toString(groupe.getListeSousGroupes().size()));
			node.getAttr().put("id", Long.toString(groupe.getId()));
			node.setData(datas);
			if (niveau > 0 || niveau == -1) {
				while (listSousGroupeIt.hasNext()) {
					InfosGroupeDsi sousGroupe = listSousGroupeIt.next();
					if (niveau != -1) {
						niveau--;
					}
					JsTreeNodeModel child = buildGroupeNode(autorisations, permissions, sousGroupe, niveau, ids, locked_dyn);
					if (child != null) {
						node.getChildren().add(child);
					}
					if (niveau != -1) {
						niveau++;
					}
				}
			} else if (niveau == -2) {
				while (listSousGroupeIt.hasNext()) {
					JsTreeNodeModel child = null;
					InfosGroupeDsi sousGroupe = listSousGroupeIt.next();
					if (!ids.contains(sousGroupe.getCode())) {
						child = buildGroupeNode(autorisations, permissions, sousGroupe, 0, null, locked_dyn);
					} else {
						child = buildGroupeNode(autorisations, permissions, sousGroupe, niveau, ids, locked_dyn);
					}
					if (child != null) {
						node.getChildren().add(child);
					}
				}
			}
		}
		return node;
	}

	@Override
	public void assertParametersConsistency(Map<String, String> parameters) throws Exception {
		if (parameters.get("CODE") == null || parameters.get("CODE").length() == 0) {
			parameters.put("CODE", JsTreeUtils.CODE_ROOT);
		}
		if (parameters.get("RACINE") == null || parameters.get("RACINE").length() == 0) {
			parameters.put("RACINE", JsTreeUtils.CODE_ROOT);
		}
	}

	@Override
	public JsTreeModel traiterRechercheDepuisRequete(HttpServletRequest req) {
		AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		HashMap<String, String> parameters;
		JsTreeModel jsTree = new JsTreeModel();
		try {
			parameters = JsTreeUtils.getParameters(req);
			assertParametersConsistency(parameters);
			Groupedsi groupe = new Groupedsi();
			groupe.setCtx(ContexteUtil.getContexteUniv());
			groupe.select(req.getParameter("CODE_RECHERCHE"), req.getParameter("TYPE"), req.getParameter("LIBELLE"), req.getParameter("CODE_STRUCTURE"), null);
			jsTree = filterTree(autorisations, groupe, parameters);
		} catch (Exception e) {
			LOG.error("impossible de generer l'arborescence des rubriques", e);
			jsTree = new JsTreeModel();
		}
		return jsTree;
	}

	@Override
	public JsTreeModel traiterFiltreDepuisRequete(HttpServletRequest req) {
		AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		HashMap<String, String> parameters;
		JsTreeModel jsTree = new JsTreeModel();
		final String query = StringUtils.defaultString(req.getParameter("QUERY"), StringUtils.EMPTY);
		try {
			parameters = JsTreeUtils.getParameters(req);
			assertParametersConsistency(parameters);
			parameters.put("NIVEAU", "-1");
			Groupedsi groupeFiltrage = new Groupedsi();
			groupeFiltrage.init();
			groupeFiltrage.setCtx(ContexteUtil.getContexteUniv());
			groupeFiltrage.selectNoCount("WHERE LIBELLE like '%" + query + "%'");
			jsTree = filterTree(autorisations, groupeFiltrage, parameters);
			openAllNodes(jsTree.getNodes());
		} catch (Exception e) {
			LOG.error(String.format("An error occured trying to filter Rubrique tree for query \"%s\"", query), e);
		}
		return jsTree;
	}

	@Override
	public String traiterAction(AutorisationBean autorisations, Map<String, String[]> parametresDeLaRequete) {
		String action = StringUtils.EMPTY;
		String message = StringUtils.EMPTY;
		if (parametresDeLaRequete.get("ACTION") != null) {
			action = parametresDeLaRequete.get("ACTION")[0];
		}
		if (action.equals("SUPPRIMER")) {
			final String[] ids = parametresDeLaRequete.get("IDS_GROUPE[]");
			if (ids != null) {
				for (String currentId : ids) {
					final Groupedsi groupedsi = new Groupedsi();
					groupedsi.setCtx(ContexteUtil.getContexteUniv());
					groupedsi.setIdGroupedsi(Long.valueOf(currentId));
					try {
						groupedsi.retrieve();
						final String libelleGroupe = groupedsi.getLibelle();
						message = String.format(MessageHelper.getCoreMessage("BO_SERVICES_ARBRE_SUPPRESSION_OK"), String.format(
							MessageHelper.getCoreMessage("BO_SERVICES_ARBRE_GROUP"), libelleGroupe));
						traiterSuppression(groupedsi);
					} catch (Exception e) {
						LOG.error("la suppression du groupe a echouee", e);
						throw new ErreurAsyncException(e.getMessage());
					}
				}
			}
		}
		return message;
	}

	private void traiterSuppression(Groupedsi groupedsi) throws Exception {
		groupedsi.delete();
		Groupeutilisateur.deleteParGroupe(ContexteUtil.getContexteUniv(), groupedsi.getCode());
		cache.flush();
	}

	public static String getPath(String root, String code, String separator) {
		List<String> path = new ArrayList<>();
		if (StringUtils.isBlank(root)) {
			root = JsTreeUtils.CODE_ROOT;
		}
		if (StringUtils.isBlank(code)) {
			return root;
		}
		try {
			InfosGroupeDsi groupe = Groupedsi.renvoyerItemGroupeDsi(code);
			while (StringUtils.isNotBlank(groupe.getCode()) && !root.equals(groupe.getCode())) {
				path.add(groupe.getIntitule());
				groupe = Groupedsi.renvoyerItemGroupeDsi(groupe.getCodeGroupePere());
			}
		} catch (Exception e) {
			LOG.error(String.format("An error occured trying to compute Group tree path to element %s from %s", code, root), e);
		}
		Collections.reverse(path);
		return StringUtils.join(path, separator);
	}

	@Override
	protected JsTreeModel filterTree(AutorisationBean autorisations, Groupedsi groupe, HashMap<String, String> parameters) {
		JsTreeModel jsTree = new JsTreeModel();
		try {
			String permissions = parameters.get("PERMISSION");
			InfosGroupeDsi groupeRacine = Groupedsi.renvoyerItemGroupeDsi(parameters.get("RACINE"));
			int niveau = Integer.parseInt(parameters.get("NIVEAU"));
			boolean locked_dyn = Boolean.parseBoolean(StringUtils.defaultString(parameters.get("LOCK_DYN"), "false"));
			final Set<String> ids = new HashSet<>();
			while (groupe.nextItem() && !groupe.getCode().equals(parameters.get("CODE"))) {
				ids.add(groupe.getCode());
				InfosGroupeDsi groupePere = Groupedsi.renvoyerItemGroupeDsi(groupe.getCodeGroupePere());
				while (StringUtils.isNotBlank(groupePere.getCode())) {
					ids.add(groupePere.getCode());
					groupePere = Groupedsi.renvoyerItemGroupeDsi(groupePere.getCodeGroupePere());
				}
			}
			if (ids.isEmpty()) {
				return jsTree;
			}
			ids.add("00");
			jsTree.getNodes().add(buildGroupeNode(autorisations, permissions, groupeRacine, niveau, ids, locked_dyn));
			return jsTree;
		} catch (Exception e) {
			LOG.error("impossible de generer l'arborescence des rubriques", e);
			jsTree = new JsTreeModel();
		}
		return jsTree;
	}

	private static boolean hasVisibleChildren(String permissions, AutorisationBean autorisations, InfosGroupeDsi groupe, Set<String> ids) {
		for (InfosGroupeDsi currentChild : groupe.getListeSousGroupes()) {
			try {
				boolean selectable = JsTreeUtils.isGroupeSelectable(permissions, autorisations, currentChild, ids);
				boolean visible = JsTreeUtils.isGroupeVisible(permissions, autorisations, currentChild, selectable, ids);
				if (visible) {
					return true;
				}
			} catch (Exception e) {}
		}
		return false;
	}

	@Override
	public String getSelectedIds(String string) {
		if (StringUtils.isEmpty(string)) {
			return StringUtils.EMPTY;
		}
		final List<String> selectedIds = new ArrayList<>();
		final String[] selections = string.split(";");
		for (String currentSelection : selections) {
			try {
				InfosGroupeDsi groupe = Groupedsi.renvoyerItemGroupeDsi(currentSelection);
				if (groupe != null && StringUtils.isNotBlank(groupe.getCode())) {
					selectedIds.add(Long.toString(groupe.getId()));
				}
			} catch (Exception e) {
				LOG.error("An error occured trying to retrieve Rubrique", e);
			}
		}
		return StringUtils.join(selectedIds, ";");
	}
}
