/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.tree.processus;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Groupedsi;
import com.univ.objetspartages.om.InfosGroupeDsi;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.Rubrique;
import com.univ.tree.bean.JsTreeDataModel;
import com.univ.tree.bean.JsTreeNodeModel;
import com.univ.tree.utils.JsTreeUtils;
import com.univ.utils.EscapeString;

/**
 * Arbre spécifique pour le processus de saisie de rubrique
 *
 * @author olivier.camon
 *
 */
public class RubriqueJsTreeProcessus extends RubriquesJsTree {

	private static final Logger LOG = LoggerFactory.getLogger(RubriqueJsTreeProcessus.class);

	public static final String ID_BEAN = "rubriqueJsTreeProcessus";

	private static final String PERMISSION_MODIFICATION = "/TECH/rub/M";

	private static final String PERMISSION_SUPPRESSION = "/TECH/rub/S";

	@Override
	protected JsTreeNodeModel buildRubriqueNode(final AutorisationBean autorisations, final String permissions, final InfosRubriques rubrique, int niveau,
		final Set<String> ids) throws Exception {
		JsTreeNodeModel node = null;
		// Ajoute les éléments à l'arbre
		final Iterator<InfosRubriques> listSousRubriquesIt = rubrique.getListeSousRubriques().iterator();
		InfosRubriques sousRubrique = null;
		final boolean selectable = isRubriqueSelectable(autorisations, rubrique, ids);
		if (isRubriqueVisible(autorisations, rubrique, ids, selectable)) {
			node = new JsTreeNodeModel();
			// on ajoute l'élément
			final JsTreeDataModel datas = new JsTreeDataModel();
			if (rubrique.getCode().equals(JsTreeUtils.CODE_ROOT)) {
				datas.setTitle(" ");
				node.getAttr().put("rel", "root");
				node.getAttr().put("class", "rubrique_root");
				node.setState("open");
			} else {
				final List<String> groups = new ArrayList<>();
				for (final String groupCode : rubrique.getGroupesDsi()) {
					final InfosGroupeDsi infos = Groupedsi.renvoyerItemGroupeDsi(groupCode);
					if (StringUtils.isNotBlank(infos.getCode())) {
						groups.add(infos.getIntitule());
					}
				}
				if (!selectable) {
					node.getAttr().put("rel", "not_selectable");
				} else {
					final InfosSite infos = Site.renvoyerItemSiteParRubrique(rubrique.getCode());
					if (infos != null) {
						node.getAttr().put("rel", "rubrique_site");
					}
				}
				final String restrained = StringUtils.join(groups, " ,");
				final StringBuffer cssClass = new StringBuffer(EscapeString.escapeAttributHtml("rubrique_" + rubrique.getTypeRubrique()));
				cssClass.append(rubrique.getGroupesDsi().size() > 0 ? " restrained" : "");
				final StringBuffer finalTitle = new StringBuffer(rubrique.getIntitule());
				finalTitle.append(StringUtils.isNotBlank(restrained) ? " (diffusion restreinte : " + restrained + " )" : "");
				datas.setTitle(rubrique.getIntitule());
				node.getAttr().put("class", cssClass.toString());
				node.getAttr().put("title", finalTitle.toString());
				node.getMetadata().put("libelle", rubrique.getIntitule());
				node.getMetadata().put("sCode", rubrique.getCode());
				node.getMetadata().put("idRubrique", rubrique.getIdRubrique().toString());
				node.getMetadata().put("sCodeMere", rubrique.getCodeRubriqueMere());
				node.getMetadata().put("libelleMere", Rubrique.getIntituleComplet(rubrique.getCode(), false));
				node.getMetadata().put("langue", rubrique.getLangue());
				if (rubrique.getListeSousRubriques().size() > 0 && hasVisibleChildren(permissions, autorisations, rubrique, ids)) {
					node.setState("closed");
				}
			}
			node.getMetadata().put("numchildren", Integer.toString(rubrique.getListeSousRubriques().size()));
			node.getAttr().put("id", Long.toString(rubrique.getIdRubrique()));
			node.setData(datas);
			if (niveau > 0 || niveau == -1) {
				while (listSousRubriquesIt.hasNext()) {
					sousRubrique = listSousRubriquesIt.next();
					if (niveau != -1) {
						niveau--;
					}
					final JsTreeNodeModel child = buildRubriqueNode(autorisations, permissions, sousRubrique, niveau, ids);
					if (child != null) {
						node.getChildren().add(child);
					}
					if (niveau != -1) {
						niveau++;
					}
				}
			} else if (niveau == -2) {
				while (listSousRubriquesIt.hasNext()) {
					JsTreeNodeModel child;
					sousRubrique = listSousRubriquesIt.next();
					if (!ids.contains(sousRubrique.getCode())) {
						child = buildRubriqueNode(autorisations, permissions, sousRubrique, 0, null);
					} else {
						child = buildRubriqueNode(autorisations, permissions, sousRubrique, niveau, ids);
					}
					if (child != null) {
						node.getChildren().add(child);
					}
				}
			}
		}
		return node;
	}

	private static boolean isRubriqueVisible(final AutorisationBean autorisations, final InfosRubriques rubrique, final Set<String> ids, final boolean selectable) {
		try {
			return JsTreeUtils.isRubriqueVisible(PERMISSION_MODIFICATION, autorisations, rubrique, selectable, ids) || JsTreeUtils.isRubriqueVisible(PERMISSION_SUPPRESSION,
				autorisations, rubrique, selectable, ids);
		} catch (final Exception e) {
			LOG.info("impossible de savoir si la rubrique est visible pour l'arbre du processus", e);
		}
		return Boolean.FALSE;
	}

	private static boolean isRubriqueSelectable(final AutorisationBean autorisations, final InfosRubriques rubrique, final Set<String> ids) {
		try {
			return JsTreeUtils.isRubriqueSelectable(PERMISSION_MODIFICATION, autorisations, rubrique, ids) || JsTreeUtils.isRubriqueSelectable(PERMISSION_SUPPRESSION,
				autorisations, rubrique, ids);
		} catch (final Exception e) {
			LOG.info("impossible de savoir si la rubrique est selectionnable pour l'arbre du processus", e);
		}
		return Boolean.FALSE;
	}
}
