/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.tree.processus;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.exception.ErreurAsyncException;
import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.kportal.core.config.MessageHelper;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Groupedsi;
import com.univ.objetspartages.om.InfosGroupeDsi;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.Perimetre;
import com.univ.objetspartages.om.PermissionBean;
import com.univ.objetspartages.om.Rubrique;
import com.univ.tree.bean.JsTreeDataModel;
import com.univ.tree.bean.JsTreeModel;
import com.univ.tree.bean.JsTreeNodeModel;
import com.univ.tree.bean.JsTreePath;
import com.univ.tree.utils.JsTreeUtils;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ContexteUtil;
import com.univ.utils.EscapeString;
import com.univ.utils.SessionUtil;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

public class RubriquesJsTree extends GestionJsTree<Rubrique> {

	/** l'id Spring du bean. */
	public static final String ID_BEAN = "rubriquesJsTree";

	private static Logger LOG = LoggerFactory.getLogger(RubriquesJsTree.class);

	@Override
	public JsTreeModel traiterDepuisRequete(final HttpServletRequest req) {
		final AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		HashMap<String, String> parameters;
		JsTreeModel jsTree = new JsTreeModel();
		try {
			parameters = JsTreeUtils.getParameters(req);
			assertParametersConsistency(parameters);
			final String permissions = parameters.get("PERMISSION");
			final InfosRubriques rubrique = Rubrique.renvoyerItemRubrique(parameters.get("RACINE"));
			final int niveau = Integer.parseInt(parameters.get("NIVEAU"));
			final String[] selection = StringUtils.defaultString(parameters.get("SELECTED")).split(";");
			final JsTreeNodeModel node = buildRubriqueNode(autorisations, permissions, rubrique, niveau, null);
			if (node != null) {
				jsTree.getNodes().add(node);
			}
			if (selection.length > 0) {
				loadToPath(autorisations, permissions, jsTree, selection);
			}
		} catch (final Exception e) {
			LOG.error("Impossible de generer l'arborescence des rubriques", e);
			jsTree = new JsTreeModel();
		}
		return jsTree;
	}

	private void loadToPath(final AutorisationBean autorisations, final String permissions, final JsTreeModel jsTree, final String[] selection) {
		final Set<String> loadedNodes = new HashSet<>();
		for (final String code : selection) {
			try {
				final JsTreePath path = getPathToCode(code);
				final Iterator<String> codeIt = path.getChildPath().iterator();
				final Set<String> nodes = new HashSet<>();
				String topCode = codeIt.next();
				nodes.addAll(path.getChildPath());
				if (!loadedNodes.isEmpty()) {
					while (loadedNodes.contains(topCode) && codeIt.hasNext()) {
						topCode = codeIt.next();
					}
				}
				if (StringUtils.isNotBlank(topCode)) {
					final JsTreeNodeModel parentNode = JsTreeUtils.getNodeWithCode(jsTree.getNodes().get(0), topCode);
					final InfosRubriques parent = Rubrique.renvoyerItemRubrique(topCode);
					if (parentNode != null && CollectionUtils.isEmpty(parentNode.getChildren())) {
						final JsTreeNodeModel nodeModel = buildRubriqueNode(autorisations, permissions, parent, -2, nodes);
						if (nodeModel != null && nodeModel.getChildren() != null) {
							parentNode.getChildren().addAll(nodeModel.getChildren());
						}
					}
					loadedNodes.addAll(nodes);
				}
			} catch (final Exception e) {
				LOG.error("An error occured trying to get OpenPath for rubrique tree", e);
			}
		}
	}

	private JsTreePath getPathToCode(final String code) throws Exception {
		final JsTreePath path = new JsTreePath();
		final InfosRubriques child = Rubrique.renvoyerItemRubrique(code);
		InfosRubriques parent = child.getRubriqueMere();
		path.addChild(code);
		while (parent != null && !parent.getCode().equals("00")) {
			path.addChild(parent.getCode());
			parent = parent.getRubriqueMere();
		}
		return path;
	}

	protected JsTreeNodeModel buildRubriqueNode(final AutorisationBean autorisations, final String permissions, final InfosRubriques rubrique, int niveau,
		final Set<String> ids) throws Exception {
		JsTreeNodeModel node = null;
		// Ajoute les éléments à l'arbre
		final Iterator<InfosRubriques> listSousRubriquesIt = rubrique.getListeSousRubriques().iterator();
		InfosRubriques sousRubrique = null;
		final boolean selectable = JsTreeUtils.isRubriqueSelectable(permissions, autorisations, rubrique, ids);
		if (JsTreeUtils.isRubriqueVisible(permissions, autorisations, rubrique, selectable, ids)) {
			node = new JsTreeNodeModel();
			// on ajoute l'élément
			final JsTreeDataModel datas = new JsTreeDataModel();
			if (rubrique.getCode().equals(JsTreeUtils.CODE_ROOT)) {
				datas.setTitle(" ");
				node.getAttr().put("rel", "root");
				node.getAttr().put("class", "rubrique_root");
				if(rubrique.getListeSousRubriques().size() > 0) {
					node.setState("open");
				}
			} else {
				final List<String> groups = new ArrayList<>();
				for (final String groupCode : rubrique.getGroupesDsi()) {
					final InfosGroupeDsi infos = Groupedsi.renvoyerItemGroupeDsi(groupCode);
					if (StringUtils.isNotBlank(infos.getCode())) {
						groups.add(infos.getIntitule());
					}
				}
				if (!selectable) {
					node.getAttr().put("rel", "not_selectable");
				} else {
					final InfosSite infos = Site.renvoyerItemSiteParRubrique(rubrique.getCode());
					if (infos != null) {
						node.getAttr().put("rel", "rubrique_site");
					}
				}
				final String restrained = StringUtils.join(groups, " ,");
				final StringBuffer cssClass = new StringBuffer(EscapeString.escapeAttributHtml("rubrique_" + rubrique.getTypeRubrique()));
				cssClass.append(rubrique.getGroupesDsi().size() > 0 ? " restrained" : "");
				final StringBuffer finalTitle = new StringBuffer(rubrique.getIntitule());
				finalTitle.append(StringUtils.isNotBlank(restrained) ? " (diffusion restreinte : " + restrained + " )" : "");
				datas.setTitle(rubrique.getIntitule());
				node.getAttr().put("class", cssClass.toString());
				node.getAttr().put("title", finalTitle.toString());
				node.getMetadata().put("libelle", rubrique.getIntitule());
				node.getMetadata().put("sCode", rubrique.getCode());
				node.getMetadata().put("idRubrique", rubrique.getIdRubrique().toString());
				node.getMetadata().put("sCodeMere", rubrique.getCodeRubriqueMere());
				node.getMetadata().put("libelleMere", Rubrique.getIntituleComplet(rubrique.getCode(), false));
				node.getMetadata().put("langue", rubrique.getLangue());
				if (rubrique.getListeSousRubriques().size() > 0 && hasVisibleChildren(permissions, autorisations, rubrique, ids)) {
					node.setState("closed");
				}
			}
			node.getMetadata().put("numchildren", Integer.toString(rubrique.getListeSousRubriques().size()));
			node.getAttr().put("id", Long.toString(rubrique.getIdRubrique()));
			node.setData(datas);
			if (niveau > 0 || niveau == -1) {
				while (listSousRubriquesIt.hasNext()) {
					sousRubrique = listSousRubriquesIt.next();
					if (niveau != -1) {
						niveau--;
					}
					final JsTreeNodeModel child = buildRubriqueNode(autorisations, permissions, sousRubrique, niveau, ids);
					if (child != null) {
						node.getChildren().add(child);
					}
					if (niveau != -1) {
						niveau++;
					}
				}
			} else if (niveau == -2) {
				while (listSousRubriquesIt.hasNext()) {
					JsTreeNodeModel child = null;
					sousRubrique = listSousRubriquesIt.next();
					if (!ids.contains(sousRubrique.getCode())) {
						child = buildRubriqueNode(autorisations, permissions, sousRubrique, 0, null);
					} else {
						child = buildRubriqueNode(autorisations, permissions, sousRubrique, niveau, ids);
					}
					if (child != null) {
						node.getChildren().add(child);
					}
				}
			}
		}
		return node;
	}

	protected static boolean hasVisibleChildren(final String permissions, final AutorisationBean autorisations, final InfosRubriques rubrique, final Set<String> ids) {
		for (final InfosRubriques currentChild : rubrique.getListeSousRubriques()) {
			boolean selectable = false;
			boolean visible = false;
			try {
				selectable = JsTreeUtils.isRubriqueSelectable(permissions, autorisations, currentChild, ids);
				visible = JsTreeUtils.isRubriqueVisible(permissions, autorisations, currentChild, selectable, ids);
				if (visible) {
					return true;
				}
			} catch (final Exception e) {
				LOG.error(String.format("An error occured trying to determine if rubrique \"%s\" has visible children", rubrique.getLibelleAffichable()), e);
			}
		}
		return false;
	}

	@Override
	public JsTreeModel traiterRechercheDepuisRequete(final HttpServletRequest req) {
		final AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		HashMap<String, String> parameters;
		JsTreeModel jsTree = new JsTreeModel();
		try {
			parameters = JsTreeUtils.getParameters(req);
			assertParametersConsistency(parameters);
			final Rubrique rubrique = new Rubrique();
			rubrique.setCtx(ContexteUtil.getContexteUniv());
			rubrique.select(req.getParameter("CODE_RECHERCHE"), req.getParameter("LANGUE"), req.getParameter("INTITULE"), req.getParameter("CODE_SAISI"));
			jsTree = filterTree(autorisations, rubrique, parameters);
		} catch (final Exception e) {
			LOG.error("impossible de generer l'arborescence des rubriques", e);
			jsTree = new JsTreeModel();
		}
		return jsTree;
	}

	@Override
	protected JsTreeModel filterTree(final AutorisationBean autorisations, final Rubrique rubrique, final HashMap<String, String> parameters) {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		JsTreeModel jsTree = new JsTreeModel();
		try {
			final String permissions = parameters.get("PERMISSION");
			final InfosRubriques rubriqueRacine = Rubrique.renvoyerItemRubrique(parameters.get("RACINE"));
			final int niveau = Integer.parseInt(parameters.get("NIVEAU"));
			final Set<String> ids = new HashSet<>();
			while (rubrique.nextItem()) {
				if (!rubrique.getCode().equals(parameters.get("CODE"))) {
					ids.add(rubrique.getCode());
					InfosRubriques rubriqueMere = Rubrique.renvoyerItemRubrique(rubrique.getCodeRubriqueMere());
					while (StringUtils.isNotBlank(rubriqueMere.getCode())) {
						ids.add(rubriqueMere.getCode());
						rubriqueMere = Rubrique.renvoyerItemRubrique(rubriqueMere.getCodeRubriqueMere());
					}
				}
			}
			if (ids.isEmpty()) {
				return jsTree;
			}
			ids.add("00");
			jsTree.getNodes().add(buildRubriqueNode(autorisations, permissions, rubriqueRacine, niveau, ids));
			return jsTree;
		} catch (final Exception e) {
			LOG.error("impossible de generer l'arborescence des rubriques", e);
			jsTree = new JsTreeModel();
		}
		return jsTree;
	}

	@Override
	public void assertParametersConsistency(final Map<String, String> parameters) throws Exception {
		if (parameters.get("RACINE") == null || parameters.get("RACINE").length() == 0) {
			parameters.put("RACINE", JsTreeUtils.CODE_ROOT);
		}
		if (parameters.get("CODE") == null || parameters.get("CODE").length() == 0) {
			parameters.put("CODE", parameters.get("RACINE"));
		}
		// Test si la rubrique de sélection est une sous-rubrique de la rubrique racine
		if (!parameters.get("RACINE").equals(JsTreeUtils.CODE_ROOT) && !parameters.get("RACINE").equals(parameters.get("CODE")) && Rubrique.renvoyerItemRubrique(
			parameters.get("CODE")).contains(Rubrique.renvoyerItemRubrique(parameters.get("RACINE")))) {
			parameters.put("RACINE", JsTreeUtils.CODE_ROOT);
		}
	}

	@Override
	public String traiterAction(final AutorisationBean autorisations, final Map<String, String[]> parametresDeLaRequete) {
		String action = StringUtils.EMPTY;
		String message = StringUtils.EMPTY;
		if (parametresDeLaRequete.get("ACTION") != null) {
			action = parametresDeLaRequete.get("ACTION")[0];
		}
		if (action.equals("SUPPRIMER")) {
			final String[] codes = parametresDeLaRequete.get("CODES_RUBRIQUES");
			if (codes != null) {
				for (final String currentCode : codes) {
					String libelleRubrique;
					try {
						final InfosRubriques infos = Rubrique.renvoyerItemRubrique(currentCode);
						libelleRubrique = infos.getLibelleAffichable();
					} catch (final Exception e) {
						libelleRubrique = currentCode;
					}
					message = String.format(MessageHelper.getCoreMessage("BO_SERVICES_ARBRE_SUPPRESSION_OK"), String.format(
						MessageHelper.getCoreMessage("BO_SERVICES_ARBRE_RUBRIQUE"), libelleRubrique));
					try {
						traiterSuppression(autorisations, currentCode);
					} catch (final Exception e) {
						LOG.error("la suppression de la rubrique a echouee", e);
						throw new ErreurAsyncException(e.getMessage());
					}
				}
			}
		} else if (action.equals("DEPLACER")) {
			final String[] codes = parametresDeLaRequete.get("CODES_RUBRIQUES[]");
			final String[] codesMere = parametresDeLaRequete.get("CODES_MERE[]") != null ? parametresDeLaRequete.get("CODES_MERE[]") : new String[codes.length];
			final String[] ordres = parametresDeLaRequete.get("ORDRES[]") != null ? parametresDeLaRequete.get("ORDRES[]") : new String[codes.length];
			if (codes != null) {
				for (int i = 0; i < codes.length; i++) {
					String libelleRubrique;
					try {
						final InfosRubriques infos = Rubrique.renvoyerItemRubrique(codes[i]);
						libelleRubrique = infos.getLibelleAffichable();
					} catch (final Exception e) {
						libelleRubrique = codes[i];
					}
					message = String.format(MessageHelper.getCoreMessage("BO_SERVICES_ARBRE_DEPLACEMENT_OK"), String.format(
						MessageHelper.getCoreMessage("BO_SERVICES_ARBRE_RUBRIQUE"), libelleRubrique));
					try {
						traiterDeplacement(autorisations, codes[i], codesMere[i], ordres[i]);
					} catch (final Exception e) {
						LOG.error("la suppression de la rubrique a echouee", e);
						throw new ErreurAsyncException(e.getMessage());
					}
				}
			}
		}
		return message;
	}

	private void traiterSuppression(final AutorisationBean autorisations, final String code) throws Exception {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final Rubrique rubrique = new Rubrique();
		rubrique.setCtx(ctx);
		rubrique.init();
		if (autorisations.possedePermission(new PermissionBean("TECH", "rub", "M"), new Perimetre("*", code, "*", "*", "")) == false) {
			throw new ErreurAsyncException(MessageHelper.getCoreMessage("BO_OPERATION_INTERDITE"));
		}
		if (rubrique.select(code, "", "", "") != 1) {
			throw new ErreurAsyncException(MessageHelper.getCoreMessage("BO_RUBRIQUE_INEXISTANTE"));
		}
		rubrique.nextItem();
		rubrique.setIdRubrique(rubrique.getIdRubrique());
		rubrique.delete();
		Rubrique.rechargement();
	}

	private void traiterDeplacement(final AutorisationBean autorisations, final String code, final String codeMere, final String ordre) throws Exception {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		final InfosRubriques infosRubrique = Rubrique.renvoyerItemRubrique(code);
		final Rubrique rubrique = new Rubrique();
		rubrique.setCtx(ctx);
		rubrique.init();
		rubrique.setIdRubrique(infosRubrique.getIdRubrique());
		rubrique.retrieve();
		if (autorisations.possedePermission(new PermissionBean("TECH", "rub", "M"), new Perimetre("*", code, "*", "*", "")) == false) {
			throw new ErreurAsyncException(MessageHelper.getCoreMessage("BO_OPERATION_INTERDITE"));
		}
		if (rubrique.select(code, "", "", "") != 1) {
			throw new ErreurAsyncException(MessageHelper.getCoreMessage("BO_RUBRIQUE_INEXISTANTE"));
		}
		rubrique.setIdRubrique(rubrique.getIdRubrique());
		rubrique.setCodeRubriqueMere(StringUtils.defaultString(codeMere, StringUtils.EMPTY));
		if (!StringUtils.isBlank(ordre)) {
			final InfosRubriques rubriqueMere = Rubrique.renvoyerItemRubrique(StringUtils.isBlank(codeMere) ? JsTreeUtils.CODE_ROOT : codeMere);
			int order = 0;
			for (final InfosRubriques currentRubrique : rubriqueMere.getListeSousRubriques()) {
				if (currentRubrique.getIdRubrique().compareTo(rubrique.getIdRubrique()) != 0) {
					if (order == Integer.parseInt(ordre)) {
						order++;
					}
					final Rubrique orderedRubrique = new Rubrique();
					orderedRubrique.setCtx(ctx);
					orderedRubrique.init();
					orderedRubrique.setIdRubrique(currentRubrique.getIdRubrique());
					orderedRubrique.retrieve();
					orderedRubrique.setOrdre(Integer.toString(order));
					orderedRubrique.update();
					order++;
				}
			}
		}
		rubrique.setOrdre(ordre);
		rubrique.update();
		Rubrique.rechargement();
	}

	public static String getPath(String root, final String code, final String separator) {
		final List<String> path = new ArrayList<String>();
		if (StringUtils.isBlank(root)) {
			root = JsTreeUtils.CODE_ROOT;
		}
		if (StringUtils.isBlank(code)) {
			return root;
		}
		try {
			InfosRubriques rubrique = Rubrique.renvoyerItemRubrique(code);
			while (StringUtils.isNotBlank(rubrique.getCode()) && !root.equals(rubrique.getCode())) {
				path.add(rubrique.getLibelleAffichable());
				rubrique = Rubrique.renvoyerItemRubrique(rubrique.getCodeRubriqueMere());
			}
		} catch (final Exception e) {
			LOG.error(String.format("An error occured trying to compute Rubrique tree path to element %s from %s", code, root), e);
		}
		Collections.reverse(path);
		return StringUtils.join(path, separator);
	}

	@Override
	public JsTreeModel traiterFiltreDepuisRequete(final HttpServletRequest req) {
		final AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		HashMap<String, String> parameters;
		JsTreeModel jsTree = new JsTreeModel();
		final String query = StringUtils.defaultString(req.getParameter("QUERY"), StringUtils.EMPTY);
		try {
			parameters = JsTreeUtils.getParameters(req);
			assertParametersConsistency(parameters);
			parameters.put("NIVEAU", "-1");
			final Rubrique rubriqueFiltrage = new Rubrique();
			rubriqueFiltrage.init();
			rubriqueFiltrage.setCtx(ContexteUtil.getContexteUniv());
			final ClauseWhere whereIntiluleLike = new ClauseWhere(ConditionHelper.like("T1.INTITULE", query, "%", "%"));
			rubriqueFiltrage.selectNoCount(whereIntiluleLike.formaterSQL());
			jsTree = filterTree(autorisations, rubriqueFiltrage, parameters);
			openAllNodes(jsTree.getNodes());
		} catch (final Exception e) {
			LOG.error(String.format("An error occured trying to filter Rubrique tree for query \"%s\"", query), e);
		}
		return jsTree;
	}

	@Override
	public String getSelectedIds(final String string) {
		if (StringUtils.isEmpty(string)) {
			return StringUtils.EMPTY;
		}
		final List<String> selectedIds = new ArrayList<String>();
		final String[] selections = string.split(";");
		for (final String currentSelection : selections) {
			try {
				final InfosRubriques rubrique = Rubrique.renvoyerItemRubrique(currentSelection);
				if (rubrique != null && StringUtils.isNotBlank(rubrique.getCode())) {
					selectedIds.add(Long.toString(rubrique.getIdRubrique()));
				}
			} catch (final Exception e) {
				LOG.error("An error occured trying to retrieve Rubrique", e);
			}
		}
		return StringUtils.join(selectedIds, ";");
	}
}
