/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.tree.processus;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.session.SessionUtilisateur;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.InfosStructure;
import com.univ.objetspartages.om.PermissionBean;
import com.univ.objetspartages.om.Structure;
import com.univ.tree.bean.JsTreeDataModel;
import com.univ.tree.bean.JsTreeModel;
import com.univ.tree.bean.JsTreeNodeModel;
import com.univ.tree.bean.JsTreePath;
import com.univ.tree.utils.JsTreeUtils;
import com.univ.utils.ContexteUtil;
import com.univ.utils.EscapeString;
import com.univ.utils.SessionUtil;

public class StructuresJsTree extends GestionJsTree<List<InfosStructure>> {

	/** l'id Spring du bean. */
	public static final String ID_BEAN = "structuresJsTree";

	private static final Logger LOG = LoggerFactory.getLogger(StructuresJsTree.class);

	@Override
	public JsTreeModel traiterDepuisRequete(HttpServletRequest req) {
		AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		HashMap<String, String> parameters;
		JsTreeModel jsTree = new JsTreeModel();
		try {
			parameters = JsTreeUtils.getParameters(req);
			assertParametersConsistency(parameters);
			boolean front = parameters.get("FRONT") != null && (parameters.get("FRONT").equals("true"));
			String permissions = parameters.get("PERMISSION");
			TypeStructureFilter filter = null;
			if (!StringUtils.isEmpty(parameters.get("FILTRE"))) {
				filter = new TypeStructureFilter(parameters.get("FILTRE"));
			}
			InfosStructure structure = Structure.renvoyerItemStructure(parameters.get("RACINE"));
			int niveau = Integer.parseInt(parameters.get("NIVEAU"));
			final String[] selection = StringUtils.defaultString(parameters.get("SELECTED")).split(";");
			jsTree.getNodes().add(buildStructureNode(permissions, autorisations, filter, structure, front, niveau, null));
			if (selection.length > 0) {
				loadToPath(autorisations, permissions, filter, front, jsTree, selection);
			}
		} catch (Exception e) {
			LOG.error("impossible de generer l'arborescence des structures", e);
			jsTree = new JsTreeModel();
		}
		return jsTree;
	}

	private void loadToPath(AutorisationBean autorisations, String permissions, TypeStructureFilter filter, boolean front, final JsTreeModel jsTree,
		String[] selection) {
		List<String> loadedNodes = new ArrayList<>();
		for (String code : selection) {
			try {
				final JsTreePath path = getPathToCode(code);
				final Iterator<String> codeIt = path.getChildPath().iterator();
				final Set<String> nodes = new HashSet<>();
				String topCode = codeIt.next();
				nodes.addAll(path.getChildPath());
				if (!loadedNodes.isEmpty()) {
					while (loadedNodes.contains(topCode) && codeIt.hasNext()) {
						topCode = codeIt.next();
					}
				}
				if (StringUtils.isNotBlank(topCode)) {
					JsTreeNodeModel parentNode = JsTreeUtils.getNodeWithCode(jsTree.getNodes().get(0), topCode);
					final InfosStructure parent = Structure.renvoyerItemStructure(topCode);
					parentNode.getChildren().addAll(buildStructureNode(permissions, autorisations, filter, parent, front, -2, nodes).getChildren());
					loadedNodes.addAll(nodes);
				}
			} catch (Exception e) {
				LOG.error("An error occured trying to get OpenPath for structure tree", e);
			}
		}
	}

	private JsTreePath getPathToCode(String code) {
		final JsTreePath path = new JsTreePath();
		final InfosStructure child = Structure.renvoyerItemStructure(code);
		InfosStructure parent = child.getStructureRattachement();
		path.addChild(code);
		while (parent != null && !parent.getCode().equals("00")) {
			path.addChild(parent.getCode());
			parent = parent.getStructureRattachement();
		}
		return path;
	}

	private static JsTreeNodeModel buildStructureNode(String permissions, AutorisationBean autorisations, TypeStructureFilter filter, InfosStructure structure,
		boolean front, int niveau, Set<String> ids) {
		JsTreeNodeModel node = null;
		// Ajoute les éléments à l'arbre
		Iterator<InfosStructure> listSousStructuresIt = structure.getListeSousStructuresSortedByLibelleLong().iterator();
		boolean selectable = isStructureSelectable(permissions, autorisations, filter, structure, ids);
		if (isStructureVisible(permissions, autorisations, filter, structure, front, selectable, ids)) {
			node = new JsTreeNodeModel();
			if (!selectable) {
				node.getAttr().put("rel", "not_selectable");
			}
			// on ajoute l'élément
			JsTreeDataModel datas = new JsTreeDataModel();
			if (structure.getCode().equals(JsTreeUtils.CODE_ROOT)) {
				datas.setTitle(" ");
				node.getAttr().put("rel", "root");
				node.getAttr().put("class", "structure_root");
				if(structure.getListeSousStructures().size() > 0) {
					node.setState("open");
				}
			} else {
				datas.setTitle(structure.getLibelleLong());
				node.getAttr().put("class", EscapeString.escapeAttributHtml("structure_" + structure.getCodeObjet() + "_" + structure.getTypeStructure()));
				node.getAttr().put("title", structure.getLibelleLong());
				node.getMetadata().put("libelle", structure.getLibelleLong());
				node.getMetadata().put("sCode", structure.getCode());
				if (structure.getListeSousStructures().size() > 0 && hasVisibleChildren(permissions, autorisations, structure, filter, front, ids)) {
					node.setState("closed");
				}
			}
			node.getMetadata().put("numchildren", Integer.toString(structure.getListeSousStructures().size()));
			node.getAttr().put("id", structure.getCodeObjet() + "_" + Long.toString(structure.getIdStructure()));
			node.setData(datas);
			if (niveau > 0 || niveau == -1) {
				while (listSousStructuresIt.hasNext()) {
					InfosStructure sousStructure = listSousStructuresIt.next();
					if (niveau != -1) {
						niveau--;
					}
					JsTreeNodeModel child = buildStructureNode(permissions, autorisations, filter, sousStructure, front, niveau, ids);
					if (child != null) {
						node.getChildren().add(child);
					}
					if (niveau != -1) {
						niveau++;
					}
				}
			} else if (niveau == -2) {
				while (listSousStructuresIt.hasNext()) {
					JsTreeNodeModel child = null;
					InfosStructure sousStructure = listSousStructuresIt.next();
					if (!ids.contains(sousStructure.getCode())) {
						child = buildStructureNode(permissions, autorisations, filter, sousStructure, front, 0, null);
					} else {
						child = buildStructureNode(permissions, autorisations, filter, sousStructure, front, niveau, ids);
					}
					if (child != null) {
						node.getChildren().add(child);
					}
				}
			}
		}
		return node;
	}

	@Override
	public void assertParametersConsistency(Map<String, String> parameters) throws Exception {
		if (parameters.get("CODE") == null || parameters.get("CODE").length() == 0) {
			parameters.put("CODE", JsTreeUtils.CODE_ROOT);
		}
		if (parameters.get("RACINE") == null || parameters.get("RACINE").length() == 0) {
			parameters.put("RACINE", JsTreeUtils.CODE_ROOT);
		}
	}

	/**
	 * Teste si la structure est sélectionnable.
	 * 
	 * @param structure
	 *            La structure à tester
	 * 
	 * @return true si la structure est sélectionnable
	 *
	 */
	private static boolean isStructureSelectable(String permissions, AutorisationBean autorisations, TypeStructureFilter filter, InfosStructure structure, Set<String> ids) {
		boolean selectableItem = true;
		boolean inIds = ids == null || ids.isEmpty() || ids.contains(structure.getCode());
		// Controle le périmètre de l'utilisateur sur la structure
		if (permissions.length() > 0) {
			selectableItem = Structure.controlerPermission(autorisations, permissions, structure.getCode());
		}
		// Controle le filtre sur les types de structures
		if (filter != null && selectableItem) {
			selectableItem = filter.accept(structure.getTypeStructure());
		}
		// si la fiche en cours de saisie est une fiche annuaire (ou étudiant ou ancien étudiant),
		// que l'utilisateur est redacteur de sa fiche,
		// et qu'il n'a pas de role concernant les fiches annuaire,
		// il peut choisir la structure
		PermissionBean permissionCourante = new PermissionBean(permissions);
		if (!selectableItem && autorisations != null && autorisations.isRedacteurFicheCourante() && ("0006".equals(permissionCourante.getObjet()) || "0007".equals(permissionCourante.getObjet()) || "0013".equals(permissionCourante.getObjet())) && !autorisations.possedePermission(permissionCourante)) {
			selectableItem = true;
		}
		return selectableItem && inIds;
	}

	private static boolean hasVisibleChildren(String permissions, AutorisationBean autorisations, InfosStructure structure, TypeStructureFilter filter, boolean front,
		Set<String> ids) {
		for (InfosStructure currentChild : structure.getListeSousStructures()) {
			boolean selectable = false;
			boolean visible = false;
			try {
				selectable = isStructureSelectable(permissions, autorisations, filter, currentChild, ids);
				visible = isStructureVisible(permissions, autorisations, filter, structure, front, selectable, ids);
				if (visible) {
					return true;
				}
			} catch (Exception e) {
				LOG.error(String.format("An error occured determining children's visibility for structure %s", structure.getLibelleAffichable()), e);
			}
		}
		return false;
	}

	/**
	 * Teste si la structure est visible dans l'arbre (structure sélectionnable ou ayant un fils visible.
	 * 
	 * @param structure
	 *            La structure à tester
	 * @param selectable
	 *            true si la structure est sélectionnable
	 * 
	 * @return true si la structure est visible dans l'arbre
	 *
	 */
	private static boolean isStructureVisible(String permissions, AutorisationBean autorisations, TypeStructureFilter filter, InfosStructure structure, boolean front,
		boolean selectable, Set<String> ids) {
		boolean visible = selectable;
		boolean inIds = ids == null || ids.isEmpty() || ids.contains(structure.getCode());
		if (!selectable) // structure non sélectionnable
		{
			// on vérifie ses structures filles
			Iterator<InfosStructure> listSousStructuresIt = structure.getListeSousStructures().iterator();
			InfosStructure sousStructure = null;
			while (listSousStructuresIt.hasNext() && !visible) {
				sousStructure = listSousStructuresIt.next();
				// Si on est en front, vérifie que la structure est bien visible
				if (!front || sousStructure.isVisibleInFront()) {
					visible = isStructureVisible(permissions, autorisations, filter, sousStructure, front, isStructureSelectable(permissions, autorisations, filter, sousStructure,
						ids), ids);
				}
			}
		}
		return visible && inIds;
	}

	@Override
	public JsTreeModel traiterRechercheDepuisRequete(HttpServletRequest req) {
		return new JsTreeModel();
	}

	@Override
	public JsTreeModel traiterFiltreDepuisRequete(HttpServletRequest req) {
		AutorisationBean autorisations = (AutorisationBean) SessionUtil.getInfosSession(req).get(SessionUtilisateur.AUTORISATIONS);
		HashMap<String, String> parameters;
		JsTreeModel jsTree = new JsTreeModel();
		final String query = StringUtils.defaultString(req.getParameter("QUERY"), StringUtils.EMPTY);
		try {
			parameters = JsTreeUtils.getParameters(req);
			assertParametersConsistency(parameters);
			parameters.put("NIVEAU", "-1");
			Map<String, String> structures = Structure.getListeStructureParIntituleComplet(ContexteUtil.getContexteUniv());
			List<InfosStructure> listStructures = new ArrayList<>();
			for (Entry<String, String> structure : structures.entrySet()) {
				if (StringUtils.containsIgnoreCase(structure.getValue(), query)) {
					listStructures.add(Structure.renvoyerItemStructure(structure.getKey()));
				}
			}
			jsTree = filterTree(autorisations, listStructures, parameters);
			openAllNodes(jsTree.getNodes());
		} catch (Exception e) {
			LOG.error(String.format("An error occured trying to filter Rubrique tree for query \"%s\"", query), e);
		}
		return jsTree;
	}

	public static String getPath(String root, String code, String separator) {
		List<String> path = new ArrayList<>();
		if (StringUtils.isBlank(root)) {
			root = JsTreeUtils.CODE_ROOT;
		}
		if (StringUtils.isBlank(code)) {
			return root;
		}
		InfosStructure structure = Structure.renvoyerItemStructure(code);
		while (StringUtils.isNotBlank(structure.getCode()) && !root.equals(structure.getCode())) {
			path.add(structure.getLibelleAffichable());
			structure = structure.getStructureRattachement();
		}
		Collections.reverse(path);
		return StringUtils.join(path, separator);
	}

	@Override
	protected JsTreeModel filterTree(AutorisationBean autorisations, List<InfosStructure> structures, HashMap<String, String> parameters) {
		JsTreeModel jsTree = new JsTreeModel();
		String permissions = parameters.get("PERMISSION");
		InfosStructure structure = Structure.renvoyerItemStructure(parameters.get("RACINE"));
		int niveau = Integer.parseInt(parameters.get("NIVEAU"));
		final Set<String> ids = new HashSet<>();
		final boolean front = parameters.get("FRONT") != null && (parameters.get("FRONT").equals("true"));
		TypeStructureFilter filter = null;
		if (!StringUtils.isEmpty(parameters.get("FILTRE"))) {
			filter = new TypeStructureFilter(parameters.get("FILTRE"));
		}
		for (InfosStructure currentStructure : structures) {
			ids.add(currentStructure.getCode());
			InfosStructure structureMere = Structure.renvoyerItemStructure(currentStructure.getCodeRattachement());
			while (StringUtils.isNotBlank(structureMere.getCode())) {
				ids.add(structureMere.getCode());
				structureMere = Structure.renvoyerItemStructure(structureMere.getCodeRattachement());
			}
		}
		if (ids.isEmpty()) {
			return jsTree;
		}
		ids.add("00");
		jsTree.getNodes().add(buildStructureNode(permissions, autorisations, filter, structure, front, niveau, ids));
		return jsTree;
	}

	@Override
	public String getSelectedIds(String string) {
		if (StringUtils.isEmpty(string)) {
			return StringUtils.EMPTY;
		}
		final List<String> selectedIds = new ArrayList<>();
		final String[] selections = string.split(";");
		for (String currentSelection : selections) {
			try {
				InfosStructure structure = Structure.renvoyerItemStructure(currentSelection);
				if (structure != null && StringUtils.isNotBlank(structure.getCode())) {
					selectedIds.add(structure.getCodeObjet() + "_" + Long.toString(structure.getIdStructure()));
				}
			} catch (Exception e) {
				LOG.error("An error occured trying to retrieve Rubrique", e);
			}
		}
		return StringUtils.join(selectedIds, ";");
	}
}

/**
 * Classe implémentant un filtre sur les types de structure.
 */
class TypeStructureFilter {

	/** Liste des types de structures autorisés. */
	private String authorizedTypeStructureList = null;

	/**
	 * Constructeur.
	 * 
	 * @typeStructureList Liste des types de structures autorisés (séparation des types de structure par des ';')
	 */
	TypeStructureFilter(String typeStructureList) {
		this.authorizedTypeStructureList = typeStructureList;
	}

	/**
	 * Teste si le type de structure est accepté.
	 * 
	 * @param typeStructure
	 *            Le type de structure à tester
	 * @return true si le type est accepté
	 */
	boolean accept(String typeStructure) {
		return typeStructure == null || typeStructure.length() == 0 || authorizedTypeStructureList.contains(typeStructure);
	}
}
