/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.url;

import java.util.ArrayList;
import java.util.Hashtable;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.LangueUtil;
import com.jsbsoft.jtf.database.OMContext;
import com.kdecole.cache.aspect.TempCacheFlush;
import com.kdecole.cache.aspect.TempCacheGet;
import com.kportal.cache.AbstractCacheManager;
import com.kportal.extension.module.plugin.rubrique.BeanPageAccueil;
import com.kportal.extension.module.plugin.rubrique.FichePageAccueilRubrique.BeanFichePageAccueil;
import com.kportal.extension.module.plugin.rubrique.PageAccueilRubriqueManager;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.Rubrique;
import com.univ.objetspartages.om.Rubriquepublication;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteDao;
import com.univ.utils.ContexteUniv;
import com.univ.utils.URLResolver;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;

@Component
public class CacheUrlManager extends AbstractCacheManager {

	public static final String ID_BEAN = "cacheUrlManager";

	private static final Logger LOGGER = LoggerFactory.getLogger(CacheUrlRubrique.class);

	public static final String KEY_CACHE = "CacheUrlManager.cacheUrlRubrique";

	public static CacheUrlManager getInstance() {
		return (CacheUrlManager) ApplicationContextManager.getCoreContextBean(ID_BEAN);
	}

	/**
	 * Initialiser url navigation.
	 *
	 */
	private CacheUrlRubrique cacheUrlRubrique() {
		final CacheUrlRubrique cacheUrlRubrique = new CacheUrlRubrique();
		final Hashtable<String, String> lstUrlNavigation = cacheUrlRubrique.getLstUrlNavigation();
		final Hashtable<String, String> lstNavigationUrl = cacheUrlRubrique.getLstNavigationUrl();
		for (final InfosSite infoSite : Site.getListeInfosSites().values()) {
			final int niveauMini = infoSite.getNiveauMinReecritureRubrique();
			final int niveauMaxi = infoSite.getNiveauMaxReecritureRubrique();
			final String codeRubriqueSite = infoSite.getCodeRubrique();
			InfosRubriques rubriqueSite;
			if (StringUtils.isEmpty(codeRubriqueSite)) {
				rubriqueSite = Rubrique.getTopLevelRubrique();
			} else {
				rubriqueSite = Rubrique.renvoyerItemRubrique(codeRubriqueSite);
			}
			final String urlSite = URLResolver.getAbsoluteUrl("/", infoSite, 0);
			//on ne gère en URL friendy que les sites rubriqués
			if (StringUtils.isNotEmpty(codeRubriqueSite)) {
				final String idNavigation = infoSite.getAlias() + ";" + codeRubriqueSite;
				lstUrlNavigation.put(idNavigation, urlSite);
				lstNavigationUrl.put(urlSite, idNavigation);
			}
			// boucle sur toutes les rubriques du site (tout niveau confondu)
			for (final InfosRubriques infosRubCourante : rubriqueSite.getListeSousRubriquesTousNiveaux()) {
				final StringBuilder urlRubrique = new StringBuilder();
				final String codeRubrique = infosRubCourante.getCode();
				InfosRubriques infosRubNiveau = infosRubCourante;
				if (infoSite.getModeReecritureRubrique() != 0) {
					int niveauRubrique = infosRubNiveau.getNiveau();
					if (niveauRubrique <= niveauMaxi) {
						while (niveauRubrique != 0 && niveauRubrique >= niveauMini) {
							if (niveauRubrique <= niveauMaxi) {
								if (urlRubrique.length() > 0) {
									urlRubrique.insert(0, "/");
								}
								urlRubrique.insert(0, Chaine.formatString(infosRubNiveau.getLibelleAffichable()));
							}
							infosRubNiveau = infosRubNiveau.getRubriqueMere();
							niveauRubrique = infosRubNiveau.getNiveau();
						}
					}
				} else {
					urlRubrique.append(Chaine.formatString(infosRubCourante.getLibelleAffichable()));
				}
				final String idNavigation = infoSite.getAlias() + ";" + codeRubrique;
				final BeanPageAccueil beanAccueil = PageAccueilRubriqueManager.getInstance().getBeanPageAccueil(infosRubCourante);
				//si la rubrique n'a pas de page d'acceuil, on renvoie l'url à vide
				if (beanAccueil == null) {
					lstUrlNavigation.put(idNavigation, "");
				} else if (infoSite.getModeReecritureRubrique() != 0 || !(beanAccueil instanceof BeanFichePageAccueil)) {
					//gestion de l'unicité
					if (urlRubrique.length() == 0) {
						urlRubrique.append(Chaine.formatString(infosRubCourante.getLibelleAffichable()));
					}
					while (lstNavigationUrl.get(urlSite + urlRubrique.toString() + "/") != null) {
						if (urlRubrique.toString().endsWith("-")) {
							urlRubrique.append("+");
						} else {
							urlRubrique.append("-");
						}
					}
					urlRubrique.insert(0, urlSite).append("/");
					lstUrlNavigation.put(idNavigation, urlRubrique.toString());
					lstNavigationUrl.put(urlRubrique.toString(), idNavigation);
				}
			}
		}
		LOGGER.info("Chargement de " + lstNavigationUrl.size() + " urls de rubrique OK");
		return cacheUrlRubrique;
	}

	@TempCacheGet(key = "CacheUrlManager.getAbsoluteUrlFriendly", contextual = false)
	public String getAbsoluteUrlFriendly(final String key, final OMContext ctx, final Metatag metatag, final UrlCacheRequest ucr) throws Exception {
		if (metatag == null) {
			return getAbsoluteUrlFriendly(ctx, ucr);
		} else {
			final InfosSite infosSite = Site.renvoyerItemSite(ucr.getAlias());
			return URLResolver.getAbsoluteUrl(UrlManager.calculerUrlFiche(infosSite, Rubrique.renvoyerItemRubrique(ucr.getCodeRubrique()), metatag), ctx, infosSite);
		}
	}

	@TempCacheFlush(key = "CacheUrlManager.getAbsoluteUrlFriendly", contextual = false)
	public void refreshUrl(final String key) {
		// nothing
	}

	/**
	 * Calcule les url friendly.
	 *
	 * @param ctx
	 *            le contexte courant pour connaitre le site etc.
	 * @param ucr
	 *            le cache de la request
	 *
	 * @return l'url absolu de la fiche/rubrique
	 *
	 * @throws Exception
	 *             the exception
	 */
	private String getAbsoluteUrlFriendly(final OMContext ctx, final UrlCacheRequest ucr) throws Exception {
		final String codeRubriqueCourante = ucr.getCodeRubrique();
		final String codeRubriqueAffichage = ucr.getCodeRubriqueAffichage();
		final String codeObjet = ucr.getCodeObjet();
		final String codeFiche = ucr.getCodeFiche();
		String langueFiche = ucr.getLangueFiche();
		String etatFiche = ucr.getEtatFiche();
		if (etatFiche == null || etatFiche.length() == 0) {
			etatFiche = "0003";
		}
		if (langueFiche == null || langueFiche.length() == 0) {
			if (ctx instanceof ContexteUniv) {
				langueFiche = ((ContexteUniv) ctx).getLangue();
			} else {
				langueFiche = "0";
			}
		}
		boolean chargementFiche = false;
		final Metatag meta = new Metatag();
		meta.init();
		try (ContexteDao ctxDao = new ContexteDao()) {
			meta.setCtx(ctxDao);
			ClauseWhere whereMeta = whereMetaCodeObjetCodeLangueEtat(codeObjet, codeFiche, langueFiche, etatFiche);
			if (meta.select(whereMeta.formaterSQL()) > 0) {
				chargementFiche = true;
			} else {
				// Si la fiche n'est pas présente dans une langue étrangère,
				// on charge la fiche en francais
				if (!ctx.getLocale().equals(LangueUtil.getDefaultLocale())) {
					whereMeta = whereMetaCodeObjetCodeLangueEtat(codeObjet, codeFiche, LangueUtil.getLangueLocale(LangueUtil.getDefaultLocale()), etatFiche);
					if (meta.select(whereMeta.formaterSQL()) > 0) {
						chargementFiche = true;
					}
				}
			}
			if (!chargementFiche) {
				// on cherche l'archive éventuelle
				whereMeta = whereMetaCodeObjetCodeLangueEtat(codeObjet, codeFiche, langueFiche, "0007");
				if (meta.select(whereMeta.formaterSQL()) > 0) {
					chargementFiche = true;
				}
			}
			if (!chargementFiche) {
				return StringUtils.EMPTY;
			}
			meta.nextItem();
		}
		//on constitue les rubriques ou la fiche peut être publiée
		final ArrayList<String> lstCodeRubriquePubliable = new ArrayList<>();
		if (meta.getMetaCodeRubrique().length() > 0) {
			lstCodeRubriquePubliable.add(meta.getMetaCodeRubrique());
		}
		final Rubriquepublication rp = new Rubriquepublication();
		rp.init();
		try (ContexteDao ctxDao = new ContexteDao()) {
			rp.setCtx(ctxDao);
			if (rp.select(codeObjet, codeFiche, langueFiche) > 0) {
				while (rp.nextItem()) {
					lstCodeRubriquePubliable.add(rp.getRubriqueDest());
				}
			}
		}
		//si la fiche est publiable dans le site courant
		//si la fiche est publiable dans la rubrique courante
		//on calcule l'url avec le host courant et la rubrique courante
		//sinon
		//si la fiche est rattachée ou publiée dans le site courant
		//on calcule l'url avec le host courant et la rubrique de publication de la fiche dans le site courant
		//sinon
		//on calcule l'url avec le host courant et la rubrique courante
		//sinon
		//si la fiche est rattachée à une rubrique appartenant à un site
		//on calcule l'url avec la rubrique de la fiche et le host principal du site de la rubrique
		//sinon
		//si la fiche est publiée dans des rubriques appartenant à un site
		//on choisit la rubrique avec en priorité - 1 : site principal - 2 : sites non restreint - 3 : sites restreints
		//sinon
		//on calcule avec le host courant et la rubrique courante
		//si la fiche est publiable dans le site courant avec une priorité pour la rubrique incluse dans la rubrique courante
		String codeRubriqueAffichageTemp;
		if (StringUtils.isNotBlank(codeRubriqueAffichage)) {
			codeRubriqueAffichageTemp = codeRubriqueAffichage;
		} else {
			codeRubriqueAffichageTemp = Rubrique.getRubriquePublication(ctx, Rubrique.renvoyerItemRubrique(codeRubriqueCourante), lstCodeRubriquePubliable, true);
			if (codeRubriqueAffichageTemp == null) {
				// si la fiche est rattachée à une rubrique appartenant à un site
				for (String codeRubriqueTemp : lstCodeRubriquePubliable) {
					if (Site.determinerSiteRubrique(codeRubriqueTemp, false) != null) {
						codeRubriqueAffichageTemp = codeRubriqueTemp;
						break;
					}
				}
			}
			if (codeRubriqueAffichageTemp == null) {
				if (Site.getSitePrincipal() != null) {
					codeRubriqueAffichageTemp = Site.getSitePrincipal().getCodeRubrique();
				} else {
					codeRubriqueAffichageTemp = codeRubriqueCourante;
				}
			}
		}
		final InfosSite site = Site.determinerSiteRubrique(codeRubriqueAffichageTemp);
		final InfosRubriques rubrique = Rubrique.renvoyerItemRubrique(codeRubriqueAffichageTemp);
		return URLResolver.getAbsoluteUrl(UrlManager.calculerUrlFiche(site, rubrique, meta), ctx, site);
	}

	private static ClauseWhere whereMetaCodeObjetCodeLangueEtat(final String codeObjet, final String code, final String langue, final String etat) {
		final ClauseWhere whereCodeObjetCodeLangueEtat = new ClauseWhere(ConditionHelper.egalVarchar("META_CODE_OBJET", codeObjet));
		whereCodeObjetCodeLangueEtat.and(ConditionHelper.egalVarchar("META_CODE", code));
		whereCodeObjetCodeLangueEtat.and(ConditionHelper.egalVarchar("META_LANGUE", langue));
		whereCodeObjetCodeLangueEtat.and(ConditionHelper.egalVarchar("META_ETAT_OBJET", etat));
		return whereCodeObjetCodeLangueEtat;
	}

	@Override
	public Object getObjectToCache() throws Exception {
		return cacheUrlRubrique();
	}

	@Override
	public String getCacheName() {
		return KEY_CACHE;
	}

	@Override
	public Object getObjectKey() {
		return KEY_CACHE;
	}
}
