/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.url;

import java.net.URLEncoder;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.lang.CharEncoding;
import com.kportal.core.config.PropertyHelper;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.Rubrique;
import com.univ.utils.Chaine;
import com.univ.utils.ContexteUniv;
import com.univ.utils.ServicesUtil;
import com.univ.utils.URLResolver;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.operande.TypeOperande;

public class UrlManager {

	/**
	 * Calculer url fiche.
	 *
	 * @param idMeta
	 *            the id meta
	 *
	 * @return the string
	 */
	public static String calculerUrlApercu(final String idMeta) {
		return calculerUrl(idMeta, "apercu");
	}

	/**
	 * Calculer url fiche.
	 *
	 * @param idMeta
	 *            the id meta
	 *
	 * @return the string
	 */
	public static String calculerUrlPage(final String idMeta) {
		return calculerUrl(idMeta, "page");
	}

	/**
	 * Calculer url fiche.
	 *
	 * @param idMeta
	 *            the id meta
	 * @param libelle
	 *            the libelle
	 *
	 * @return the string
	 */
	public static String calculerUrl(final String idMeta, final String libelle) {
		String extension = PropertyHelper.getCoreProperty("application.content_extension");
		if (extension == null || extension.length() == 0) {
			extension = "htm";
		}
		String url = "/";
		url += libelle;
		url += "-";
		url += idMeta;
		url += ".";
		url += extension;
		return url;
	}

	/**
	 * Calculer url fiche.
	 *
	 * @param ctx
	 *            the ctx
	 * @param codeObjet
	 *            the code objet
	 * @param idFiche
	 *            the id fiche
	 *
	 * @return the string
	 */
	public static String calculerUrlFiche(final OMContext ctx, final String codeObjet, final Long idFiche) {
		if (StringUtils.isEmpty(codeObjet) || codeObjet.equals("0010")) {
			return StringUtils.EMPTY;
		}
		final Metatag metatag = new Metatag();
		metatag.init();
		metatag.setCtx(ctx);
		try {
			final ClauseWhere whereMetaCodeObjetEtIdFiche = new ClauseWhere(ConditionHelper.egalVarchar("META_CODE_OBJET", codeObjet));
			whereMetaCodeObjetEtIdFiche.and(ConditionHelper.egal("META_ID_FICHE", idFiche, TypeOperande.LONG));
			metatag.select(whereMetaCodeObjetEtIdFiche.formaterSQL());
			if (!metatag.nextItem()) {
				return StringUtils.EMPTY;
			}
		} catch (final Exception e) {
			return StringUtils.EMPTY;
		}
		return calculerUrlFiche(Site.determinerSiteRubrique(metatag.getMetaCodeRubrique()), Rubrique.renvoyerItemRubrique(metatag.getMetaCodeRubrique()), metatag);
	}

	/**
	 * Calculer url fiche.
	 *
	 * @param idMeta
	 *            the id meta
	 * @param codeRubrique
	 *            the code rubrique
	 * @param libelle
	 *            the libelle
	 *
	 * @return the string
	 */
	public static String calculerUrlFiche(final InfosSite site, final InfosRubriques rubrique, final Metatag meta) {
		String urlFiche = Chaine.formatString(meta.getMetaLibelleFiche());
		if (site != null && site.getModeReecritureRubrique() != 0) {
			final String urlRubrique = FrontOfficeMgr.getInstance().getUrlAccueilRubrique(site, rubrique.getCode());
			if (StringUtils.isNotEmpty(urlRubrique)) {
				urlFiche = String.format("%s%s", urlRubrique, urlFiche);
				urlFiche = StringUtils.substringAfter(urlFiche, URLResolver.getAbsoluteUrl("/", site, 0));
			}
		}
		return calculerUrl(meta.getIdMetatag().toString(), urlFiche);
	}

	/**
	 * Calculer url fiche.
	 *
	 * @param idMeta
	 *            the id meta
	 * @param ctx
	 *            the ctx
	 *
	 * @return the string
	 */
	public static String calculerUrlFiche(final OMContext ctx, final Long idMeta) {
		final Metatag metatag = new Metatag();
		metatag.init();
		metatag.setCtx(ctx);
		metatag.setIdMetatag(idMeta);
		try {
			metatag.retrieve();
			if (metatag.getMetaCodeObjet().equals("0010")) {
				return "";
			}
		} catch (final Exception e) {
			return "";
		}
		return calculerUrlFiche(Site.determinerSiteRubrique(metatag.getMetaCodeRubrique()), Rubrique.renvoyerItemRubrique(metatag.getMetaCodeRubrique()), metatag);
	}

	/**
	 * Calculer url fiche.
	 *
	 * @param metatag
	 *            the metatag
	 *
	 * @return the string
	 */
	public static String calculerUrlFiche(final Metatag metatag) {
		if (metatag.getMetaCodeObjet().equals("0010")) {
			return "";
		}
		return calculerUrlFiche(Site.determinerSiteRubrique(metatag.getMetaCodeRubrique()), Rubrique.renvoyerItemRubrique(metatag.getMetaCodeRubrique()), metatag);
	}

	/**
	 * Calculer url forward sso.
	 *
	 * @param ctx
	 *            the ctx
	 * @param urlForward
	 *            url cible une fois l'indentification propagée
	 *
	 * @return l'url
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String calculerUrlForward(final ContexteUniv ctx, final String urlForward) throws Exception {
		// forward.html?kticket=xxxxxx&FWD=urlForward
		final String redirect = URLEncoder.encode(urlForward, CharEncoding.DEFAULT);
		String extension = PropertyHelper.getCoreProperty("application.content_extension");
		if (extension == null || extension.length() == 0) {
			extension = "htm";
		}
		return "/forward." + extension + "?kticket=" + ServicesUtil.genererTicketService(ctx) + "&FWD=" + redirect;
	}
}
