/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils;

import java.util.Locale;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.LangueUtil;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.Rubrique;

/**
 * classe stockant un Contexte dans un ThreadLocal afin de ne pas à avoir à en instancier un à chaque fois ou de le passer dans toute les entêtes de méthodes
 *
 * @author olivier.camon
 *
 */
public abstract class ContexteUtil {

	private static final Logger LOGGER = LoggerFactory.getLogger(ContexteUtil.class);

	private static final ThreadLocal<ContexteUniv> threadLocal = new ThreadLocal<ContexteUniv>();

	/**
	 * Initialise un contexte a partir de la request et de la réponse A la différence des autres méthodes, le contexte est stocké dans un thread local permettant de le récupérer
	 * sans en instancié de nouveau. Cette méthode doit être appelé une seule fois dans un filtre qui se chargera aussi de le releaser afin de centraliser la gestion des contextes.
	 *
	 * @param request
	 * @param response
	 * @return
	 * @throws Exception
	 */
	public static ContexteUniv setContexteUniv(final HttpServletRequest request, final HttpServletResponse response, final ServletContext servletContext) {
		final ContexteUniv ctx = new ContexteUniv(request);
		ctx.setReponseHTTP(response);
		try {
			ctx.initialiserInfosUtilisateur();
		} catch (final Exception e) {
			LOGGER.error("impossible d'initialiser les infos utilisateurs", e);
		}
		try {
			ctx.initialiserInfosConnecteurs();
		} catch (final Exception e) {
			LOGGER.error("impossible d'initialiser les infos des connecteurs", e);
		}
		ctx.setEspaceCourant(ctx.getEspaceHistorique());
		ctx.setServletContext(servletContext);
		//Initialisation de la langue à partir de la variable LOCALE de l'URL
		initLangueUtilisateur(request, ctx);
		ajoutDansThreadLocal(ctx);
		return ctx;
	}

	private static void initLangueUtilisateur(final HttpServletRequest request, final ContexteUniv ctx) {
		String codeRubrique = StringUtils.EMPTY;
		if (ctx.getRequeteHTTP() != null) {
			codeRubrique = ctx.getRequeteHTTP().getParameter("RF");
		}
		final InfosRubriques rubriqueCourante = Rubrique.renvoyerItemRubrique(StringUtils.defaultString(codeRubrique, ctx.getCodeRubriqueHistorique()));
		final Locale localeNavigateur = request.getLocale();
		if (StringUtils.isNotBlank(rubriqueCourante.getCode())) {
			ctx.setLangue(rubriqueCourante.getLangue());
			ctx.setLocale(LangueUtil.getLocale(rubriqueCourante.getLangue()));
		} else {
			final Locale effectiveLocale = LangueUtil.getEffectiveLocale(localeNavigateur);
			ctx.setLocale(effectiveLocale);
			ctx.setLangue(LangueUtil.getLangueLocale(effectiveLocale));
		}
	}

	/**
	 * Initialise un contexteuniv lorsque l'on a pas de requête.
	 *
	 * @return
	 */
	public static ContexteUniv setContexteSansRequete() {
		final ContexteUniv ctx = new ContexteUniv("");
		ajoutDansThreadLocal(ctx);
		return ctx;
	}

	private static void ajoutDansThreadLocal(final ContexteUniv ctx) {
		final ContexteUniv oldCtx = getContexteUniv();
		if (oldCtx != null) {
			oldCtx.release();
			LOGGER.warn("Initialisation d'un contexte univ alors qu'il en existe déjà un");
		}
		threadLocal.set(ctx);
	}

	/**
	 * se charge de récupere le contexte provenant du THreadLocal,
	 *
	 * @return le contexte initialisé dans le filtre
	 */
	public static ContexteUniv getContexteUniv() {
		return threadLocal.get();
	}

	/**
	 * Release le contexte et le supprime de ThreadLocal, Cette méthode ne doit être appelé que dans le filtre afin de ne pas avoir plusieurs release du même contexte.
	 */
	public static void releaseContexteUniv() {
		final ContexteUniv ctx = getContexteUniv();
		if (ctx != null) {
			ctx.release();
			threadLocal.remove();
		} else {
			LOGGER.warn("contexte déjà libéré");
		}
	}
}
