/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils;

import java.sql.Date;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

// TODO: Auto-generated Javadoc
/**
 * Classe permettant de manipuler des dates.
 * 
 * @author fred
 */
public class DateUtil {

	/** Formatteur de date : format dd/MM/yyyy. */
	private final static SimpleDateFormat sdf = new SimpleDateFormat("dd/MM/yyyy");

	private static final Logger LOG = LoggerFactory.getLogger(DateUtil.class);

	/**
	 * Gets the simple date format.
	 * 
	 * @return the simple date format
	 */
	public static SimpleDateFormat getSimpleDateFormat() {
		return (SimpleDateFormat) sdf.clone();
	}

	/**
	 * Parse une date au format dd/MM/yyyy.
	 * 
	 * @param sDate
	 *            La chaine à parser
	 * 
	 * @return La date demandée
	 */
	public static Date parseDate(final String sDate) {
		Date date = null;
		if (StringUtils.isNotBlank(sDate)) {
			try {
				date = new Date(getSimpleDateFormat().parse(sDate).getTime());
			} catch (final ParseException e) {
				LOG.info("date incorrecte : " + sDate);
			}
		}
		return date != null ? date : new Date(0);
	}

	/**
	 * Formate une date au format dd/MM/yyyy.
	 * 
	 * @param date
	 *            La date à formater
	 * 
	 * @return La chaine formatée
	 */
	public static String formatDate(final Date date) {
		return date == null ? "" : getSimpleDateFormat().format(date);
	}

	/**
	 * Renvoie le premier jour de la semaine pour la date passée en paramètre.
	 * 
	 * @param date
	 *            La date de départ
	 * 
	 * @return La date demandée
	 */
	public static Date getFirstDayOfWeek(final Date date) {
		final GregorianCalendar cal = new GregorianCalendar();
		cal.setTime(date);
		cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY); // lundi
		return new Date(cal.getTimeInMillis());
	}

	/**
	 * Renvoie le dernier jour de la semaine pour la date passée en paramètre.
	 * 
	 * @param date
	 *            La date de départ
	 * 
	 * @return La date demandée
	 */
	public static Date getLastDayOfWeek(final Date date) {
		final GregorianCalendar cal = new GregorianCalendar();
		cal.setTime(date);
		cal.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY); // dimanche
		return new Date(cal.getTimeInMillis());
	}

	/**
	 * Renvoie le premier jour du mois pour la date passée en paramètre.
	 * 
	 * @param date
	 *            La date de départ
	 * 
	 * @return La date demandée
	 */
	public static Date getFirstDayOfMonth(final Date date) {
		final GregorianCalendar cal = new GregorianCalendar();
		cal.setTime(date);
		cal.set(Calendar.DAY_OF_MONTH, cal.getActualMinimum(Calendar.DAY_OF_MONTH));
		return new Date(cal.getTimeInMillis());
	}

	/**
	 * Renvoie le dernier jour du mois pour la date passée en paramètre.
	 * 
	 * @param date
	 *            La date de départ
	 * 
	 * @return La date demandée
	 */
	public static Date getLastDayOfMonth(final Date date) {
		final GregorianCalendar cal = new GregorianCalendar();
		cal.setTime(date);
		cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));
		return new Date(cal.getTimeInMillis());
	}

	/**
	 * Renvoie la date correspondant à la date plus le nombre de jours demandés.
	 * 
	 * @param date
	 *            La date de départ
	 * @param nbDays
	 *            Le nombre de jours à ajouter (positif) ou retirer (négatif)
	 * 
	 * @return La date demandée
	 */
	public static Date addDays(final Date date, final int nbDays) {
		final GregorianCalendar cal = new GregorianCalendar();
		cal.setTime(date);
		cal.add(Calendar.DAY_OF_MONTH, nbDays);
		return new Date(cal.getTimeInMillis());
	}
}
