/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

// TODO: Auto-generated Javadoc
/**
 * The Class FileUtil.
 */
public class FileUtil {

	/** Logger available to subclasses. */
	private static final Logger LOGGER = LoggerFactory.getLogger(FileUtil.class);

	private static final Pattern fileRegex = Pattern.compile("[\\p{L}\\w -:]+(((\\.|-)([0-9a-zA-Z]+))$)");

	/**
	 * Copie de fichiers.
	 *
	 * @param in
	 *            the in
	 * @param out
	 *            the out
	 *
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public static void copyInputStream(final InputStream in, final OutputStream out) throws IOException {
		IOUtils.copy(in,out);
	}

	/**
	 * Vidage d'un répertoire.
	 *
	 * @param directory
	 *            the directory
	 *
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public static void emptyDirectory(final File directory) throws IOException {
		FileUtils.cleanDirectory(directory);
	}

	/**
	 * Vidage & suppression d'un répertoire.
	 *
	 * @param directory
	 *            the directory
	 *
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public static void emptyAndDeleteDirectory(final File directory) throws IOException {
		FileUtils.deleteDirectory(directory);
	}

	/**
	 * copie le fichier source dans le fichier resultat retourne vrai si cela réussit.
	 *
	 * @param source
	 *            the source
	 * @param destination
	 *            the destination
	 *
	 * @return la taille du fichier copié
	 */
	public static int copierFichier(final File source, final File destination) {
		return copierFichier(source, destination, false);
	}

	/**
	 * copie le fichier source dans le fichier resultat retourne vrai si cela réussit.
	 *
	 * @param source
	 *            the source
	 * @param destination
	 *            the destination
	 * @param deleteSource
	 *            de la source après copie
	 *
	 * @return la taille du fichier copié
	 */
	public static int copierFichier(final File source, final File destination, final boolean deleteSource) {
		final int poids = 0;
		try {
			primitiveCopierFichier(source, destination, deleteSource);
		} catch (final IOException e) {
			LOGGER.error(String.format("Erreur lors de l'ecriture du fichier %s", destination.getAbsolutePath()), e);
		}
		return poids;
	}

	public static int primitiveCopierFichier(final File source, final File destination, final boolean deleteSource) throws IOException {
		int poids = 0;

		if (source.isDirectory()) {
			FileUtils.copyDirectory(source, destination);
		} else {
			FileUtils.copyFile(source, destination);
			poids = (int)destination.length();
		}
		if (deleteSource) {
			source.delete();
		}
		return poids;
	}

	/**
	 * Calculer poids en Ko.
	 *
	 * @param f
	 *            the f
	 *
	 * @return the int
	 */
	public static int calculerPoids(final File f) {
		try {
			return (int)Math.ceil(FileUtils.sizeOf(f)/1024.0);
		} catch (IllegalArgumentException e) {
			LOGGER.error(String.format("Fichier %s introuvable.", f.getName()), e);
		}
		return 0;
	}

	/**
	 * Gets the extension.
	 *
	 * @param filename
	 *            the filename
	 *
	 * @return the extension
	 */
	public static String getExtension(final String filename) {
		final Matcher fileMatcher = fileRegex.matcher(filename);
		String res = StringUtils.EMPTY;
		if (fileMatcher.matches()) {
			if (fileMatcher.groupCount() > 0) {
				res = fileMatcher.group(4);
			}
		}
		return res.toLowerCase();
	}

	public static String mkdir(final String string) {
		final File file = new File(string);
		if (!file.exists()) {
			file.mkdirs();
		}
		return string;
	}
}
