/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import java.util.TreeSet;
import java.util.Vector;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.core.InfoBean;
import com.kportal.cms.mail.MailFormaterManager;
import com.kportal.cms.mail.MailNotificationFormater;
import com.kportal.core.config.MessageHelper;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.DiffusionSelective;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.PermissionBean;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Rubriquepublication;
import com.univ.objetspartages.om.Utilisateur;
import com.univ.objetspartages.processus.SaisieFiche;

/**
 * The Class MailUtil.
 * 
 * @author Romain
 * 
 *         To change the template for this generated type comment go to Window - Preferences - Java - Code Generation - Code and Comments
 */
public class MailUtil {

	private final static Logger LOGGER = LoggerFactory.getLogger(MailUtil.class);

	private static MailFormaterManager getFormaterManager() {
		return (MailFormaterManager) ApplicationContextManager.getCoreContextBean(MailFormaterManager.ID_BEAN);
	}

	private static List<MailNotificationFormater> getFormaters(final String name) {
		if (getFormaterManager().getFormaters().get(name) != null) {
			return getFormaterManager().getFormaters().get(name);
		}
		return getFormaterManager().getDefaultFormaters();
	}

	/**
	 * Envoyer mail retour validation.
	 *
	 * @param infoBean
	 *            the info bean
	 * @param ficheUniv
	 *            the fiche univ
	 * @param meta
	 *            the meta
	 * @param miseEnLigne
	 *            the mise en ligne
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void envoyerMailRetourValidation(final InfoBean infoBean, final FicheUniv ficheUniv, final Metatag meta, final boolean miseEnLigne) throws Exception {
		for (final MailNotificationFormater formater : getFormaters(ficheUniv.getClass().getName())) {
			// AA : on utilise l'historique pour savoir qui a fait la demande de validation
			// pour l'envoyer a cet utilisateur en destinataire principal et
			// on met le redacteur de la fiche en copie.
			LOGGER.debug("Parcours de l'historique pour retrouver le demandeur de validation : " + meta.getMetaHistorique());
			final Vector<String> vHistorique = meta.getVecteurHistorique();
			final Iterator<String> itHistorique = vHistorique.iterator();
			String evenement = "", item = "", auteurModification = "";
			//chaque item est forme ainsi : [evenement/dateModif/code_auteur/etat_objet]";
			while (itHistorique.hasNext()) {
				item = itHistorique.next();
				if (item.length() > 0) {
					item = item.substring(1);
					final StringTokenizer st = new StringTokenizer(item, "/");
					if (st.hasMoreTokens()) {
						//l'action est le premier élément
						evenement = st.nextToken();
						//si l'action est bien une demande de validation, on récupère l'auteur
						if (evenement.equals(Metatag.HISTORIQUE_DEMANDE_VALIDATION) && st.hasMoreTokens()) {
							st.nextToken();
							if (st.hasMoreTokens()) {
								auteurModification = st.nextToken();
								break;
							}
						}
					}
				}
			}
			//Si on a trouvé le demandeur, il est destinataire sinon on met le redacteur de la fiche
			final Utilisateur validateur = Utilisateur.getUtilisateur(ContexteUtil.getContexteUniv().getCode());
			String message = "";
			if (miseEnLigne) {
				message = formater.getMessageRetourMiseEnLigne(infoBean.getValues(), ficheUniv, meta, validateur);
			} else {
				message = formater.getMessageRetourNonValide(infoBean.getValues(), ficheUniv, meta, validateur);
			}
			if (StringUtils.isNotEmpty(message)) {
				//Envoi du mail
				final ArrayList<String> lstDest = new ArrayList<>();
				if (auteurModification.length() > 0) {
					LOGGER.debug("Envoi mail retour validation au demandeur : " + auteurModification);
					final Utilisateur ut = Utilisateur.getUtilisateur(auteurModification);
					if (ut.getCode().length() > 0) {
						lstDest.add(ut.getAdresseMail());
					}
				}
				//Ajout a la liste des destinataires l'utilisateur anonyme
				if (ficheUniv.getCodeRedacteur().equalsIgnoreCase(com.univ.objetspartages.om.Utilisateur.UTILISATEUR_ANONYME)) {
					if (StringUtils.isNotEmpty(meta.getMetaMailAnonyme())) {
						lstDest.add(meta.getMetaMailAnonyme());
					}
				}
				if (lstDest.size() > 0) {
					final String objet = StringUtils.defaultIfEmpty(MessageHelper.getCoreMessage("ST_MAIL_OBJET_RE_DEMANDE_VALIDATION"), "Re: Demande de validation");
					final ThreadMail tm = new ThreadMail(lstDest, objet, message);
					final Thread th = new Thread(tm);
					th.start();
				}
			}
		}
	}

	/**
	 * Envoyer mail demande validation.
	 *
	 * @param infoBean
	 *            the info bean
	 * @param ficheUniv
	 *            the fiche univ
	 * @param meta
	 *            the meta
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void envoyerMailDemandeValidation(final InfoBean infoBean, final FicheUniv ficheUniv, final Metatag meta) throws Exception {
		for (final MailNotificationFormater formater : getFormaters(ficheUniv.getClass().getName())) {
			String chaineValidateur = "";
			if (infoBean.get("LISTE_VALIDATEURS") != null) {
				chaineValidateur = infoBean.getString("LISTE_VALIDATEURS");
			}
			if (! chaineValidateur.equalsIgnoreCase("[AUCUNS]")) {
				final String codeRubriqueValidation = ficheUniv.getCodeRubrique();
				final String publicsVisesValidation = "";
				String espaceValidation = "";
				if (ficheUniv instanceof DiffusionSelective) {
					// suppression périmètres groupes sur les fiches
					if (((DiffusionSelective) ficheUniv).getDiffusionModeRestriction().equals("4")) {
						espaceValidation = ((DiffusionSelective) ficheUniv).getDiffusionPublicViseRestriction();
					}
				}
				// affinage de l'envoi de mail sur le niveau d'approbation
				String codeAction = "V";
				if (StringUtils.isNotEmpty(meta.getMetaNiveauApprobation())) {
					codeAction = meta.getMetaNiveauApprobation();
				}
				// structures secondaires
				final TreeSet<String> mails = Utilisateur.getListeUtilisateursPossedantPermission(new PermissionBean("FICHE",
					infoBean.getString("CODE_OBJET"), codeAction), AutorisationBean.getStructuresPerimetreFiche(ficheUniv), codeRubriqueValidation, publicsVisesValidation,
					espaceValidation, true);
				//recuperation de l'utilisateur demandeur de la validation
				final Utilisateur utilisateur_demandeur = Utilisateur.getUtilisateur(ContexteUtil.getContexteUniv().getCode());
				final String message = formater.getMessageDemandeValidation(infoBean.getValues(), ficheUniv, meta, utilisateur_demandeur);
				if (StringUtils.isNotEmpty(message)) {
					final ArrayList<String> lDestMail = new ArrayList<>();
					final Vector<String> vValidateur = Chaine.getVecteurPointsVirgules(chaineValidateur);
					for (final String validateur : mails) {
						// filtrage des validateurs si liste spécifiée
						final String[] tValidateur = validateur.split(";", -2);
						final String adresseMail = tValidateur[0];
						final String code = tValidateur[2];
						if (vValidateur.size() == 0 || vValidateur.contains(tValidateur[2])) {
							lDestMail.add(tValidateur[0]);
						}
					}
					// envoi des mails dans un thread parallele
					final String objet = StringUtils.defaultIfEmpty(MessageHelper.getCoreMessage("ST_MAIL_OBJET_DEMANDE_VALIDATION"), "Demande de validation");
					final ThreadMail tm = new ThreadMail(lDestMail, objet, message);
					final Thread th = new Thread(tm);
					th.start();
				}
			}
		}
	}

	/**
	 * Envoyer mail notification.
	 *
	 * @param infoBean
	 *            the info bean
	 * @param ficheUniv
	 *            the fiche univ
	 * @param meta
	 *            the meta
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static void envoyerMailNotification(final InfoBean infoBean, final FicheUniv ficheUniv, final Metatag meta) throws Exception {
		final ContexteUniv ctx = ContexteUtil.getContexteUniv();
		// TODO : supprimer cette boucle et revoir la notion de MailNotificationFormater
		for (final MailNotificationFormater formater : getFormaters(ficheUniv.getClass().getName())) {
			// Envoi d'un mail de notification spécifique pour le collaboratif
			if (! FicheUnivMgr.isFicheCollaborative(ficheUniv)) {
				final String codeObjet = ReferentielObjets.getCodeObjet(ficheUniv);
				final String codeRubriqueValidation = ficheUniv.getCodeRubrique();
				final String publicsVisesValidation = "";
				String espaceValidation = "";
				if (ficheUniv instanceof DiffusionSelective) {
					if (((DiffusionSelective) ficheUniv).getDiffusionModeRestriction().equals("4")) {
						espaceValidation = ((DiffusionSelective) ficheUniv).getDiffusionPublicViseRestriction();
					}
				}
				/* JSS 20041031: structures secondaires */
				final TreeSet<String> mails = Utilisateur.getListeUtilisateursPossedantPermission(new PermissionBean("FICHE", codeObjet, "N"),
						AutorisationBean.getStructuresPerimetreFiche(ficheUniv), codeRubriqueValidation, publicsVisesValidation, espaceValidation, true);
				/* rubriques de publication */
				final Vector<String> rubPubs = Rubriquepublication.getListeRubriquesPublication(ContexteUtil.getContexteUniv(), ficheUniv);
				for (String rubPub : rubPubs) {
					mails.addAll(Utilisateur.getListeUtilisateursPossedantPermission(new PermissionBean("FICHE", codeObjet, "N"),
							AutorisationBean.getStructuresPerimetreFiche(ficheUniv), rubPub, publicsVisesValidation, espaceValidation, true));
				}
				/* recuperation de l'utilisateur à notifier */
				final Utilisateur utilisateur_demandeur = Utilisateur.getUtilisateur(ContexteUtil.getContexteUniv().getCode());
				final String message = formater.getMessageNotification(infoBean.getValues(), ficheUniv, meta, utilisateur_demandeur);
				if (StringUtils.isNotEmpty(message)) {
					final ArrayList<String> lDestMail = new ArrayList<>();
					for (final String validateur : mails) {
						// RP20050811 filtrage du code redacteur
						final String[] tValidateur = validateur.split(";", -2);
						if (! tValidateur[2].equals(utilisateur_demandeur.getCode())) {
							lDestMail.add(tValidateur[0]);
						}
					}
					// envoi des mails dans un thread parallele
					String userLabel = StringUtils.EMPTY;
					if (ctx != null) {
						userLabel = Utilisateur.getLibelle(ctx.getCode());
					}
					userLabel = StringUtils.defaultString(userLabel, MessageHelper.getCoreMessage("MAIL.NOTIFICATION.AN_ANON"));
					final String objet = String.format(MessageHelper.getCoreMessage("MAIL.NOTIFICATION_OBJECT"), ReferentielObjets.getLibelleObjet(codeObjet), ficheUniv.getLibelleAffichable(), userLabel);
					final ThreadMail tm = new ThreadMail(lDestMail, objet, message);
					final Thread th = new Thread(tm);
					th.start();
				}
			}
		}
	}

	/**
	 * Controle si les validateurs possèdent le droit d'approbation choisi.
	 * 
	 * @param processus
	 *            the processus
	 * @param _fiche
	 *            the _fiche
	 * @param _chaineUtilisateur
	 *            the _chaine utilisateur
	 * @param _codeApprobation
	 *            the _code approbation
	 * 
	 * @return true, if controler liste validateur
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public static boolean controlerListeValidateur(final SaisieFiche processus, final FicheUniv _fiche, final String _chaineUtilisateur, final String _codeApprobation)
		throws Exception {
		boolean res = true;
		if (_chaineUtilisateur != null && _chaineUtilisateur.length() > 0 && ! _chaineUtilisateur.equals("[AUCUNS]")) {
			final String codeObjet = ReferentielObjets.getCodeObjet(_fiche);
			final String codeRubriqueValidation = _fiche.getCodeRubrique();
			final String publicsVisesValidation = "";
			String espaceValidation = "";
			if (_fiche instanceof DiffusionSelective) {
				// JSS 20051201 : suppression périmètres groupes sur les fiches
				//publicsVisesValidation = ((DiffusionSelective) _fiche).getDiffusionPublicVise();
				if (((DiffusionSelective) _fiche).getDiffusionModeRestriction().equals("4")) {
					espaceValidation = ((DiffusionSelective) _fiche).getDiffusionPublicViseRestriction();
				}
			}
			final TreeSet<String> mails = Utilisateur.getListeUtilisateursPossedantPermission(new PermissionBean("FICHE", codeObjet, _codeApprobation),
				AutorisationBean.getStructuresPerimetreFiche(_fiche), codeRubriqueValidation, publicsVisesValidation, espaceValidation, false);
			final String utilisateur[] = _chaineUtilisateur.split(";", -2);
			for (String anUtilisateur : utilisateur) {
				if (!mails.contains(anUtilisateur)) {
					res = false;
					break;
				}
			}
		}
		return res;
	}

	/**
	 * Vérifie le format d'un mail.
	 * 
	 * @param mail
	 *            the mail
	 * 
	 * @return true, if verif mail
	 */
	public static boolean verifMail(final String mail) {
		return Pattern.matches("[_A-Za-z0-9-]+(\\.[_A-Za-z0-9-]+)*@[A-Za-z0-9-]+(\\.[A-Za-z0-9-]+)" + "*(\\.[_A-Za-z0-9-]+)", mail);
	}
}
