/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils;

import java.sql.Date;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.document.CompressionTools;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryParser.MultiFieldQueryParser;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TermRangeQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TopFieldCollector;
import org.apache.lucene.search.WildcardQuery;
import org.apache.lucene.search.highlight.Formatter;
import org.apache.lucene.search.highlight.Highlighter;
import org.apache.lucene.search.highlight.QueryScorer;
import org.apache.lucene.search.highlight.SimpleHTMLFormatter;
import org.apache.lucene.search.highlight.SimpleSpanFragmenter;
import org.apache.lucene.util.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.core.Formateur;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.textsearch.CibleRecherche;
import com.jsbsoft.jtf.textsearch.FrenchAnalyzer;
import com.jsbsoft.jtf.textsearch.Index;
import com.jsbsoft.jtf.textsearch.ResultatRecherche;
import com.jsbsoft.jtf.textsearch.Searcher;
import com.jsbsoft.jtf.textsearch.sitesdistants.RechercheSitesDistants;
import com.jsbsoft.jtf.textsearch.util.BooleanQueryBySite;
import com.jsbsoft.jtf.textsearch.util.SearchPropertiesHelper;
import com.kportal.core.config.PropertyHelper;
import com.univ.collaboratif.om.Espacecollaboratif;
import com.univ.collaboratif.om.InfosEspaceCollaboratif;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.om.Groupedsi;
import com.univ.objetspartages.om.InfosGroupeDsi;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Rubrique;

public class RechercheUtil {

	private static final Logger LOG = LoggerFactory.getLogger(RechercheUtil.class);

	/**
	 * Traiter recherche.
	 *
	 * @param ctx     the ctx
	 * @param request the request
	 * @return the vector
	 * @throws Exception the exception
	 */
	public static Vector<ResultatRecherche> traiterRecherche(final OMContext ctx, final String request) throws Exception {
		final Searcher searcher = Searcher.getInstance();
		if (searcher.getSearcher() == null) {
			return new Vector<>();
		}
		return new Vector<>(search(ctx, searcher, request));
	}

	/**
	 *
	 * @param ctx
	 * @param request
	 * @return
	 * @throws Exception
	 */
	public static int getTotalResultats(final OMContext ctx, final String request) throws Exception {
		final Searcher searcher = Searcher.getInstance();
		if (searcher.getSearcher() == null) {
			return 0;
		}
		return getTotalHits(ctx, searcher, request);
	}

	/**
	 * Traite la recherche.
	 *
	 * @param ctx the ctx
	 * @param request the request
	 * @return the vector
	 * @throws Exception the exception
	 */
	private static List<ResultatRecherche> search(final OMContext ctx, final Searcher searcher, String request) throws Exception {
		request = EscapeString.unescapeURL(request);
		final BooleanQueryBySite query = getQuery(ctx, request);
		// traitement de la recherche fulltext
		final BooleanQuery searchQuery = getSearchQuery(searcher.getAnalyzer(), request, query.getAlias());
		if (!searchQuery.clauses().isEmpty()) {
			query.add(searchQuery, BooleanClause.Occur.MUST);
		}
		// TRI ET PAGINATION : pagination uniquement si pas de tri
		final String tri = RequeteUtil.renvoyerParametre(request, "TRI");
		final String page = RequeteUtil.renvoyerParametre(request, "PAGE");
		final List<SortField> sortFields = new ArrayList<>();
		boolean reverse = true;
		if (tri.startsWith("DATE")) {
			reverse = !tri.endsWith("_REV");
			sortFields.add(new SortField(Index.MISE_EN_LIGNE, SortField.STRING, reverse));
		} else if (tri.startsWith("OBJET")) {
			reverse = tri.endsWith("_REV");
			sortFields.add(new SortField(Index.OBJET, SortField.STRING, reverse));
		} else if (tri.startsWith("SITE")) {
			reverse = tri.endsWith("_REV");
			sortFields.add(new SortField(Index.CODE_SITE, SortField.STRING, reverse));
		} else {
			sortFields.add(SortField.FIELD_SCORE);
		}
		final Sort sort = new Sort(sortFields.toArray(new SortField[sortFields.size()]));
		int nbResultatsParPage = Searcher.MAX_RESULT;
		int debut = 0, end = 0;
		if (page == null || !page.equalsIgnoreCase("false")) {
			// par defaut pagination = 10 resultats par page
			nbResultatsParPage = 10;
			if (PropertyHelper.getCoreProperty("recherche.nbrespage") != null) {
				try {
					nbResultatsParPage = Integer.parseInt(PropertyHelper.getCoreProperty("recherche.nbrespage"));
				} catch (final Exception ignore) {}
			}
			if (RequeteUtil.renvoyerParametre(request, "FROM").length() > 0) {
				try {
					debut = Integer.parseInt(RequeteUtil.renvoyerParametre(request, "FROM"));
				} catch (final Exception ignore) {}
			}
		}
		end = debut + nbResultatsParPage;
		//gestion du nombre de résultats max
		final List<ResultatRecherche> listeFiches = new ArrayList<>();
		int maxResult = searcher.getSearcher().maxDoc();
		if (maxResult > 0) {
			if (PropertyHelper.getCoreProperty("recherche.nbresmax") != null) {
				try {
					maxResult = Integer.parseInt(PropertyHelper.getCoreProperty("recherche.nbresmax"));
				} catch (final Exception ignore) {}
			}
			final TopFieldCollector tfc = TopFieldCollector.create(sort, maxResult, false, true/*trackDocScores */, true /* trackMaxScore */, false /* docsInOrder */);
			searcher.getSearcher().search(query, tfc);
			final int totalHits = tfc.getTotalHits();
			final TopDocs results = tfc.topDocs();
			final float maxScore = results.getMaxScore();
			final int nbHits = results.scoreDocs.length;
			//parcours des résultats, et stockage dans un arraylist
			LOG.debug("" + System.currentTimeMillis() + ", nb resultats : " + nbHits);
			// HightLighter
			final Formatter formatter = new SimpleHTMLFormatter("<mark class=\"highlight\">", "</mark>");
			final QueryScorer queryScorer = new QueryScorer(searchQuery);
			final Highlighter highlighter = new Highlighter(formatter, queryScorer);
			highlighter.setTextFragmenter(new SimpleSpanFragmenter(queryScorer, 200));
			highlighter.setMaxDocCharsToAnalyze(Integer.MAX_VALUE);
			if (nbHits > 0) {
				if (nbHits < end) {
					end = nbHits;
				}
				for (int i = debut; i < end; i++) {
					final Document doc = searcher.getSearcher().doc(results.scoreDocs[i].doc);
					final ResultatRecherche resultatRecherche = new ResultatRecherche();
					resultatRecherche.setObjet(doc.get(Index.OBJET));
					resultatRecherche.setIdFiche(doc.get(Index.ID_FICHE));
					resultatRecherche.setLangue(doc.get(Index.LANGUE));
					resultatRecherche.setCodeRubrique(doc.get(Index.CODE_RUBRIQUE));
					resultatRecherche.setTitre(doc.get(Index.TITLE));
					resultatRecherche.setUrl(doc.get(Index.URL));
					resultatRecherche.setDateModification(doc.get(Index.LAST_MODIFIED));
					resultatRecherche.setDescription(doc.get(Index.DESCRIPTION));
					resultatRecherche.setScore(results.scoreDocs[i].score / maxScore);
					resultatRecherche.setRank(i);
					resultatRecherche.setTotal(nbHits);
					resultatRecherche.setTotalHits(totalHits);
					resultatRecherche.setHighlightedTextContent(StringUtils.defaultIfEmpty(highlighter.getBestFragment(searcher.getAnalyzer(), Index.CONTENT,
						CompressionTools.decompressString(doc.getBinaryValue((Index.CONTENT)))), ""));
					resultatRecherche.setHighlightedTextContentFile(StringUtils.defaultIfEmpty(highlighter.getBestFragment(searcher.getAnalyzer(), Index.CONTENT_FILE,
						CompressionTools.decompressString(doc.getBinaryValue((Index.CONTENT_FILE)))), ""));
					//on stocke la chaine qui permettra de faire un RequeteUtil.lireFiche dans la jsp
					listeFiches.add(resultatRecherche);
				}
			}
		} else {
			LOG.warn("Aucun document dans l'index de recherche");
		}
		return listeFiches;
	}

	/**
	 * Gets the total hits.
	 *
	 * @param ctx     the ctx
	 * @param request the request
	 * @return the total hits
	 * @throws Exception the exception
	 */
	private static int getTotalHits(final OMContext ctx, final Searcher searcher, String request) throws Exception {
		request = EscapeString.unescapeURL(request);
		final BooleanQueryBySite query = getQuery(ctx, request);
		query.add(getSearchQuery(searcher.getAnalyzer(), request, query.getAlias()), BooleanClause.Occur.MUST);
		final TopDocs top = searcher.getSearcher().search(query, Searcher.MAX_RESULT);
		return top.totalHits;
	}

	/**
	 * Calcul de la requete Lucene
	 *
	 * @param ctx the ctx
	 * @param request the request
	 * @return the vector
	 * @throws Exception the exception
	 */
	private static BooleanQueryBySite getQuery(final OMContext ctx, final String request) throws Exception {
		final BooleanQueryBySite query = new BooleanQueryBySite();
		int maxClauses = 2048;
		final String paramMaxClauses = PropertyHelper.getCoreProperty("lucene.maxclauses");
		if (paramMaxClauses != null && paramMaxClauses.length() > 0) {
			try {
				maxClauses = Integer.parseInt(paramMaxClauses);
			} catch (final Exception ignored) {}
		}
		ContexteUniv ctxUniv = null;
  		if (ctx instanceof ContexteUniv) {
			ctxUniv = (ContexteUniv) ctx;
   		}
		BooleanQuery.setMaxClauseCount(maxClauses);
		// debut de traitement sur les rubriques
		final BooleanQuery qB = new BooleanQuery();
		final boolean dsiActif = "1".equals(PropertyHelper.getCoreProperty("dsi.activation"));
		// recherche dans un site ou une rubrique
		String codeRubrique = RequeteUtil.renvoyerParametre(request, "CODE_RUBRIQUE");
		if (codeRubrique == null) {
			codeRubrique = "";
		}
		final Boolean rechercheArborescente = Boolean.valueOf(RequeteUtil.renvoyerParametre(request, "SEARCH_SOUSRUBRIQUES"));
		// on peut forcer la recherche à un site distant
		// @deprecated@ CODE_SITE_DISTANT = ancien paramètre (v5.0), à remplacer par SITES_DISTANTS (multivalué)
		final String codeSiteDistant = RequeteUtil.renvoyerParametre(request, "CODE_SITE_DISTANT");
		final String sitesDistants = RequeteUtil.renvoyerParametre(request, "SITES_DISTANTS");
		final String sitesDistantsExclus = RequeteUtil.renvoyerParametre(request, "SITES_DISTANTS_EXCLUS");
		// si on recherche sur l'ensemble des sites distants on restreint également la rubrique
		if ((codeSiteDistant.equals("") || codeSiteDistant.equalsIgnoreCase("TOUS")) && sitesDistants.length() == 0) {
			// en front on restreint la rubrique au site courant si il est cloisonné
			InfosSite infosSite = null;
			String codeRubriqueSite = null;
			if (ctxUniv != null && ctxUniv.isCalculListeResultatsFront()) {
				infosSite = ctxUniv.getInfosSite();
				if (infosSite != null && infosSite.getRestriction() == 1) {
					codeRubriqueSite = infosSite.getCodeRubrique();
				}
			}
			// si la rubrique a été forcée au site courant
			if (codeRubriqueSite != null && codeRubriqueSite.length() > 0) {
				final InfosRubriques infosRub = Rubrique.renvoyerItemRubrique(codeRubrique);
				// restriction au site uniquement si la rubrique n'appartient pas au site
				if (infosRub.getCode().length() == 0) {
					codeRubrique = codeRubriqueSite;
				} else if (!infosSite.isRubriqueVisibleInSite(infosRub)) {
					codeRubrique = Rubrique.CODE_RUBRIQUE_INEXISTANTE;
				}
			}
			final Collection<InfosRubriques> listeRubriques = new HashSet<>();
			InfosRubriques rubrique;
			if (codeRubrique.length() == 0) {
				rubrique = Rubrique.getTopLevelRubrique();
				listeRubriques.add(Rubrique.renvoyerItemRubrique(""));
			} else {
				rubrique = Rubrique.renvoyerItemRubrique(codeRubrique);
			}
			if (dsiActif && ctxUniv != null && ctxUniv.isCalculListeResultatsFront()) {
				if (Rubrique.controlerRestrictionRubrique(ctxUniv, codeRubrique)) {
					if (rechercheArborescente) {
						listeRubriques.addAll(Rubrique.determinerListeSousRubriquesAutorisees(ctxUniv, rubrique));
					}
				} else {
					// si aucun droit on bloque la recherche par un code bidon
					rubrique = new InfosRubriques(Rubrique.CODE_RUBRIQUE_INEXISTANTE);
				}
				listeRubriques.add(rubrique);
			} else {
				if (rechercheArborescente) {
					listeRubriques.addAll(rubrique.getListeSousRubriquesTousNiveaux());
				}
				listeRubriques.add(rubrique);
			}
			// on ajoute l'ensemble des sites distants
			final RechercheSitesDistants rsd = new RechercheSitesDistants();
			final Collection<CibleRecherche> listeSites = rsd.getLstCible();
			if (listeSites.size() > 0 && !"TOUS".equals(sitesDistantsExclus)) {
				List<String> listeSitesDistants = null;
				List<String> listeSitesDistantsExclus = null;
				if (sitesDistants.length() > 0 && !"TOUS".equals(sitesDistants)) {
					// on ajoute le ou les sites distants demandés
					listeSitesDistants = Arrays.asList(sitesDistants.split(";"));
				} else if (sitesDistantsExclus.length() > 0) {
					// on ajoute tous les sites sauf ceux qui sont exclus
					listeSitesDistantsExclus = Arrays.asList(sitesDistantsExclus.split(";"));
				}
				for (final CibleRecherche infosSiteDistant : listeSites) {
					if (listeSitesDistants == null && listeSitesDistantsExclus == null) {
						// on ajoute l'ensemble des sites distants
						addRubriqueClause(qB, infosSiteDistant.getCode());
					} else if (listeSitesDistants != null && listeSitesDistants.contains(infosSiteDistant.getCode())) {
						// on ajoute le ou les sites distants demandés
						addRubriqueClause(qB, infosSiteDistant.getCode());
					} else if (listeSitesDistantsExclus != null && !listeSitesDistantsExclus.contains(infosSiteDistant.getCode())) {
						// on ajoute tous les sites sauf ceux qui sont exclus
						addRubriqueClause(qB, infosSiteDistant.getCode());
					}
				}
			}
			for (final InfosRubriques sousRubrique : listeRubriques) {
				addRubriqueClause(qB, sousRubrique.getCode());
			}
		} else if (!codeSiteDistant.equals("") && !codeSiteDistant.equalsIgnoreCase("TOUS")) {
			// on ajoute le site distant demandé
			addRubriqueClause(qB, codeSiteDistant);
		}
		if (qB.getClauses().length > 0) {
			query.add(qB, BooleanClause.Occur.MUST);
			// fin de la requete sur les rubriques
		}
		// calcul du site courant pour personnalisation de la recherche
		String alias = "";
		if (SearchPropertiesHelper.executeRechercheBySite() && StringUtils.isNotEmpty(codeRubrique)) {
			final InfosSite infosSite = Site.determinerSiteRubrique(codeRubrique);
			alias = infosSite.getAlias();
		}
		List<String> listeObjetsExclus = new ArrayList<>();
		final String chaineObjetsExclus = RequeteUtil.renvoyerParametre(request, "SEARCH_EXCLUSIONOBJET");
		if (chaineObjetsExclus.length() > 0) {
			listeObjetsExclus = Chaine.getVecteurPointsVirgules(chaineObjetsExclus.replaceAll(" ", ";"));
		}
		final String objet = RequeteUtil.renvoyerParametre(request, "OBJET");
		// si on recherche sur un seul objet
		if (StringUtils.isNotEmpty(objet) && !objet.equals("TOUS")) {
			if (!listeObjetsExclus.contains(objet)) {
				// pas besoin de boost
				query.add(new TermQuery(new Term(Index.OBJET, objet.toLowerCase())), BooleanClause.Occur.MUST);
			}
		}
		// si exclusion ou personnalisation du boost par objet
		else if (listeObjetsExclus.size() > 0 || SearchPropertiesHelper.hasBoostParObjet(alias)) {
			if (SearchPropertiesHelper.executeRechercheBySite() || SearchPropertiesHelper.hasBoostParObjet(alias)) {
				final List<String> listeObjets = new ArrayList<>();
				listeObjets.addAll(ReferentielObjets.getReferentiel().getCodesObjetsTries());
				// boost par objet
				for (final String codeObjet : listeObjets) {
					final String nomObjet = ReferentielObjets.getNomObjet(codeObjet);
					if (!listeObjetsExclus.contains(nomObjet)) {
						final TermQuery tq = new TermQuery(new Term(Index.OBJET, nomObjet.toLowerCase()));
						tq.setBoost(SearchPropertiesHelper.getBoostParObjet(alias, nomObjet) * 1F);
						query.add(tq, BooleanClause.Occur.SHOULD);
					}
				}
			} else {
				// pas besoin de boost en exclusion
				for (final String nomObjet : listeObjetsExclus) {
					query.add(new TermQuery(new Term(Index.OBJET, nomObjet.toLowerCase())), BooleanClause.Occur.MUST_NOT);
				}
			}
		}
		// parsing des dates
		boolean controlerDate = false;
		String dateDebut = "19700101";
		String dateFin = "20990101";
		String dateRequete = RequeteUtil.renvoyerParametre(request, "DATE_DEBUT");
		if (dateRequete.length() > 0) {
			final Date date = Formateur.parserDate(dateRequete);
			dateDebut = new SimpleDateFormat("yyyyMMdd").format(date);
			controlerDate = true;
		}
		dateRequete = RequeteUtil.renvoyerParametre(request, "DATE_FIN");
		if (dateRequete.length() > 0) {
			final Date date = Formateur.parserDate(dateRequete);
			dateFin = new SimpleDateFormat("yyyyMMdd").format(date);
			controlerDate = true;
		}
		// LANGUE
		final String langue = RequeteUtil.renvoyerParametre(request, "LANGUE_SEARCH");
		// ETAT_OBJET
		//recherche uniquement sur les fiches en ligne
		query.add(new TermQuery(new Term(Index.ETAT_FICHE, "0003")), BooleanClause.Occur.MUST);
		// LANGUE
		if (StringUtils.isNotEmpty(langue)) {
			query.add(new TermQuery(new Term(Index.LANGUE, langue)), BooleanClause.Occur.MUST);
		}
		// DATE
		if (controlerDate) {
			final TermRangeQuery qD = new TermRangeQuery(Index.LAST_MODIFIED, dateDebut, dateFin, true, true);
			query.add(qD, BooleanClause.Occur.MUST);
		}
		// ESPACE COLLAB
		final String codeEspace = RequeteUtil.renvoyerParametre(request, "ESPACE");
		if (StringUtils.isNotEmpty(codeEspace)) {
			final BooleanQuery bQMode = new BooleanQuery();
			bQMode.add(new TermQuery(new Term(Index.MODE_RESTRICTION, "4")), BooleanClause.Occur.MUST);
			bQMode.add(new WildcardQuery(new Term(Index.PUBLIC_VISE_RESTRICTION, "*;" + codeEspace + ";*")), BooleanClause.Occur.MUST);
			query.add(bQMode, BooleanClause.Occur.MUST);
		} else if (dsiActif && ctxUniv != null && ctxUniv.isCalculListeResultatsFront()) {
			final BooleanQuery bQMode = new BooleanQuery();
			//si pas de restriction
			bQMode.add(new TermQuery(new Term(Index.MODE_RESTRICTION, "0")), BooleanClause.Occur.SHOULD);
			//si loggué, on liste groupes et profils
			if (ctxUniv.getKsession().length() > 0) {
				// liste des groupes de l'utilisateur connecté
				final TreeSet<String> groupes = new TreeSet<>();
				for (String groupeCourant : ctxUniv.getGroupesDsi()) {
					groupes.add(groupeCourant);
					InfosGroupeDsi infoGroupe = Groupedsi.renvoyerItemGroupeDsi(groupeCourant);
					/* On recherche les items de niveau supérieur */
					final int niveauItemCourant = infoGroupe.getNiveau();
					int niveau = niveauItemCourant - 1;
					String codeGroupe = "";
					while (niveau > 0) {
						codeGroupe = infoGroupe.getCodeGroupePere();
						groupes.add(codeGroupe);
						infoGroupe = Groupedsi.renvoyerItemGroupeDsi(codeGroupe);
						niveau--;
					}
				}
				// groupes
				if (groupes.size() > 0) {
					final BooleanQuery bQMode2 = new BooleanQuery();
					bQMode2.add(new TermQuery(new Term(Index.MODE_RESTRICTION, "2")), BooleanClause.Occur.MUST);
					final BooleanQuery bQMode3 = new BooleanQuery();
					bQMode3.add(new TermQuery(new Term(Index.MODE_RESTRICTION, "3")), BooleanClause.Occur.MUST);
					final BooleanQuery bQPublicVise = new BooleanQuery();
					final BooleanQuery bQPublicViseR = new BooleanQuery();
					final Iterator<String> i = groupes.iterator();
					String groupe = "";
					while (i.hasNext()) {
						groupe = i.next();
						bQPublicVise.add(new WildcardQuery(new Term(Index.PUBLIC_VISE, "*;/" + groupe + ";*")), BooleanClause.Occur.SHOULD);
						bQPublicViseR.add(new WildcardQuery(new Term(Index.PUBLIC_VISE_RESTRICTION, "*;/" + groupe + ";*")), BooleanClause.Occur.SHOULD);
					}
					// bqMode2 : mode_restriction = 2 ET public_vise contient un des profils/groupes de l'utilisateur
					bQMode2.add(bQPublicVise, BooleanClause.Occur.MUST);
					// bqMode3 : mode_restriction = 3 ET public_vise_restriction contient un des profils/groupes de l'utilisateur
					bQMode3.add(bQPublicViseR, BooleanClause.Occur.MUST);
					// ... ou bqMode2 ou bqMode3 ou ...
					bQMode.add(bQMode2, BooleanClause.Occur.SHOULD);
					bQMode.add(bQMode3, BooleanClause.Occur.SHOULD);
				}
				// liste des espaces de l'utilisateur connecté
				final TreeSet<String> espacesUtilisateur = new TreeSet<>();
				if (Espacecollaboratif.isExtensionActivated()) {
					final Collection<InfosEspaceCollaboratif> listeEspaces = Espacecollaboratif.getListeEspaces();
					for (InfosEspaceCollaboratif espace : listeEspaces) {
						if (Espacecollaboratif.estMembreEspace(ctxUniv, espace)) {
							espacesUtilisateur.add(espace.getCode());
						}
					}
				}
				if (espacesUtilisateur.size() > 0) {
					final BooleanQuery bQMode4 = new BooleanQuery();
					bQMode4.add(new TermQuery(new Term(Index.MODE_RESTRICTION, "4")), BooleanClause.Occur.MUST);
					final BooleanQuery bQEspaces = new BooleanQuery();
					for (String espaceCourant : espacesUtilisateur) {
						bQEspaces.add(new WildcardQuery(new Term(Index.PUBLIC_VISE_RESTRICTION, "*;" + espaceCourant + ";*")), BooleanClause.Occur.SHOULD);
					}
					// bqMode4 : mode_restriction = 4 ET public_vise_restriction contient un des espaces de l'utilisateur
					bQMode4.add(bQEspaces, BooleanClause.Occur.MUST);
					bQMode.add(bQMode4, BooleanClause.Occur.SHOULD);
				}
			}//fin si session
			if (bQMode.getClauses().length > 0) {
				query.add(bQMode, BooleanClause.Occur.MUST);
			}
		}//fin DSI
		// DATE MISE EN LIGNE
		if (ctxUniv != null && ctxUniv.isCalculListeResultatsFront()) {
			final BooleanQuery dateQuery = new BooleanQuery();
			final GregorianCalendar gc = new GregorianCalendar();
			String upper = new SimpleDateFormat("yyyyMMdd").format(new Date(gc.getTimeInMillis()));
			String lower = null;
			for (int boost = SearchPropertiesHelper.getBoostDate(); boost > 1; boost--) {
				gc.set(Calendar.YEAR, gc.get(Calendar.YEAR) - 1);
				lower = new SimpleDateFormat("yyyyMMdd").format(new Date(gc.getTimeInMillis()));
				final TermRangeQuery rQ = new TermRangeQuery(Index.MISE_EN_LIGNE, lower, upper, false, true);
				rQ.setBoost(boost * 10000F);
				dateQuery.add(rQ, BooleanClause.Occur.SHOULD);
				upper = lower;
			}
			// pas de boost
			final TermRangeQuery rQ = new TermRangeQuery(Index.MISE_EN_LIGNE, null, upper, false, true);
			dateQuery.add(rQ, BooleanClause.Occur.SHOULD);
			query.add(dateQuery, BooleanClause.Occur.MUST);
		}
		final BooleanClause[] bc = query.getClauses();
		LOG.debug("détail de la requete:");
		LOG.debug(query.toString());
		for (final BooleanClause element : bc) {
			LOG.debug(element.getQuery().toString() + ", obligatoire : " + element.isRequired() + ", interdit :" + element.isProhibited());
		}
		// ajout de l'alias
		if (StringUtils.isNotEmpty(alias)) {
			query.setAlias(alias);
		}
		return query;
	}

	/**
	 * Gets the search query. calcul de la requete de recherche texte
	 *
	 * @param request the request
	 * @return the search query
	 * @throws ParseException the parse exception
	 */
	private static BooleanQuery getSearchQuery(final FrenchAnalyzer analyzer, final String request, final String alias) throws ParseException {
		final String sQuery = RequeteUtil.renvoyerParametre(request, "QUERY");
		final String documentJoints = RequeteUtil.renvoyerParametre(request, "DOCUMENT_JOINT");
		String[] parsers = null;
		if (documentJoints.equals("1")) {
			parsers = new String[] { Index.TITLE, Index.DESCRIPTION, Index.KEYWORDS, Index.CONTENT, Index.CONTENT_FILE };
		} else {
			parsers = new String[] { Index.TITLE, Index.DESCRIPTION, Index.KEYWORDS, Index.CONTENT };
		}
		final Map<String, Float> boosts = new HashMap<>();
		final MultiFieldQueryParser objParser = new MultiFieldQueryParser(Version.LUCENE_36, parsers, analyzer, boosts);
		final BooleanQuery searchQuery = new BooleanQuery();
		final BooleanQuery searchSousQuery = new BooleanQuery();
		// la recherche vide est autorisée on renvoit une requete vide qui ne sera pas prise en compte
		if (StringUtils.isEmpty(sQuery) && Boolean.valueOf(RequeteUtil.renvoyerParametre(request, "SEARCH_EMPTYQUERY"))) {
			return searchQuery;
		}
		// requete1 termes exacts rapprochés entre guillemets ""
		String sQuery1 = "";
		// requete2 tous les termes exacts présents AND
		String sQuery2 = "";
		// requete3 au moins un terme exact présent OR
		String sQuery3 = "";
		// requete4 tous les wildcards présents AND *
		String sQuery4 = "";
		// requete5 au moins une wildcard présent OR *
		String sQuery5 = "";
		final String s = "\\s*((([^\"\\[\\]\\s]+))|(\\\"([^\"]+)\\\"))\\s*";
		final Pattern p = Pattern.compile(s);
		final Matcher m = p.matcher(sQuery);
		boolean googleLike = false;
		while (m.find()) {
			final String word = m.group(1);
			// un terme
			if (word.length() > 1 || StringUtils.isNumeric(word)) {
				if (!(word.startsWith("\"") && word.endsWith("\""))) {
					if (sQuery1.length() > 0) {
						sQuery1 += " ";
					}
					if (word.startsWith("-") || word.startsWith("+")) {
						sQuery1 += word.substring(0, 1) + escape(word.substring(1));
						googleLike = true;
					} else if (word.equalsIgnoreCase("OR") || word.equalsIgnoreCase("AND")) {
						if (m.find()) {
							sQuery1 += word.toUpperCase() + " " + escape(m.group(1));
							googleLike = true;
						}
					} else {
						sQuery1 += escape(word);
						// on ne traite le mot que si ce n'est pas un stopword
						if (!analyzer.getStoptable().contains(word)) {
							if (sQuery2.length() > 0) {
								sQuery2 += " AND ";
								sQuery3 += " OR ";
							}
							sQuery2 += escape(word);
							sQuery3 += escape(word);
							if (word.length() > 2) {
								if (sQuery4.length() > 0) {
									sQuery4 += " AND ";
									sQuery5 += " OR ";
								}
								// pour les wildcards on enlève les accents
								sQuery4 += Chaine.supprimerISOLatin1Accent(escape(word)) + "*";
								sQuery5 += Chaine.supprimerISOLatin1Accent(escape(word)) + "*";
							}
						}
					}
				}
				// une phrase exacte
				else {
					final Query q = objParser.parse(escape(word));
					q.setBoost(2F);
					// obligatoire
					searchQuery.add(q, BooleanClause.Occur.MUST);
				}
			}
		}
		if (!googleLike) {
			if (sQuery1.length() > 0) {
				final Query q = objParser.parse("\"" + sQuery1 + "\"");
				q.setBoost(SearchPropertiesHelper.getBoostRecherchePhrase(alias) * 1F);
				searchSousQuery.add(q, BooleanClause.Occur.SHOULD);
			}
			if (sQuery2.length() > 0) {
				Query q = objParser.parse(sQuery2);
				q.setBoost(SearchPropertiesHelper.getBoostRechercheAND(alias) * 1F);
				searchSousQuery.add(q, BooleanClause.Occur.SHOULD);
				if (SearchPropertiesHelper.executeRechercheOR(alias)) {
					q = objParser.parse(sQuery3);
					q.setBoost(SearchPropertiesHelper.getBoostRechercheOR(alias) * 1F);
					searchSousQuery.add(q, BooleanClause.Occur.SHOULD);
				}
				if (SearchPropertiesHelper.executeRechercheLIKE(alias)) {
					if (sQuery4.length() > 0) {
						q = objParser.parse(sQuery4);
						q.setBoost(SearchPropertiesHelper.getBoostRechercheLIKE(alias) * 1F);
						searchSousQuery.add(q, BooleanClause.Occur.SHOULD);
						q = objParser.parse(sQuery5);
						q.setBoost(SearchPropertiesHelper.getBoostRechercheLIKE(alias) * 1F);
						searchSousQuery.add(q, BooleanClause.Occur.SHOULD);
					}
				}
				if (searchSousQuery.getClauses().length > 0) {
					searchQuery.add(searchSousQuery, BooleanClause.Occur.MUST);
				}
			}
		} else if (sQuery1.length() > 0) {
			final Query q = objParser.parse(sQuery1);
			searchQuery.add(q, BooleanClause.Occur.MUST);
		}
		return searchQuery;
	}

	/**
	 * Adds the rubrique clause.
	 *
	 * @param query        the query
	 * @param codeRubrique the code rubrique
	 */
	private static void addRubriqueClause(final BooleanQuery query, final String codeRubrique) {
		query.add(new TermQuery(new Term(Index.CODE_RUBRIQUE, codeRubrique)), BooleanClause.Occur.SHOULD);
	}

	/**
	 * Test.
	 *
	 * @param word
	 *            the word
	 *
	 * @return the string
	 */
	private static String escape(final String word) {
		// + - && || ! ( ) { } [ ] ^ " ~ * ? : \ %
		final int lg = word.length();
		final StringBuilder sb = new StringBuilder(lg);
		for (int i = 0; i < lg; i++) {
			final char c = word.charAt(i);
			if (c == '+' || c == '-' || c == '&' || c == '|' || c == '!' || c == '\\' || c == '(' || c == ')' || c == '{' || c == '}' || c == '[' || c == ']' || c == '^' || c == '~' || c == '*' || c == '?' || c == ':' || c == '%') {
				sb.append('\\');
			}
			sb.append(c);
		}
		return sb.toString();
	}
}
