/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils;

import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TreeSet;
import java.util.Vector;

import javax.portlet.PortletException;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponseWrapper;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import com.jsbsoft.jtf.core.ApplicationContextManager;
import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.jsbsoft.jtf.identification.GestionnaireIdentification;
import com.jsbsoft.jtf.identification.ValidateurCAS;
import com.kportal.core.config.MessageHelper;
import com.kportal.core.config.PropertyHelper;
import com.kportal.core.webapp.WebAppUtil;
import com.univ.multisites.InfosSite;
import com.univ.multisites.Site;
import com.univ.objetspartages.cache.CacheServiceManager;
import com.univ.objetspartages.om.InfosPreferences;
import com.univ.objetspartages.om.Preferences;
import com.univ.objetspartages.om.SSOBean;
import com.univ.objetspartages.om.Service;
import com.univ.objetspartages.om.ServiceBean;
import com.univ.portail.service.registry.PortletOccurenceRegistry;
import com.univ.portail.util.Valeur;
import com.univ.services.ReponseService;
import com.univ.services.RequeteService;
import com.univ.services.ServicesInvoker;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.xml.NodeUtil;

// TODO: Auto-generated Javadoc
/**
 * The Class ServicesUtil. Classe utilitaire de gestion des services - fontions de formatage - gestion des caches
 */
public class ServicesUtil {

	private static org.slf4j.Logger LOG = LoggerFactory.getLogger(ServicesUtil.class);

	/** Url d'appel de la vue maxi d'un service */
	private final static String URL_ACTIVATION_SERVICE = "/adminsite/sso/activation_service.jsp?service=";

	public static CacheServiceManager getCacheServiceManager() {
		return (CacheServiceManager) ApplicationContextManager.getCoreContextBean(CacheServiceManager.ID_BEAN);
	}

	public static Map<String, ServiceBean> getServices() {
		Map<String, ServiceBean> res = getCacheServiceManager().call();
		if (res == null) {
			res = new HashMap<String, ServiceBean>();
		}
		return res;
	}

	/**
	 * calcule la liste des codes des services autorisés pour l'utilisateur courant
	 *
	 * appelée automatiquement à la construction du contexteUniv et disponible dans le contexte.
	 *
	 * @param ctx
	 *            the ctx
	 *
	 * @return the tree set
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static TreeSet<String> calculerCodesServicesAutorises(final ContexteUniv ctx) throws Exception {
		final TreeSet<String> listeServices = new TreeSet<String>();
		/** ************************************ */
		/* Parcours de tous les services */
		/** ************************************ */
		for (final ServiceBean service : getServices().values()) {
			boolean serviceAutorise = false;
			/* Contrôle sur les restrictions */
			if (service.getDiffusionModeRestriction().equals("3")) {
				serviceAutorise = true;
			} else if (service.getDiffusionModeRestriction().equals("0") && ctx.getCode().length() > 0) {
				serviceAutorise = true;
			} else {
				Vector<String> groupesRestriction = new Vector<String>();
				if (service.getDiffusionModeRestriction().equals("1")) {
					groupesRestriction = service.getDiffusionPublicVise();
				}
				if (service.getDiffusionModeRestriction().equals("2")) {
					groupesRestriction = service.getDiffusionPublicViseRestriction();
				}
				/* Test sur chaque groupe */
				final Enumeration<String> eRestriction = groupesRestriction.elements();
				while (eRestriction.hasMoreElements()) {
					final String codeGroupeRestriction = eRestriction.nextElement();
					if (ctx.getGroupesDsiAvecAscendants().contains(codeGroupeRestriction)) {
						serviceAutorise = true;
					}
				}
			}
			if (serviceAutorise) {
				listeServices.add(service.getCode());
			}
		}
		return listeServices;
	}

	/**
	 * calcule la liste des services en mode "PUSH" pour l'utilisateur courant.
	 *
	 * @param ctx
	 *            the ctx
	 *
	 * @return the vector
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static Vector<ServiceBean> calculerListeServicesPush(final ContexteUniv ctx) throws Exception {
		// Pour éviter les doublons
		final TreeSet<String> listeCodesServices = new TreeSet<String>();
		// Pour le résultat
		final Vector<ServiceBean> vListeServices = new Vector<ServiceBean>();
		/* Parcours de tous les services */
		for (final ServiceBean service : getServices().values()) {
			// Test autorisation
			if (ctx.getListeServicesAutorises().contains(service.getCode())) {
				boolean pushOK = false;
				if (service.getDiffusionMode().equals("0")) {
					pushOK = true;
				}
				if (service.getDiffusionMode().equals("1")) {
					// Test "push" sur chaque groupe
					final Vector<String> groupesDiffusion = service.getDiffusionPublicVise();
					final Enumeration<String> eDiffusion = groupesDiffusion.elements();
					while (eDiffusion.hasMoreElements()) {
						final String codeGroupeDiffusion = eDiffusion.nextElement();
						if (ctx.getGroupesDsiAvecAscendants().contains(codeGroupeDiffusion)) {
							pushOK = true;
						}
					}
				}
				if (pushOK && (listeCodesServices.contains(service.getCode()) == false)) {
					listeCodesServices.add(service.getCode());
					vListeServices.add(service);
				}
			}
		}
		return vListeServices;
	}

	/**
	 * Traitement des tags spécifiques aux services.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _texte
	 *            the _texte
	 * @param idPage
	 *            the id page
	 *
	 * @return the string
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String transformerTagsServices(final ContexteUniv _ctx, final String _texte, final String idPage) throws Exception {
		return transformerTagsServices(_ctx, _texte, idPage, true);
	}

	/**
	 * Traitement des tags spécifiques aux services.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _texte
	 *            the _texte
	 * @param idPage
	 *            the id page
	 * @param bInterpreterPortlets
	 *            the b interpreter portlets
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String transformerTagsServices(final ContexteUniv _ctx, final String _texte, final String idPage, final boolean bInterpreterPortlets) throws Exception {
		// Liste des services à exécuter
		final Map<String, RequeteService> requetesServices = new Hashtable<String, RequeteService>();
		final Map<String, Valeur> occurencesService = new Hashtable<String, Valeur>();
		String res = transformerTagsServicesHorsContenu(_ctx, _texte, requetesServices, occurencesService, idPage, bInterpreterPortlets);
		if (requetesServices.size() > 0) {
			final Map<String, ReponseService> reponsesServices = new ServicesInvoker().executer(_ctx, requetesServices);
			// Remplacement des requêtes par leur réponse
			final Iterator<String> iter = reponsesServices.keySet().iterator();
			while (iter.hasNext()) {
				final String cleRequete = iter.next();
				final ReponseService reponse = reponsesServices.get(cleRequete);
				String contenu = "";
				if (reponse != null) {
					if (reponse.isTerminated()) {
						contenu = reponse.getReponse();
					} else {
						contenu = MessageHelper.getCoreMessage("ST_SERVICES_DELAI_DEPASSE");
					}
				}
				res = StringUtils.replace(res, cleRequete, contenu);
			}
		}
		return res;
	}

	/**
	 * Traitement des tags spécifiques aux services (sauf les contenus qui seront interprétés dans des threads).
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _texte
	 *            the _texte
	 * @param _requetesServices
	 *            the _requetes services
	 * @param occurencesService
	 *            the occurences service
	 * @param idPage
	 *            the id page
	 * @param bInterpreterPortlets
	 *            the b interpreter portlets
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String transformerTagsServicesHorsContenu(final ContexteUniv _ctx, final String _texte, final Map<String, RequeteService> _requetesServices,
		final Map<String, Valeur> occurencesService, final String idPage, final boolean bInterpreterPortlets) throws Exception {
		String texte = _texte;
		int indexTexte = 0;
		int indexDebutMotCle = -1;
		int indexFinMotCle = 0;
		/** ********************************************* */
		/* Condition d'insertion du service */
		/** ********************************************* */
		indexTexte = 0;
		indexDebutMotCle = -1;
		indexFinMotCle = 0;
		StringBuffer newTexte = new StringBuffer(texte.length() + 100);
		/* Boucle sur chaque mot clé */
		while ((indexDebutMotCle = texte.indexOf("[service_debut", indexTexte)) != -1) {
			// Recopie portion avant le mot- clé
			newTexte.append(texte.substring(indexTexte, indexDebutMotCle));
			// Extraction de la chaine
			indexFinMotCle = texte.indexOf("]", indexDebutMotCle);
			final String contenuTag = Chaine.remplacerPointsVirgules(texte.substring(indexDebutMotCle + 1, indexFinMotCle));
			// Analyse du code service
			String codeService = "";
			final StringTokenizer st = new StringTokenizer(contenuTag, "*");
			int indiceToken = 0;
			while (st.hasMoreTokens()) {
				final String itemTag = st.nextToken();
				if (indiceToken == 1) {
					codeService = itemTag;
				}
				indiceToken++;
			}
			// Recherche du tag de fin
			final String tagFinGroupe = "[service_fin]";
			final int indexFinGroupe = texte.indexOf(tagFinGroupe, indexDebutMotCle);
			// Fin du traitement si tag incomplet
			if (indexFinGroupe == -1) {
				indexTexte = indexDebutMotCle;
				break;
			}
			final int indexDebutMotCleSuivant = texte.indexOf("[service_debut", indexDebutMotCle + 1);
			if ((indexDebutMotCleSuivant != -1) && (indexDebutMotCleSuivant < indexFinGroupe)) {
				indexTexte = indexDebutMotCle;
				break;
			}
			if ((codeService.length() > 0)) {
				//si le service est en encadre, on interprète pas les portlets
				if (_ctx.getListeServicesAutorises().contains(codeService) && (bInterpreterPortlets || !getService(codeService).getType().equals("P"))) {
					_ctx.setServiceCourant(codeService);
					newTexte.append(transformerTagsServicesHorsContenu(_ctx, texte.substring(indexFinMotCle + 1, indexFinGroupe), _requetesServices, occurencesService, idPage,
						bInterpreterPortlets));
					_ctx.setServiceCourant("");
				}
			}
			// Repositionnement apres les services
			indexFinMotCle = texte.indexOf("]", indexFinGroupe);
			indexTexte = indexFinMotCle + 1;
		}
		if (indexTexte < texte.length()) {
			newTexte.append(texte.substring(indexTexte));
		}
		/** ************************** */
		/* Transformation des tags */
		/* spécifique à un service */
		/** ************************** */
		if (_ctx.getServiceCourant().length() > 0) {
			texte = new String(newTexte);
			indexTexte = 0;
			indexDebutMotCle = -1;
			indexFinMotCle = 0;
			newTexte = new StringBuffer(texte.length() + 100);
			final ServiceBean service = getService(_ctx.getServiceCourant());
			while ((indexDebutMotCle = texte.indexOf("[", indexTexte)) != -1) {
				// Recopie portion avant le mot- clé
				newTexte.append(texte.substring(indexTexte, indexDebutMotCle));
				// Recherche de mot-cle
				indexFinMotCle = texte.indexOf("]", indexDebutMotCle);
				String valeurRemplacement = "";
				boolean estUnMotcle = false;
				String motCle = "";
				if (indexFinMotCle != -1) {
					motCle = texte.substring(indexDebutMotCle + 1, indexFinMotCle);
					if (motCle.startsWith("service_contenu")) {
						String nomVue = "";
						final int idxPtV = motCle.indexOf(";");
						if (idxPtV != -1) {
							nomVue = motCle.substring(idxPtV + 1);
						}
						final int iRequete = _requetesServices.size();
						final String cleRequete = "#requete#" + iRequete + "#";
						_requetesServices.put(cleRequete, new RequeteService(service, nomVue, ""));
						valeurRemplacement = cleRequete;
						estUnMotcle = true;
					} else if (motCle.startsWith("service_link")) {
						String nomLink = "";
						final int idx = motCle.indexOf(";");
						if (idx != -1) {
							nomLink = motCle.substring(idx + 1);
						} else {
							// nom du lien non renseigné ==> on prend le nom du service
							nomLink = service.getIntitule();
						}
						final String target = determinerTargetService(_ctx, service);
						valeurRemplacement = "<a href=\"" + determinerUrlAccueilService(_ctx, service) + "\"";
						if (target.length() > 0) {
							valeurRemplacement += " onclick=\"window.open(this.href, '" + target + "'); return false;\"";
						}
						valeurRemplacement += ">" + nomLink + "</a>";
						estUnMotcle = true;
					} else if (motCle.equals("service_url")) {
						valeurRemplacement = determinerUrlAccueilService(_ctx, service);
						estUnMotcle = true;
					} else if (motCle.equals("service_target")) {
						valeurRemplacement = determinerTargetService(_ctx, service);
						estUnMotcle = true;
					}
				}
				if (estUnMotcle) {
					newTexte.append(valeurRemplacement);
					indexTexte = indexFinMotCle + 1;
				} else {
					newTexte.append("[");
					indexTexte = indexDebutMotCle + 1;
				}
			} // Recopie dernière portion de texte
			newTexte.append(texte.substring(indexTexte));
		}
		/* Analyse des tags [service;CODE] */
		texte = new String(newTexte);
		indexTexte = 0;
		indexDebutMotCle = -1;
		indexFinMotCle = 0;
		newTexte = new StringBuffer(texte.length() + 100);
		while ((indexDebutMotCle = texte.indexOf("[service", indexTexte)) != -1) {
			// Recopie portion avant le mot- clé
			newTexte.append(texte.substring(indexTexte, indexDebutMotCle));
			// Recherche de mot-cle
			indexFinMotCle = texte.indexOf("]", indexDebutMotCle);
			String valeurRemplacement = "";
			boolean estUnMotcle = false;
			String motCle = "";
			if (indexFinMotCle != -1) {
				motCle = texte.substring(indexDebutMotCle + 1, indexFinMotCle);
				if (motCle.startsWith("service")) {
					String codeService = "";
					String nomVue = "";
					final String tokens[] = motCle.split(";", -2);
					try {
						if (tokens.length > 1) {
							codeService = tokens[1];
						}
						if (tokens.length > 2) {
							nomVue = tokens[2];
						}
						final ServiceBean service = getService(codeService);
						if (_ctx.getListeServicesAutorises().contains(codeService) && (bInterpreterPortlets || !service.getType().equals("P"))) {
							/* Déclaration de l'occurence (jsr 168) du service */
							String codePortlet = "";
							if (service.getType().equals("P")) {
								if (occurencesService.containsKey(codeService)) {
									occurencesService.get(codeService).augmenterIndice(1);
								} else {
									occurencesService.put(codeService, new Valeur());
								}
								codePortlet = service.getCode() + "_occ_" + idPage + "_" + occurencesService.get(codeService).toString();
								PortletOccurenceRegistry.ajouterFragment(codePortlet);
							}
							final int iRequete = _requetesServices.size();
							final String cleRequete = "#requete#" + iRequete + "#";
							_requetesServices.put(cleRequete, new RequeteService(service, nomVue, codePortlet));
							valeurRemplacement = cleRequete;
							// Ajout header, footer sauf pour les portlets
							if (!service.getType().equals("P")) {
								valeurRemplacement = getHeaderService(_ctx, service) + valeurRemplacement + getFooterService(_ctx);
							}
						} else {
							// Pas de droits d'accès au service
							valeurRemplacement = "";
						}
						estUnMotcle = true;
					} catch (final PortletException ex) {
						//Exception lors de la création du fragment portlet.
						newTexte.append("Le service '" + codeService + "' n'existe pas.");
					}
				}
			}
			if (estUnMotcle) {
				newTexte.append(valeurRemplacement);
				indexTexte = indexFinMotCle + 1;
			} else {
				newTexte.append("[");
				indexTexte = indexDebutMotCle + 1;
			}
		} // Recopie dernière portion de texte
		newTexte.append(texte.substring(indexTexte));
		texte = new String(newTexte);
		return texte;
	}

	/**
	 * Ajout paramètres standard pour un service.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _service
	 *            the _service
	 * @param url
	 *            the url
	 * @param ticket
	 *            the ticket
	 *
	 * @return the string
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String ajouterParametresService(final ContexteUniv _ctx, final ServiceBean _service, final String url, final String ticket) throws Exception {
		// JB 20060206 : ajout de parametrage specifique pour les services
		// soit une implementation par service, soit une implementation pour tous les services
		String nomClasseParametrage = PropertyHelper.getCoreProperty(IParametrageService.PARAM_JTF_SERVICE_CLASSE_DEBUT + "." + _service.getCode() + IParametrageService.PARAM_JTF_SERVICE_CLASSE_FIN);
		if (nomClasseParametrage == null) {
			nomClasseParametrage = PropertyHelper.getCoreProperty(IParametrageService.PARAM_JTF_SERVICE_CLASSE);
		}
		if (nomClasseParametrage != null) {
			final IParametrageService parametrageServiceSpecif = (IParametrageService) Class.forName(nomClasseParametrage).newInstance();
			return parametrageServiceSpecif.getUrlParametree(_ctx, _service, url, ticket);
		}
		return new DefautParametrageService().getUrlParametree(_ctx, _service, url, ticket);
	}

	/**
	 * Détermine pour un service l'url permettant d'accéder à ce service (redirige le cas échéant sur la page de génération de ticket).
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _service
	 *            the _service
	 *
	 * @return @throws Exception
	 *
	 */
	public static String determinerUrlAccueilService(final ContexteUniv _ctx, final ServiceBean _service) {
		String url = "";
		if (_service.getType().equals("U")) {
			url = URL_ACTIVATION_SERVICE + _service.getCode();
		} else if ("P".equals(_service.getType())) {
			//service portlet
			url = "/portal/_st_" + _service.getCode() + "/maximized?AFFICHAGE=unitaire&amp;CODE_SERVICE=" + _service.getCode();
		}
		return url;
	}

	/**
	 * Détermine l'url de base d'un service de type url.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _service
	 *            the _service
	 *
	 * @return the string
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String determinerUrlServiceTypeUrl(final ContexteUniv _ctx, final ServiceBean _service) throws Exception {
		String url = _service.getUrl();
		if (url.length() > 0) {
			String kticket = "";
			if (_service.getJetonKportal().equals("1") || _ctx.getKsession().length() > 0) {
				kticket = ServicesUtil.genererTicketService(_ctx);
			}
			url = ajouterParametresService(_ctx, _service, url, kticket);
			// 20051220 JB : mode proxy specifie
			if ("1".equals(_service.getProxyCas())) {
				final ValidateurCAS validateurCAS = GestionnaireIdentification.getInstance().getValidateurCAS();
				if (validateurCAS != null) {
					url = validateurCAS.getUrlCasServiceLogin(url);
				}
			}
		}
		LOG.debug("*** url service appelee : " + url);
		return url;
	}

	/**
	 * Valider ticket.
	 *
	 * @param kticket
	 *            the kticket
	 *
	 * @return the string
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String validerTicket(final String kticket) throws Exception {
		String ksession = "";
		// Lecture du ticket pour en déduire le ksession
		final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		final DocumentBuilder builder = factory.newDocumentBuilder();
		final File f = new File(WebAppUtil.getSessionsPath() + File.separator + "kticket_" + kticket + ".xml");
		if (f.exists()) {
			final Document document = builder.parse(f);
			final Element rootElement = document.getDocumentElement();
			final Node donnees = rootElement;
			if (donnees.hasChildNodes()) {
				Node nextFeuille = donnees.getFirstChild();
				while (nextFeuille != null) {
					if (nextFeuille.getNodeName().equals("KSESSION")) {
						ksession = NodeUtil.extraireValeurNode(nextFeuille);
					}
					nextFeuille = nextFeuille.getNextSibling();
				}
			}
			// On supprime le ticket
			f.delete();
		} else {
			LOG.error("Le fichier " + f.getName() + " est introuvable.");
		}
		return ksession;
	}

	/**
	 * Supprime le ticket.
	 *
	 * @param kticket
	 *            the kticket
	 *
	 */
	public static void supprimerTicket(final String kticket) {
		final File f = new File(WebAppUtil.getSessionsPath() + File.separator + "kticket_" + kticket + ".xml");
		if (f.exists()) {
			f.delete();
		}
	}

	/**
	 * Gets the sSO bean.
	 *
	 * @param request
	 *            the request
	 *
	 * @return the sSO bean
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static SSOBean getSSOBean(final HttpServletRequest request) throws Exception {
		SSOBean ssoBean = null;
		// appel du serveur distant
		if ((request.getParameter("ksite") != null && request.getParameter("ksite").length() > 0) || (request.getParameter("kportal_host") != null && request.getParameter(
			"kportal_host").length() > 0)) {
			ssoBean = getSSOBeanFromRequest(request);
		}
		// validation local
		else {
			final String ksession = validerTicket(request.getParameter("kticket"));
			// Lecture du Bean SSO
			ssoBean = RequeteConnecteur.lireBeanSSO(ksession);
		}
		return ssoBean;
	}

	/**
	 * Détermine pour un service la target du service permettant d'accéder à ce service.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _service
	 *            the _service
	 *
	 * @return le code du service si url popup = 1 ?!?
	 */
	public static String determinerTargetService(final ContexteUniv _ctx, final ServiceBean _service) {
		String target = "";
		if (_service.getUrlPopup().equals("1")) {
			target = _service.getCode();
		}
		return target;
	}

	/**
	 * Ajoute le header d'un service.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param _service
	 *            the _service
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	private static String getHeaderService(final ContexteUniv _ctx, final ServiceBean _service) throws Exception {
		final HttpServletRequestWrapper wrappedRequest = new HttpServletRequestWrapper(_ctx.getRequeteHTTP());
		final StringWriter storedWriter = new StringWriter();
		final PrintWriter writer = new PrintWriter(storedWriter);
		final HttpServletResponseWrapper wrappedResponse = new ServiceWrappedResponse(_ctx.getReponseHTTP(), writer);
		final ServletContext ctxServlet = _ctx.getServletContext();
		wrappedRequest.setAttribute("titre", _service.getIntitule());
		final RequestDispatcher rd = ctxServlet.getRequestDispatcher("/adminsite/portal/service_header.jsp");
		rd.include(wrappedRequest, wrappedResponse);
		return storedWriter.toString();
	}

	/**
	 * Ajoute le foort d'un service.
	 *
	 * @param _ctx
	 *            the _ctx
	 *
	 * @return @throws Exception
	 *
	 * @throws Exception
	 *             the exception
	 */
	private static String getFooterService(final ContexteUniv _ctx) throws Exception {
		final HttpServletRequestWrapper wrappedRequest = new HttpServletRequestWrapper(_ctx.getRequeteHTTP());
		final StringWriter storedWriter = new StringWriter();
		final PrintWriter writer = new PrintWriter(storedWriter);
		final HttpServletResponseWrapper wrappedResponse = new ServiceWrappedResponse(_ctx.getReponseHTTP(), writer);
		final ServletContext ctxServlet = _ctx.getServletContext();
		final RequestDispatcher rd = ctxServlet.getRequestDispatcher("/adminsite/portal/service_footer.jsp");
		rd.include(wrappedRequest, wrappedResponse);
		return storedWriter.toString();
	}

	/**
	 * Génération d'un ticket temporaire à partir duquel on retrouvera le ksession.
	 *
	 * @param _ctx
	 *            the _ctx
	 *
	 * @return the string
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static String genererTicketService(final ContexteUniv _ctx) throws Exception {
		final long random = (long) (Math.random() * 10000.);
		final String ticket = Long.toString(System.currentTimeMillis()) + "_" + random;
		if (StringUtils.isEmpty(_ctx.getKsession())) {
			throw new ErreurApplicative("Veuillez vous connecter");
		}
		final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		final DocumentBuilder builder = factory.newDocumentBuilder();
		final Document document = builder.newDocument();
		final Node nodeFiche = document.createElement("KTICKET");
		NodeUtil.addNode(nodeFiche, "KSESSION", _ctx.getKsession());
		document.appendChild(nodeFiche);
		final TransformerFactory transFactory = TransformerFactory.newInstance();
		final Transformer transformer = transFactory.newTransformer();
		final DOMSource source = new DOMSource(document);
		final File fichierSortie = new File(WebAppUtil.getSessionsPath() + File.separator + "kticket_" + ticket + ".xml");
		fichierSortie.createNewFile();
		try (FileWriter fw = new FileWriter(fichierSortie)) {
			final StreamResult sr = new StreamResult(fw);
			transformer.setOutputProperty(OutputKeys.ENCODING, "utf-8");
			transformer.transform(source, sr);
		}
		return ticket;
	}

	/**
	 * appel du SSO kportal.
	 *
	 * @param request
	 *            the request
	 *
	 * @return le bean SSO (code retour = '0' si OK)
	 *
	 * @throws Exception
	 *             the exception
	 */
	private static SSOBean getSSOBeanFromRequest(final HttpServletRequest request) throws Exception {
		final SSOBean bean = new SSOBean();
		final String kticket = request.getParameter("kticket");
		String urlServeur = request.getParameter("ksite");
		if (urlServeur == null || urlServeur.length() == 0) {
			urlServeur = request.getParameter("kportal_host");
		}
		// controle si site de confiance
		if (!validerUrl(urlServeur)) {
			bean.setCodeRetour("600");
			return bean;
		}
		final InfosSite infosSite = Site.renvoyerItemSiteParHost(request.getServerName());
		final String uri = "/adminsite/sso/validation_ticket.jsp?kticket=" + kticket + "&kreferer=" + URLResolver.getAbsoluteBoUrl("/", infosSite);
		final javax.xml.parsers.DocumentBuilderFactory factory = javax.xml.parsers.DocumentBuilderFactory.newInstance();
		final javax.xml.parsers.DocumentBuilder builder = factory.newDocumentBuilder();
		final String url = urlServeur + uri;
		final org.w3c.dom.Document document = builder.parse(url);
		final org.w3c.dom.Node arbreXML = document.getDocumentElement();
		final org.w3c.dom.NodeList nodeList = arbreXML.getChildNodes();
		if (nodeList != null) {
			/* Analyse du résultat */
			for (int i = 0; i < nodeList.getLength(); i++) {
				final org.w3c.dom.Node donnees = nodeList.item(i);
				if (donnees != null) {
					if (donnees.getFirstChild() != null) {
						if (donnees.getNodeName().compareTo("CIVILITE") == 0) {
							bean.setCivilite(donnees.getFirstChild().getNodeValue().trim());
						}
						if (donnees.getNodeName().compareTo("CODE_UTILISATEUR_KPORTAL") == 0) {
							bean.setCodeKportal(donnees.getFirstChild().getNodeValue().trim());
						}
						if (donnees.getNodeName().compareTo("CODE_GESTION_KPORTAL") == 0) {
							bean.setCodeGestion(donnees.getFirstChild().getNodeValue().trim());
						}
						if (donnees.getNodeName().compareTo("EMAIL") == 0) {
							bean.setEmail(donnees.getFirstChild().getNodeValue().trim());
						}
						if (donnees.getNodeName().compareTo("GROUPES") == 0) {
							bean.setGroupes(donnees.getFirstChild().getNodeValue().trim());
						}
						if (donnees.getNodeName().compareTo("NOM") == 0) {
							bean.setNom(donnees.getFirstChild().getNodeValue().trim());
						}
						if (donnees.getNodeName().compareTo("PRENOM") == 0) {
							bean.setPrenom(donnees.getFirstChild().getNodeValue().trim());
						}
						if (donnees.getNodeName().compareTo("PROFIL") == 0) {
							bean.setProfil(donnees.getFirstChild().getNodeValue().trim());
						}
						if (donnees.getNodeName().compareTo("STRUCTURE") == 0) {
							bean.setStructure(donnees.getFirstChild().getNodeValue().trim());
						}
						if (donnees.getNodeName().compareTo("KSESSION") == 0) {
							bean.setKsession(donnees.getFirstChild().getNodeValue().trim());
						}
						if (donnees.getNodeName().compareTo("CODE_RETOUR") == 0) {
							bean.setCodeRetour(donnees.getFirstChild().getNodeValue().trim());
						}
					}
				}
			}
		}
		return bean;
	}

	/**
	 * Valider url.
	 *
	 * @param sUrl
	 *            the s url
	 *
	 * @return true, if successful
	 */
	public static boolean validerUrl(final String sUrl) {
		final boolean res = false;
		URL url = null;
		try {
			url = new URL(sUrl);
		} catch (final MalformedURLException e) {
			return false;
		}
		final String host = url.getHost();
		final String param = PropertyHelper.getCoreProperty("sso.sites_confiance");
		if (param != null) {
			final String aliasHost[] = param.split(";");
			for (final String element : aliasHost) {
				if (host.matches(element)) {
					return true;
				}
			}
		}
		return res;
	}

	/**
	 * Valider site.
	 *
	 * @param sUrl
	 *            the s url
	 * @return true, if successful
	 */
	public static boolean validerSite(final String sUrl) {
		URL url = null;
		try {
			url = new URL(sUrl);
		} catch (final MalformedURLException e) {
			return false;
		}
		return (Site.renvoyerItemSiteParHost(url.getHost()).getAlias().length() > 0);
	}

	/**
	 * Gets the service.
	 *
	 * @param codeService
	 *            the code service
	 *
	 * @return the service
	 */
	public static ServiceBean getService(final String codeService) {
		if (getServices().containsKey(codeService)) {
			return getServices().get(codeService);
		}
		return new ServiceBean();
	}

	/**
	 * modification de l'ordre des services.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param vCodesServices
	 *            the v codes services
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static synchronized void ordonnerInstancesServices(final OMContext _ctx, final List<String> vCodesServices) throws Exception {
		final Service service = new Service();
		service.setCtx(_ctx);
		int ordre = 0;
		final Iterator<String> iter = vCodesServices.iterator();
		while (iter.hasNext()) {
			final String codeService = iter.next();
			final ClauseWhere whereCodeService = new ClauseWhere(ConditionHelper.egalVarchar("CODE", codeService));
			if (service.select(whereCodeService.formaterSQL()) > 0) {
				service.nextItem();
				service.setOrdre(new Integer(ordre));
				service.update();
				ordre++;
			}
		}
	}

	/**
	 * Suppression des paramètres d'un service.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param codeService
	 *            the code service
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static synchronized void removeInstanceService(final OMContext _ctx, final String codeService) throws Exception {
		if (codeService != null) {
			final Service service = new Service();
			service.setCtx(_ctx);
			final ClauseWhere whereCodeService = new ClauseWhere(ConditionHelper.egalVarchar("CODE", codeService));
			if (service.select(whereCodeService.formaterSQL()) > 0) {
				service.nextItem();
				service.delete();
			}
			final Preferences pref = new Preferences();
			pref.setCtx(_ctx);
			pref.init();
			pref.deleteAllPreferences(codeService);
		}
	}

	/**
	 * Sauvegarde des paramètres d'un service.
	 *
	 * @param _ctx
	 *            the _ctx
	 * @param serviceBean
	 *            the service bean
	 *
	 * @throws Exception
	 *             the exception
	 */
	public static synchronized void setInstanceService(final OMContext _ctx, final ServiceBean serviceBean) throws Exception {
		final Service service = new Service();
		service.setCtx(_ctx);
		boolean creation = false;
		final ClauseWhere whereCodeService = new ClauseWhere(ConditionHelper.egalVarchar("CODE", serviceBean.getCode()));
		if (service.select(whereCodeService.formaterSQL()) > 0) {
			service.nextItem();
		} else {
			service.init();
			creation = true;
		}
		/* insertion services */
		service.setCode(serviceBean.getCode());
		service.setIntitule(serviceBean.getIntitule());
		service.setType(serviceBean.getType());
		service.setEncodage(serviceBean.getEncodage());
		service.setExpirationCache(serviceBean.getExpirationCache());
		service.setJetonKportal(serviceBean.getJetonKportal());
		service.setProxyCas(serviceBean.getProxyCas());
		service.setUrl(serviceBean.getUrl());
		service.setPortletDefinitionId(serviceBean.getPortletDefinitionId());
		service.setUrlPopup(serviceBean.getUrlPopup());
		service.setVueReduiteType(serviceBean.getVueReduiteType());
		service.setVueReduiteUrl(serviceBean.getVueReduiteUrl());
		service.setVueReduiteCtx(serviceBean.getVueReduiteCtx());
		service.setVueReduiteUrlCrossCtx(serviceBean.getVueReduiteUrlCrossCtx());
		service.setDiffusionMode(serviceBean.getDiffusionMode());
		service.setDiffusionPublicVise(Chaine.convertirAccolades(serviceBean.getDiffusionPublicVise()));
		service.setDiffusionModeRestriction(serviceBean.getDiffusionModeRestriction());
		service.setDiffusionPublicViseRestriction(Chaine.convertirAccolades(serviceBean.getDiffusionPublicViseRestriction()));
		if (creation) {
			service.add();
		} else {
			service.update();
		}
		final Preferences prefGenerales = new Preferences();
		prefGenerales.setCtx(_ctx);
		prefGenerales.init();
		// Suppression anciennes valeurs
		prefGenerales.selectPreferencesGenerales(service.getCode());
		while (prefGenerales.nextItem()) {
			prefGenerales.delete();
		}
		final List<InfosPreferences> listePreferencesGenerales = serviceBean.getPreferencesGenerales();
		final Iterator<InfosPreferences> iter = listePreferencesGenerales.iterator();
		InfosPreferences infosPref = null;
		while (iter.hasNext()) {
			infosPref = iter.next();
			prefGenerales.init();
			prefGenerales.setService(serviceBean.getCode());
			prefGenerales.setType(Preferences.TYPE_SERVICE);
			prefGenerales.setNom(infosPref.getNom());
			prefGenerales.setValeur(infosPref.getValeursAsString());
			prefGenerales.add();
		}
	}
}
