/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;

import java.io.File;
import java.io.IOException;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jsbsoft.jtf.lang.CharEncoding;
import com.kportal.servlet.ExtensionServlet;

public abstract class SimpleDownloadServlet extends ExtensionServlet {

    private static final long serialVersionUID = -2328660109979013684L;

    private static final Logger LOG = LoggerFactory.getLogger(SimpleDownloadServlet.class);

    // Le chemin absolu vers le fichier
    protected abstract String getFilePath();

    // Le type du fichier (ex : text/csv)
    protected abstract String getFileType();

    // Calcul de l'autorisation
    protected abstract boolean isAuthorized();

    // Calcul du nom de fichier proposé en téléchargement
    protected String getDownloadedFileName(File file) {
        return file.getName();
    }

    /**
     * Process incoming HTTP GET requests.
     *
     * @param request  Object that encapsulates the request to the servlet
     * @param response Object that encapsulates the response from the servlet
     * @throws ServletException the servlet exception
     * @throws IOException      Signals that an I/O exception has occurred.
     */
    @Override
    public void doGet(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
        performTask(request, response);
    }

    /**
     * Process incoming requests for information.
     *
     * @param request  Object that encapsulates the request to the servlet
     * @param response Object that encapsulates the response from the servlet
     */
    public void performTask(final HttpServletRequest request, final HttpServletResponse response) {
        int status = 0;
        ContexteUniv ctx = ContexteUtil.getContexteUniv();
        if (isAuthorized()) {
            final File f = new File(getFilePath());
            if (f.exists()) {
                try (final ServletOutputStream writer = response.getOutputStream()) {
                    String contentType = StringUtils.EMPTY;
                    if (StringUtils.isEmpty(getFileType())) {
                        contentType = "text/plain";
                    }
                    response.setContentType(contentType);
                    response.setCharacterEncoding(CharEncoding.DEFAULT.toLowerCase());
                    response.setHeader("Content-Disposition", "attachment;filename=\"" + getDownloadedFileName(f) + "\"");
                    FileUtils.copyFile(f, writer);
                } catch (final IOException e) {
                    LOG.error("Erreur lors de la lecture/ecriture du fichier", e);
                }
            } else if (getFilePath().length() > 0) {
                status = HttpServletResponse.SC_NOT_FOUND;
            }
        } else {
            status = HttpServletResponse.SC_FORBIDDEN;
        }

        if (status != 0) {
            response.setStatus(status);
            String pageErreur = "";
            if (status == HttpServletResponse.SC_NOT_FOUND) {
                pageErreur = ctx.getInfosSite().getJspFo() + "/error/404.jsp";
            } else {
                pageErreur = ctx.getInfosSite().getJspFo() + "/error/403.jsp";
            }
            try {
                final RequestDispatcher rd = getServletContext().getRequestDispatcher(pageErreur);
                rd.forward(request, response);
            } catch (final ServletException | IOException e) {
                LOG.error("erreur lors du forward de la request", e);
            }
        }
    }
}
