/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils.json;

import java.io.File;
import java.io.IOException;
import java.text.DateFormat;

import org.codehaus.jackson.JsonGenerationException;
import org.codehaus.jackson.JsonParseException;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.type.TypeReference;

/**
 * Classe permettant de manipuler des flux en JSON
 * 
 */
public class CodecJSon {

	public static File encodeObjectToJsonInFile(final String path, final Object obj, final DateFormat dateFormat, final boolean autoDetectGetters) throws JsonGenerationException,
		JsonMappingException, IOException {
		final File file = new File(path);
		if (!file.exists()) {
			file.getParentFile().mkdirs();
			file.createNewFile();
		}
		final ObjectMapper mapper = new ObjectMapper();
		if (dateFormat != null) {
			mapper.setDateFormat(dateFormat);
		}
		if (!autoDetectGetters) {
			mapper.configure(org.codehaus.jackson.map.SerializationConfig.Feature.AUTO_DETECT_GETTERS, autoDetectGetters);
			mapper.configure(org.codehaus.jackson.map.SerializationConfig.Feature.AUTO_DETECT_IS_GETTERS, autoDetectGetters);
			mapper.configure(org.codehaus.jackson.map.SerializationConfig.Feature.FAIL_ON_EMPTY_BEANS, false);
		}
		mapper.writeValue(file, obj);
		return file;
	}

	/**
	 * Retourne un {@link Object} encodé en json dans un {@link File}
	 * 
	 * @param obj
	 *            Object à encoder
	 * @param dateFormat
	 *            format de la date à utiliser
	 * @return object encodé
	 * @throws JsonGenerationException
	 * @throws JsonMappingException
	 * @throws IOException
	 */
	public static File encodeObjectToJsonInFile(final String path, final Object obj, final DateFormat dateFormat) throws Exception {
		return encodeObjectToJsonInFile(path, obj, dateFormat, true);
	}

	/**
	 * Retourne un {@link Object} encodé en json dans un {@link File}
	 * 
	 * @param obj
	 *            Object à encoder
	 * @param autoDetectGetters
	 *            desactive le parsing automatique des getters
	 * @return object encodé
	 * @throws JsonGenerationException
	 * @throws JsonMappingException
	 * @throws IOException
	 */
	public static File encodeObjectToJsonInFile(final String path, final Object obj, final boolean autoDetectGetters) throws Exception {
		return encodeObjectToJsonInFile(path, obj, null, autoDetectGetters);
	}

	/**
	 * Retourne un {@link Object} encodé en json dans un {@link File}
	 * 
	 * @param obj
	 *            Object à encoder
	 * @return object encodé
	 * @throws JsonGenerationException
	 * @throws JsonMappingException
	 * @throws IOException
	 */
	public static File encodeObjectToJsonInFile(final String path, final Object obj) throws Exception {
		return encodeObjectToJsonInFile(path, obj, null, true);
	}

	/**
	 * Retourne un {@link Object} encodé en json dans un {@link String}
	 * 
	 * @param obj
	 *            Object à encoder
	 * @param dateFormat
	 *            format de la date à utiliser
	 * @param autoDetectGetters
	 *            desactive le parsing automatique des getters
	 * @return object encodé
	 * @throws JsonGenerationException
	 * @throws JsonMappingException
	 * @throws IOException
	 */
	public static String encodeObjectToJSonInString(final Object obj, final DateFormat dateFormat, final boolean autoDetectGetters) throws Exception {
		final ObjectMapper mapper = new ObjectMapper();
		if (dateFormat != null) {
			mapper.setDateFormat(dateFormat);
		}
		if (!autoDetectGetters) {
			mapper.configure(org.codehaus.jackson.map.SerializationConfig.Feature.AUTO_DETECT_GETTERS, autoDetectGetters);
			mapper.configure(org.codehaus.jackson.map.SerializationConfig.Feature.AUTO_DETECT_IS_GETTERS, autoDetectGetters);
			mapper.configure(org.codehaus.jackson.map.SerializationConfig.Feature.FAIL_ON_EMPTY_BEANS, false);
		}
		return mapper.writeValueAsString(obj);
	}

	/**
	 * Retourne un {@link Object} encodé en json dans un {@link String}
	 * 
	 * @param obj
	 *            Object à encoder
	 * @param dateFormat
	 *            format de la date à utiliser
	 * @return object encodé
	 * @throws JsonGenerationException
	 * @throws JsonMappingException
	 * @throws IOException
	 */
	public static String encodeObjectToJSonInString(final Object obj, final DateFormat dateFormat) throws Exception {
		return encodeObjectToJSonInString(obj, dateFormat, true);
	}

	/**
	 * Retourne un {@link Object} encodé en json dans un {@link String}
	 * 
	 * @param obj
	 *            Object à encoder
	 * @param autoDetectGetters
	 *            desactive le parsing automatique des getters
	 * @return object encodé
	 * @throws JsonGenerationException
	 * @throws JsonMappingException
	 * @throws IOException
	 */
	public static String encodeObjectToJSonInString(final Object obj, final boolean autoDetectGetters) throws Exception {
		return encodeObjectToJSonInString(obj, null, autoDetectGetters);
	}

	/**
	 * Retourne un {@link Object} encodé en json dans un {@link String}
	 * 
	 * @param obj
	 *            Object à encoder
	 * @return object encodé
	 * @throws JsonGenerationException
	 * @throws JsonMappingException
	 * @throws IOException
	 */
	public static String encodeObjectToJSonInString(final Object obj) throws Exception {
		return encodeObjectToJSonInString(obj, null, true);
	}

	/**
	 * Retourne la valeur dans la class passée en paramètre (dans le typeReference)
	 * 
	 * <br/>
	 * (Utile pour les types complexe)
	 * 
	 * @param valeur
	 *            valeur JSon
	 * @param typeReference
	 *            référence le type de retour (exemples: new TypeReference<MaClass>(){} / new TypeReference<List<MaClass>>(){})
	 * @return L'object valeur décodé dans la classe
	 * @throws JsonParseException
	 * @throws JsonMappingException
	 * @throws IOException
	 */
	public static <T> T decodeStringJSonToClass(final String valeur, final TypeReference<T> typeReference) throws Exception {
		final ObjectMapper mapper = new ObjectMapper();
		return mapper.readValue(valeur, typeReference);
	}

	public static <T> T decodeStringJSonToClass(final String valeur, final Class<T> typeClass) throws Exception {
		final ObjectMapper mapper = new ObjectMapper();
		return mapper.readValue(valeur, typeClass);
	}

	public static <T> T decodeFileJSonToClass(final File valeur, final TypeReference<T> typeReference) throws Exception {
		final ObjectMapper mapper = new ObjectMapper();
		return mapper.readValue(valeur, typeReference);
	}

	public static <T> T decodeFileJSonToClass(final File valeur, final Class<T> typeClass) throws Exception {
		final ObjectMapper mapper = new ObjectMapper();
		return mapper.readValue(valeur, typeClass);
	}
}
