/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils.recherche;

import static org.apache.commons.lang3.StringUtils.isNotEmpty;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.database.OMContext;
import com.jsbsoft.jtf.exception.ErreurApplicative;
import com.kportal.cms.objetspartages.annotation.FicheAnnotationHelper;
import com.kportal.core.config.MessageHelper;
import com.kportal.extension.module.plugin.rubrique.BeanPageAccueil;
import com.kportal.extension.module.plugin.rubrique.FichePageAccueilRubrique.BeanFichePageAccueil;
import com.kportal.extension.module.plugin.rubrique.PageAccueilRubriqueManager;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.Metatag;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.url.FrontOfficeMgr;
import com.univ.utils.FicheUnivMgr;
import com.univ.utils.sql.RequeteSQL;
import com.univ.utils.sql.clause.ClauseJoin;
import com.univ.utils.sql.clause.ClauseJoin.TypeJointure;
import com.univ.utils.sql.clause.ClauseWhere;
import com.univ.utils.sql.condition.Condition;
import com.univ.utils.sql.condition.ConditionList;
import com.univ.utils.sql.criterespecifique.ConditionHelper;
import com.univ.utils.sql.criterespecifique.LimitHelper;
import com.univ.utils.sql.criterespecifique.RequeteSQLHelper;
import com.univ.utils.sql.operande.TypeOperande;

/**
 * Classe helper de recherche de fiche univ avec critères multiples (recherche directe)
 *
 */
public class RechercheMultificheHelper {

	public static ResultatRechercheMultifiche rerchercherParmisToutesLesFiches(final OMContext ctx, final AutorisationBean autorisations, final RequeteMultifiche requete)
		throws Exception {
		return rerchercherParmisToutesLesFiches(ctx, autorisations, requete.getsLibelle(), requete.getsCodeObjet(), requete.getsCodeFiche(), requete.getsCodeRubrique(),
			requete.getsCodeRattachement(), requete.getsCodeRedacteur(), requete.getsIdMeta(), requete.getUrlFiche(), requete.getDateDebutCreation(), requete.getDateFinCreation(),
			requete.getDateDebutModification(), requete.getDateFinModification(), requete.getDateDebutMiseEnLigne(), requete.getDateFinMiseEnLigne());
	}

	public static ResultatRechercheMultifiche rerchercherParmisToutesLesFiches(final OMContext ctx, final AutorisationBean autorisations, final String sLibelle,
		final String sCodeObjet, final String sCodeFiche, final String sCodeRubrique, final String sCodeRattachement, final String sCodeRedacteur, final String urlFiche,
		final Date dateDebutCreation, final Date dateFinCreation, final Date dateDebutModification, final Date dateFinModification, final Date dateDebutMiseEnLigne,
		final Date dateFinMiseEnLigne) throws Exception {
		return rerchercherParmisToutesLesFiches(ctx, autorisations, sLibelle, sCodeObjet, sCodeFiche, sCodeRubrique, sCodeRattachement, sCodeRedacteur, "", urlFiche,
			dateDebutCreation, dateFinCreation, dateDebutModification, dateFinModification, dateDebutMiseEnLigne, dateFinMiseEnLigne);
	}

	public static ResultatRechercheMultifiche rerchercherParmisToutesLesFiches(final OMContext ctx, final AutorisationBean autorisations, final String sLibelle,
		final String sCodeObjet, final String sCodeFiche, final String sCodeRubrique, final String sCodeRattachement, final String sCodeRedacteur, final String sIdMeta,
		final String urlFiche, final Date dateDebutCreation, final Date dateFinCreation, final Date dateDebutModification, final Date dateFinModification,
		final Date dateDebutMiseEnLigne, final Date dateFinMiseEnLigne) throws Exception {
		return rerchercherParmisToutesLesFiches(ctx, autorisations, sLibelle, sCodeObjet, sCodeFiche, sCodeRubrique, sCodeRattachement, sCodeRedacteur, sIdMeta, urlFiche,
			dateDebutCreation, dateFinCreation, dateDebutModification, dateFinModification, dateDebutMiseEnLigne, dateFinMiseEnLigne, StringUtils.EMPTY, StringUtils.EMPTY,
			StringUtils.EMPTY);
	}

	/**
	 * Execute une recherche de fiches univ multicritères. Execute en priorité une recherche sur l'idMeta si il est présent, sur urlFiche si elle est présente et que c'est une url
	 * de fiche (pas une url de rubrique) et sur les autres critères sinon
	 *
	 * @param ctx
	 *            Le contexte KPortal
	 * @param autorisations
	 *            Une instance de AutorisationBean permettant de ne ramener que les fiches autorisées
	 * @param sLibelle
	 *            Le libelle de la fiche univ recherchée
	 * @param sCodeObjet
	 *            Le code objet de la fiche univ recherchée
	 * @param sCodeFiche
	 *            Le code fiche de la fiche univ recherchée
	 * @param sCodeRubrique
	 *            Le code de la rubrique de la fiche univ recherchée
	 * @param sCodeRattachement
	 *            Le code rattachement de la fiche univ recherchée
	 * @param sCodeRedacteur
	 *            Le code rédacteur de la fiche univ recherchée
	 * @param sIdMeta
	 *            L'id du meta de la fiche univ recherchée
	 * @param urlFiche
	 *            L'url de la fiche univ recherchée
	 * @param dateDebutCreation
	 *            La borne inférieure de la date de création de la fiche univ recherchée
	 * @param dateFinCreation
	 *            La borne supérieure de la date de création de la fiche univ recherchée
	 * @param dateDebutModification
	 *            La borne inférieure de la date de modification de la fiche univ recherchée
	 * @param dateFinModification
	 *            La borne supérieure de la date de modification de la fiche univ recherchée
	 * @param dateDebutMiseEnLigne
	 *            La borne inférieure de la date de mise en ligne de la fiche univ recherchée
	 * @param dateFinMiseEnLigne
	 *            La borne supérieure de la date de mise en ligne de la fiche univ recherchée
	 * @return Un objet ResultatRechercheMultifiche contenant les résultats de la recherche
	 * @throws ErreurApplicative
	 * @throws Exception
	 */
	public static ResultatRechercheMultifiche rerchercherParmisToutesLesFiches(final OMContext ctx, final AutorisationBean autorisations, final String sLibelle, String sCodeObjet,
		String sCodeFiche, final String sCodeRubrique, final String sCodeRattachement, final String sCodeRedacteur, final String sIdMeta, final String urlFiche,
		final Date dateDebutCreation, final Date dateFinCreation, final Date dateDebutModification, final Date dateFinModification, final Date dateDebutMiseEnLigne,
		final Date dateFinMiseEnLigne, final String langue, final String etatObjet, final String nombre) throws ErreurApplicative, Exception {
		final Collection<Metatag> lesMetas = new ArrayList<>();
		// On enleve les paramètres de requete de l'url
		final String urlFicheSansParametres = StringUtils.substringBefore(urlFiche, "?");
		// l'url est renseignee, on trouve la fiche à partir de l'id meta
		// present
		// dans l'url (si c'est une url de fiche)
		if (StringUtils.isNotBlank(urlFicheSansParametres) && !urlFicheSansParametres.endsWith("/")) {
			try {
				final Metatag metatagRecherche = rechercheMetaParUrlFiche(ctx, autorisations, urlFicheSansParametres);
				if (metatagRecherche != null) {
					lesMetas.add(metatagRecherche);
				}
			} catch (final Exception e) {
				throw new ErreurApplicative(MessageHelper.getCoreMessage("ST_RECHERCHE_DIRECTE_URL_INVALIDE"));
			}
		}
		// Sinon formulaire de recherche transversale
		else {
			// Dans le cas ou une url de rubrique est saisie on récupere le
			// codeFiche et le codeObjet en fonction de l'url
			if (StringUtils.isNotBlank(urlFicheSansParametres)) {
				// url de rubrique
				if (urlFicheSansParametres.endsWith("/")) {
					final InfosRubriques infosRubriques = FrontOfficeMgr.getInstance().getInfosRubriqueByUrl(urlFicheSansParametres);
					final BeanPageAccueil beanAccueil = PageAccueilRubriqueManager.getInstance().getBeanPageAccueil(infosRubriques);
					if (beanAccueil != null && beanAccueil instanceof BeanFichePageAccueil) {
						sCodeFiche = ((BeanFichePageAccueil) beanAccueil).getCode();
						sCodeObjet = ReferentielObjets.getCodeObjet(((BeanFichePageAccueil) beanAccueil).getObjet());
					}
				}
			}
			// execution de la recherche multicriteres
			lesMetas.addAll(rechercherParCriteresMultiples(ctx, autorisations, sLibelle, sCodeObjet, sCodeFiche, sCodeRubrique, sCodeRattachement, sCodeRedacteur,
				dateDebutCreation, dateFinCreation, dateDebutModification, dateFinModification, dateDebutMiseEnLigne, dateFinMiseEnLigne, langue, etatObjet, nombre));
		}
		final ResultatRechercheMultifiche result = new ResultatRechercheMultifiche();
		result.getResultats().addAll(lesMetas);
		result.setCodeObjet(sCodeObjet);
		return result;
	}

	/**
	 * Recherche un Metatag en fonction de l'url d'une fiche univ et des autorisations sur la fiche correspondante.
	 *
	 * @param ctx
	 *            Le contexte KPortal.
	 * @param autorisations
	 *            L'objet AutorisationBean qui permettra de vérifier les droits sur la fiche liée au méta.
	 * @param urlFiche
	 *            L'url de la fiche dont on recherche le metatag.
	 * @return Le Metatag correspondant à la recherche ou null.
	 * @throws Exception
	 */
	private static Metatag rechercheMetaParUrlFiche(final OMContext ctx, final AutorisationBean autorisations, final String urlFiche) throws Exception {
		if (!urlFiche.contains("-")) {
			return null;
		}
		final Long idMeta = new Long(urlFiche.substring(urlFiche.lastIndexOf("-") + 1, urlFiche.lastIndexOf(".")));
		return rechercheMetaParIdMeta(ctx, autorisations, idMeta);
	}

	/**
	 * Recherche un Metatag en fonction de son id et des autorisations sur la fiche correspondante.
	 *
	 * @param ctx
	 *            Le contexte KPortal
	 * @param autorisations
	 *            L'objet AutorisationBean qui permettra de vérifier les droits sur la fiche liée au méta
	 * @param idMeta
	 *            L'id du Metatag recherché
	 * @return Le Metatag correspondant à la recherche ou null
	 * @throws Exception
	 */
	private static Metatag rechercheMetaParIdMeta(final OMContext ctx, final AutorisationBean autorisations, final Long idMeta) throws Exception {
		final Metatag meta = new Metatag();
		meta.setCtx(ctx);
		meta.init();
		meta.setIdMetatag(idMeta);
		try {
			meta.retrieve();
		} catch (final Exception e) {
			return null;
		}
		final FicheUniv ficheUniv = FicheUnivMgr.init(meta);
		if (!autorisations.estAutoriseAModifierLaFiche(ficheUniv)) {
			throw new ErreurApplicative(MessageHelper.getCoreMessage("ST_RECHERCHE_DIRECTE_OPERATION_INTERDITE"));
		}
		return meta;
	}

	/**
	 * Ajoute un critere limitant la recherche sur colonne de type date bornée par les deux dates en paramètre
	 *
	 * @param nomColonne
	 *            Le nom de la colonne date que l'on veut bornée
	 * @param dateDebut
	 *            La borne inferieure (exclusive)
	 * @param dateFin
	 *            La borne superieure(exclusive)
	 * @return La condition créé.
	 */
	private static Condition traiterDate(final String nomColonne, final Date dateDebut, final Date dateFin) {
		final ConditionList conditionSurDate = new ConditionList();
		if (dateDebut != null) {
			conditionSurDate.setPremiereCondtion(ConditionHelper.greaterThan(nomColonne, dateDebut, TypeOperande.DATE));
		}
		if (dateFin != null) {
			conditionSurDate.and(ConditionHelper.lessThan(nomColonne, dateFin, TypeOperande.DATE));
		}
		return conditionSurDate;
	}

	/**
	 * Execute une recherche de fiches univ multicritères.
	 *
	 * @param ctx
	 *            Le contexte KPortal
	 * @param autorisations
	 *            Une instance de AutorisationBean permettant de ne ramener que les fiches autorisées
	 * @param libelle
	 *            Le libelle de la fiche univ recherchée
	 * @param codeObjet
	 *            Le code objet de la fiche univ recherchée
	 * @param codeFiche
	 *            Le code fiche de la fiche univ recherchée
	 * @param codeRubrique
	 *            Le code de la rubrique de la fiche univ recherchée
	 * @param codeRattachement
	 *            Le code rattachement de la fiche univ recherchée
	 * @param codeRedacteur
	 *            Le code rédacteur de la fiche univ recherchée
	 * @param debutCreation
	 *            La borne inférieure de la date de création de la fiche univ recherchée
	 * @param finCreation
	 *            La borne supérieure de la date de création de la fiche univ recherchée
	 * @param debutModification
	 *            La borne inférieure de la date de modification de la fiche univ recherchée
	 * @param finModification
	 *            La borne supérieure de la date de modification de la fiche univ recherchée
	 * @param debutMiseEnLigne
	 *            La borne inférieure de la date de mise en ligne de la fiche univ recherchée
	 * @param finMiseEnLigne
	 *            La borne supérieure de la date de mise en ligne de la fiche univ recherchée
	 * @return Un objet ResultatRechercheMultifiche contenant les résultats de la recherche
	 * @throws Exception
	 */
	private static Collection<Metatag> rechercherParCriteresMultiples(final OMContext ctx, final AutorisationBean autorisations, final String libelle, final String codeObjet,
		final String codeFiche, final String codeRubrique, final String codeRattachement, final String codeRedacteur, final Date debutCreation, final Date finCreation,
		final Date debutModification, final Date finModification, final Date debutMiseEnLigne, final Date finMiseEnLigne, final String langue, final String etatObjet,
		final String nombre) throws Exception {
		final RequeteSQL requeteSelect = new RequeteSQL();
		final ClauseWhere where = new ClauseWhere();
		if (isNotEmpty(libelle)) {
			where.setPremiereCondition(ConditionHelper.rechercheMots("T1.META_LIBELLE_FICHE", libelle));
		}
		if (isNotEmpty(codeRedacteur)) {
			where.and(ConditionHelper.egalVarchar("T1.META_CODE_REDACTEUR", codeRedacteur));
		}
		if (isNotEmpty(codeFiche)) {
			where.and(ConditionHelper.egalVarchar("T1.META_CODE", codeFiche));
		}
		if (isNotEmpty(codeRattachement)) {
			where.and(ConditionHelper.getConditionStructure("T1.META_CODE_RATTACHEMENT", codeRattachement));
		}
		if (isNotEmpty(langue)) {
			if (!"0000".equals(langue)) {
				where.and(ConditionHelper.egalVarchar("T1.META_LANGUE", langue));
			}
		}
		if (isNotEmpty(etatObjet)) {
			if (!"0000".equals(etatObjet)) {
				where.and(ConditionHelper.egalVarchar("T1.META_ETAT_OBJET", etatObjet));
			}
		}
		where.and(traiterDate("T1.META_DATE_CREATION", debutCreation, finCreation));
		where.and(traiterDate("T1.META_DATE_MODIFICATION", debutModification, finModification));
		where.and(traiterDate("T1.META_DATE_MISE_EN_LIGNE", debutMiseEnLigne, finMiseEnLigne));
		if (isNotEmpty(codeObjet) && !"0000".equals(codeObjet)) {
			final FicheUniv fiche = ReferentielObjets.instancierFiche(ReferentielObjets.getNomObjet(codeObjet));
			where.and(RequeteSQLHelper.traiterConditionDsiMeta(autorisations, fiche, "", codeRubrique, null));
		} else {
			final ConditionList conditionSurObjets = new ConditionList();
			for (final String nomObjet : ReferentielObjets.getListeNomsObjet()) {
				final FicheUniv fiche = ReferentielObjets.instancierFiche(nomObjet);
				if (fiche != null && FicheAnnotationHelper.isAccessibleBo(fiche)) {
					fiche.setCodeRubrique(codeRubrique);
					conditionSurObjets.or(RequeteSQLHelper.traiterConditionDsiMeta(autorisations, fiche, "", codeRubrique, null));
				}
			}
			where.and(conditionSurObjets);
		}
		where.and(ConditionHelper.getConditionRubPubSuivantAction(ctx, "", "T1.META_CODE_RUBRIQUE", codeRubrique));
		requeteSelect.where(where);
		requeteSelect.join(jointurePourClauseWhere(where));
		requeteSelect.limit(LimitHelper.ajouterCriteresLimitesEtOptimisation(ctx, nombre));
		final Metatag meta = new Metatag();
		meta.init();
		meta.setCtx(ctx);
		meta.select(requeteSelect.formaterRequete());
		final Collection<Metatag> result = new ArrayList<>();
		while (meta.nextItem()) {
			result.add(meta.clone());
		}
		return result;
	}

	private static ClauseJoin jointurePourClauseWhere(final ClauseWhere where) {
		final ClauseJoin joinRubPub = new ClauseJoin();
		if (where.formaterSQL().contains("RUB_PUB.")) {
			joinRubPub.setTypeJointure(TypeJointure.LEFT_JOIN);
			joinRubPub.setNomTable("RUBRIQUEPUBLICATION RUB_PUB");
			joinRubPub.on(ConditionHelper.critereJointureSimple("T1.META_CODE", "RUB_PUB.CODE_FICHE_ORIG"));
			joinRubPub.and(ConditionHelper.critereJointureSimple("T1.META_LANGUE", "RUB_PUB.LANGUE_FICHE_ORIG"));
			joinRubPub.and(ConditionHelper.critereJointureSimple("T1.META_CODE_OBJET", "RUB_PUB.TYPE_FICHE_ORIG"));
		}
		return joinRubPub;
	}
}
