/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils.sql.criterespecifique;

import java.util.Hashtable;
import java.util.Vector;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.database.OMContext;
import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.FicheRattachementsSecondaires;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.Perimetre;
import com.univ.objetspartages.om.PermissionBean;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.StructureModele;
import com.univ.utils.ClassMetaUtil;
import com.univ.utils.sql.ConstanteSQL;
import com.univ.utils.sql.condition.Condition;
import com.univ.utils.sql.condition.ConditionList;

/**
 * Classe utilitaire permettant de calculer les conditions pour le BO lors d'une requête SQL. C'est une migration de SQLUtil.ajouterCriteresBO(String, OMContext, FicheUniv)
 * 
 * @author olivier.camon
 */
public final class CritereBO {

	/**
	 * 
	 * @param ctx
	 * @param fiche
	 * @return
	 */
	protected static Condition traiterCritereBO(final OMContext ctx, final FicheUniv fiche) {
		// la montagne russe commence!
		ConditionList critereBO = new ConditionList();
		Boolean aucuneRestriction = Boolean.FALSE;
		boolean isATraiterBO = isATraiterBO(ctx);
		if (!isATraiterBO) {
			return critereBO;
		}
		AutorisationBean autorisations = (AutorisationBean) ctx.getDatas().get("AUTORISATIONS");
		if (autorisations != null) {
			Hashtable<String, Vector<Perimetre>> listePermissions = autorisations.getListePermissions();
			String codeObjet = ReferentielObjets.getCodeObjet(fiche);
			for (String clePermission : listePermissions.keySet()) {
				PermissionBean permission = new PermissionBean(clePermission);
				// on traite les permissions sur l'objet courant
				if (isPermissionATraiter(permission, codeObjet)) {
					Vector<Perimetre> perimetres = listePermissions.get(clePermission);
					// on parcourt l'ensemble des périmètres assocciés à la permission
					for (Perimetre perimetreCourant : perimetres) {
						ConditionList conditionsSelectionPerimetre = new ConditionList();
						if (isPerimetreSansRestriction(perimetreCourant)) {
							aucuneRestriction = true;
						} else {
							conditionsSelectionPerimetre = traiterCriterePerimetre(fiche, perimetreCourant);
							if (!conditionsSelectionPerimetre.isEmpty()) {
								critereBO.or(conditionsSelectionPerimetre);
							}
						}
					}
				}
			}
			if (!aucuneRestriction) {
				critereBO.or(ConditionHelper.egalVarchar("CODE_REDACTEUR", autorisations.getCode()));
			}
		}
		if (critereBO.isEmpty() && !aucuneRestriction) {
			critereBO.and(ConstanteSQL.CONDITION_IMPOSSIBLE);
		}
		return critereBO;
	}

	private static ConditionList traiterCriterePerimetre(final FicheUniv fiche, final Perimetre perimetre) {
		ConditionList conditionsSelectionPerimetre = new ConditionList();
		boolean criteresPertinents = Boolean.FALSE;
		String codeRubrique = perimetre.getCodeRubrique();
		if (StringUtils.isNotEmpty(codeRubrique)) {
			criteresPertinents = Boolean.TRUE;
			Condition conditionRubrique = traiterConditionRubrique(codeRubrique, fiche);
			conditionsSelectionPerimetre.setPremiereCondtion(conditionRubrique);
		}
		String codeStructure = perimetre.getCodeStructure();
		if (StringUtils.isNotEmpty(codeStructure)) {
			criteresPertinents = Boolean.TRUE;
			ConditionList conditionStructure = traiterConditionStructure(codeStructure, fiche);
			if (!conditionStructure.isEmpty()) {
				conditionsSelectionPerimetre.and(conditionStructure);
			}
		}
		String codeEspaceCollaboratif = perimetre.getCodeEspaceCollaboratif();
		if (StringUtils.isNotEmpty(codeEspaceCollaboratif)) {
			ConditionList conditionsEspaceCollab = new ConditionList(ConditionHelper.egalVarchar("T1.DIFFUSION_MODE_RESTRICTION", "4"));
			conditionsEspaceCollab.and(ConditionHelper.egalVarchar("T1.DIFFUSION_PUBLIC_VISE_RESTRICTION", codeEspaceCollaboratif));
			conditionsSelectionPerimetre.and(conditionsEspaceCollab);
			criteresPertinents = Boolean.TRUE;
		}
		if (!criteresPertinents) {
			return new ConditionList();
		}
		return conditionsSelectionPerimetre;
	}

	private static Condition traiterConditionRubrique(final String codeRubrique, final FicheUniv fiche) {
		Condition conditionRubrique = null;
		String nomCodeRubrique = ClassMetaUtil.getClassMetaValue(fiche, "NOM_CODE_RUBRIQUE");
		if (ConstanteSQL.JOCKER_CODE.equals(codeRubrique)) {
			conditionRubrique = ConditionHelper.egalVarchar(nomCodeRubrique, "");
		} else {
			conditionRubrique = ConditionHelper.getConditionRubrique(nomCodeRubrique.toUpperCase(), codeRubrique);
		}
		return conditionRubrique;
	}

	private static boolean isATraiterBO(final OMContext ctx) {
		Boolean isATraiterBO = Boolean.TRUE;
		if (ctx.getDatas() != null) {
			String controle = (String) ctx.getDatas().get("CONTROLE_PERIMETRE_BO");
			isATraiterBO = "1".equals(controle);
		}
		return isATraiterBO;
	}

	private static boolean isPermissionATraiter(final PermissionBean permission, final String codeObjet) {
		return permission.getType().equals("FICHE") && permission.getObjet().equals(codeObjet) && permission.getAction().equals("M");
	}

	private static boolean isPerimetreSansRestriction(final Perimetre perimetre) {
		return StringUtils.isEmpty(perimetre.getCodeRubrique()) && StringUtils.isEmpty(perimetre.getCodeStructure()) && StringUtils.isEmpty(perimetre.getCodeProfil()) && StringUtils.isEmpty(perimetre.getCodeGroupe());
	}

	private static ConditionList traiterConditionStructure(final String codeStructure, final FicheUniv fiche) {
		ConditionList conditionStructure = new ConditionList();
		String nomCodeStructure = ClassMetaUtil.getClassMetaValue(fiche, "NOM_CODE_RATTACHEMENT");
		if (ConstanteSQL.JOCKER_CODE.equals(codeStructure)) {
			conditionStructure.setPremiereCondtion(ConditionHelper.egalVarchar(nomCodeStructure, ""));
			if (fiche instanceof FicheRattachementsSecondaires) {
				nomCodeStructure = ClassMetaUtil.getClassMetaValue(fiche, "NOM_CODE_RATTACHEMENT_AUTRES");
				conditionStructure.and(ConditionHelper.egalVarchar(nomCodeStructure, ""));
			}
		} else {
			conditionStructure.setPremiereCondtion(ConditionHelper.getConditionStructure(nomCodeStructure.toUpperCase(), codeStructure));
			if (fiche instanceof StructureModele) {
				conditionStructure.or(ConditionHelper.egalVarchar("CODE", codeStructure));
			}
			if (fiche instanceof FicheRattachementsSecondaires) {
				nomCodeStructure = ClassMetaUtil.getClassMetaValue(fiche, "NOM_CODE_RATTACHEMENT_AUTRES");
				conditionStructure.or(ConditionHelper.getConditionStructureMultiple(nomCodeStructure.toUpperCase(), codeStructure));
			}
		}
		return conditionStructure;
	}
}
