/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils.sql.criterespecifique;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;

import org.apache.commons.lang3.StringUtils;

import com.jsbsoft.jtf.database.OMContext;
import com.univ.multisites.InfosSite;
import com.univ.objetspartages.om.FicheUniv;
import com.univ.objetspartages.om.InfosRubriques;
import com.univ.objetspartages.om.ReferentielObjets;
import com.univ.objetspartages.om.Rubrique;
import com.univ.utils.ContexteUniv;
import com.univ.utils.IRequeteurConstantes;
import com.univ.utils.sql.ConstanteSQL;
import com.univ.utils.sql.condition.Condition;
import com.univ.utils.sql.condition.ConditionList;

/**
 * Classe utilitaire calculant les conditions sur la table RUBRIQUEPUBLICATION.
 * 
 * @author olivier.camon
 * 
 */
public final class CritereRubriquePublicationHelper {

	protected static Condition traiterConditionRubPubIdRubriqueAndFicheOrig(final FicheUniv fiche) {
		final String codeObjet = ReferentielObjets.getCodeObjet(fiche);
		final ConditionList critereRubPub = new ConditionList(ConditionHelper.isNull("RUB_PUB.ID_RUBRIQUEPUBLICATION"));
		critereRubPub.or(ConditionHelper.egalVarchar("RUB_PUB.TYPE_FICHE_ORIG", codeObjet));
		return critereRubPub;
	}

	protected static Condition traiterCritereRubPubDSIFiche(final OMContext ctx, final FicheUniv ficheUniv, final String codeRubrique) {
		final Collection<String> listeCodeRubriques = getCodesRubriquesSuivantCritereDSIDeLaFiche(ctx, ficheUniv, codeRubrique);
		final ConditionList conditionRubPub = new ConditionList(ConditionHelper.in("T1.CODE_RUBRIQUE", listeCodeRubriques));
		conditionRubPub.or(ConditionHelper.in("RUB_PUB.RUBRIQUE_DEST", listeCodeRubriques));
		return conditionRubPub;
	}

	private static Collection<String> getCodesRubriquesSuivantCritereDSIDeLaFiche(final OMContext ctx, final FicheUniv ficheUniv, String codeRubrique) {
		final Collection<String> listeCodesRubriques = new ArrayList<String>();
		String codeRubriqueSite = null;
		boolean rechercheArborescente = Boolean.TRUE;
		InfosSite infosSite = null;
		ContexteUniv ctxUniv = null;
		if (ctx instanceof ContexteUniv && ((ContexteUniv) ctx).isCalculListeResultatsFront()) {
			infosSite = ctx.getInfosSite();
			ctxUniv = (ContexteUniv) ctx;
			if (isSiteCloisonne(infosSite)) {
				codeRubriqueSite = infosSite.getCodeRubrique();
			}
		}
		if (StringUtils.isNotEmpty(codeRubrique) && codeRubrique.endsWith("_NOARBO")) {
			codeRubrique = StringUtils.substringBefore(codeRubrique, "_NOARBO");
			rechercheArborescente = Boolean.FALSE;
		}
		InfosRubriques rubrique = Rubrique.renvoyerItemRubrique(codeRubrique);
		codeRubrique = getCodeRubriqueCloisonnee(infosSite, rubrique, codeRubrique, codeRubriqueSite);
		rubrique = Rubrique.renvoyerItemRubrique(codeRubrique);
		if (isCodeExisteMaisPasRubrique(codeRubrique, rubrique)) {
			codeRubrique = Rubrique.CODE_RUBRIQUE_INEXISTANTE;
		}
		final Collection<InfosRubriques> listeRubriques = getListeRubriques(ctxUniv, rubrique, codeRubrique, IRequeteurConstantes.MODE_CONSULTATION, rechercheArborescente);
		for (final InfosRubriques rubCourante : listeRubriques) {
			listeCodesRubriques.add(rubCourante.getCode());
		}
		return listeCodesRubriques;
	}

	protected static Condition traiterCritereRubriquePublicationSuivantAction(final OMContext ctx, final String action, final String nomColonne, final String codeRubrique) {
		final Collection<String> listeCodeRubriques = getCodesRubriquesSuivantAction(ctx, action, codeRubrique);
		final ConditionList conditionRubPub = new ConditionList(ConditionHelper.in(nomColonne, listeCodeRubriques));
		conditionRubPub.or(ConditionHelper.in("RUB_PUB.RUBRIQUE_DEST", listeCodeRubriques));
		return conditionRubPub;
	}

	public static Collection<String> getCodesRubriquesSuivantAction(final OMContext ctx, final String action, String codeRubrique) {
		final Collection<String> listeCodesRubriques = new ArrayList<String>();
		String codeRubriqueSite = null;
		InfosSite infosSite = null;
		final boolean rechercheArborescente = !StringUtils.endsWith(codeRubrique, "_NOARBO");
		codeRubrique = StringUtils.removeEnd(codeRubrique, "_NOARBO");
		if (IRequeteurConstantes.MODE_CONSULTATION.equals(action)) {
			infosSite = ctx.getInfosSite();
			if (isSiteCloisonne(infosSite)) {
				codeRubriqueSite = infosSite.getCodeRubrique();
			}
		}
		InfosRubriques rubrique = Rubrique.renvoyerItemRubrique(codeRubrique);
		codeRubrique = getCodeRubriqueCloisonnee(infosSite, rubrique, codeRubrique, codeRubriqueSite);
		rubrique = Rubrique.renvoyerItemRubrique(codeRubrique);
		if (isCodeExisteMaisPasRubrique(codeRubrique, rubrique)) {
			codeRubrique = Rubrique.CODE_RUBRIQUE_INEXISTANTE;
		}
		final Collection<InfosRubriques> listeRubriques = getListeRubriques(ctx, rubrique, codeRubrique, action, rechercheArborescente);
		for (final InfosRubriques rubCourante : listeRubriques) {
			listeCodesRubriques.add(rubCourante.getCode());
		}
		return listeCodesRubriques;
	}

	private static Collection<InfosRubriques> getListeRubriques(final OMContext ctxUniv, InfosRubriques rubrique, final String codeRubrique, final String action,
		final boolean rechercheArborescente) {
		Collection<InfosRubriques> listeRubriques = new HashSet<InfosRubriques>();
		if (isCodeRubriqueExistante(codeRubrique)) {
			if (IRequeteurConstantes.MODE_CONSULTATION.equals(action) && ctxUniv != null && ctxUniv instanceof ContexteUniv && "1".equals(ConstanteSQL.ACTIVATION_DSI)) {
				if (Rubrique.controlerRestrictionRubrique((ContexteUniv) ctxUniv, codeRubrique)) {
					if (rechercheArborescente) {
						listeRubriques = Rubrique.determinerListeSousRubriquesAutorisees((ContexteUniv) ctxUniv, rubrique);
					}
				} else {
					rubrique = new InfosRubriques(Rubrique.CODE_RUBRIQUE_INEXISTANTE);
				}
				listeRubriques.add(rubrique);
			} else if (StringUtils.isNotEmpty(codeRubrique)) {
				if (rechercheArborescente) {
					listeRubriques = rubrique.getListeSousRubriquesTousNiveaux();
				}
				listeRubriques.add(rubrique);
			}
		} else {
			listeRubriques.add(new InfosRubriques(Rubrique.CODE_RUBRIQUE_INEXISTANTE));
		}
		return listeRubriques;
	}

	private static boolean isSiteCloisonne(final InfosSite infosSite) {
		return infosSite != null && infosSite.getRestriction() == 1;
	}

	private static String getCodeRubriqueCloisonnee(final InfosSite infosSite, final InfosRubriques rubrique, final String codeRubrique, final String codeRubriqueSite) {
		String codeRubriqueRetour = codeRubrique;
		if (StringUtils.isNotEmpty(codeRubriqueSite)) {
			if (isCodeExisteMaisPasRubrique(codeRubrique, rubrique)) {
				codeRubriqueRetour = Rubrique.CODE_RUBRIQUE_INEXISTANTE;
			} else if (StringUtils.isEmpty(codeRubrique)) {
				codeRubriqueRetour = codeRubriqueSite;
			} else if (!infosSite.isRubriqueVisibleInSite(rubrique)) {
				codeRubriqueRetour = Rubrique.CODE_RUBRIQUE_INEXISTANTE;
			}
		}
		return codeRubriqueRetour;
	}

	private static boolean isCodeExisteMaisPasRubrique(final String codeRubrique, final InfosRubriques infoRub) {
		return StringUtils.isNotEmpty(codeRubrique) && StringUtils.isEmpty(infoRub.getCode());
	}

	private static boolean isCodeRubriqueExistante(final String codeRubrique) {
		return !Rubrique.CODE_RUBRIQUE_INEXISTANTE.equals(codeRubrique);
	}
}
