/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.utils.sql.operande;

import java.sql.Date;
import java.util.Collection;

/**
 * Fabrique permettant de construire les opérandes en fonction de leur type.
 *
 * @author olivier.camon
 *
 */
public class OperandeFactory {

	/**
	 * Retourne le bon type d'opérande en fonction du type choisie. Si le type n'est pas encore prévue, l'opérande par défaut est de type VARCHAR. Si le type choisi ne correspond
	 * pas au type d'objet de valeurOperande, une Exception de type {@link IllegalArgumentException } est lancée.
	 *
	 * @param valeurOperande
	 *            La valeur de l'opérande
	 * @param typeValeur
	 *            le type d'opérande VARCHAR...
	 * @return L'instance d'opérande correspondant à ce que l'on souhaite.
	 */
	@SuppressWarnings("unchecked")
	public static Operande creerOperande(final Object valeurOperande, final TypeOperande typeValeur) {
		Operande operandeACreer = null;
		if (typeValeur == null) {
			gererErreur(valeurOperande, typeValeur);
		}
		switch (typeValeur) {
			case VARCHAR:
				if (valeurOperande instanceof String) {
					operandeACreer = new VarcharOperande((String) valeurOperande);
				} else {
					gererErreur(valeurOperande, typeValeur);
				}
				break;
			case NOM_CHAMP:
				if (valeurOperande instanceof String) {
					operandeACreer = new NomChamp((String) valeurOperande);
				} else {
					gererErreur(valeurOperande, typeValeur);
				}
				break;
			case LISTE_VARCHAR:
				if (valeurOperande instanceof Collection<?>) {
					operandeACreer = new ListeVarchar((Collection<String>) valeurOperande);
				} else {
					gererErreur(valeurOperande, typeValeur);
				}
				break;
			case DATE:
				if (valeurOperande instanceof Date) {
					operandeACreer = new OperandeDate((Date) valeurOperande, Boolean.FALSE);
				} else {
					gererErreur(valeurOperande, typeValeur);
				}
				break;
			case DATE_ET_HEURE:
				if (valeurOperande instanceof Date) {
					operandeACreer = new OperandeDate((Date) valeurOperande, Boolean.TRUE);
				} else {
					gererErreur(valeurOperande, typeValeur);
				}
				break;
			case LONG:
				if (valeurOperande instanceof Long) {
					operandeACreer = new LongOperande((Long) valeurOperande);
				} else {
					gererErreur(valeurOperande, typeValeur);
				}
				break;
			case INTEGER:
				if (valeurOperande instanceof Integer) {
					operandeACreer = new IntegerOperande((Integer) valeurOperande);
				} else {
					gererErreur(valeurOperande, typeValeur);
				}
				break;
			case NON_ECHAPABLE:
				if (valeurOperande instanceof String) {
					operandeACreer = new NonEchapable((String) valeurOperande);
				} else {
					gererErreur(valeurOperande, typeValeur);
				}
				break;
			default:
				if (valeurOperande instanceof String) {
					operandeACreer = new VarcharOperande((String) valeurOperande);
				} else {
					gererErreur(valeurOperande, typeValeur);
				}
				break;
		}
		return operandeACreer;
	}

	private static void gererErreur(final Object valeurOperande, final TypeOperande typeValeur) {
		if (typeValeur == null) {
			throw new IllegalArgumentException("Impossible de creer une Operande de type null");
		} else if (valeurOperande == null) {
			throw new IllegalArgumentException("Impossible de creer une Operande de type : " + typeValeur.toString() + " avec une valeur null");
		} else {
			throw new IllegalArgumentException("Impossible de creer une Operande de type : " + typeValeur.toString() + " avec une valeur d'instance : " + valeurOperande.getClass());
		}
	}
}
