/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.xhtml;

import java.util.Iterator;

import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.Groupedsi;
import com.univ.objetspartages.om.InfosGroupeDsi;

// TODO: Auto-generated Javadoc
/**
 * Classe représentant un arbre Javascript de groupes DSI.
 * 
 * @author FBI
 */
public class JavascriptTreeGroupedsi {

	/** Arbre Javascript. */
	private JavascriptTree tree = null;

	/** Liste des autorisations de l'utilisateur. */
	private AutorisationBean autorisations = null;

	/** Permission demandée (ex : FICHE/0016/M). */
	private String permissions = null;

	/**
	 * Constructeur.
	 * 
	 * @param autorisations
	 *            Liste des permissions de l'utilisateur
	 * @param permissions
	 *            Permission demandée (ex : FICHE/0016/M)
	 */
	public JavascriptTreeGroupedsi(final AutorisationBean autorisations, final String permissions) {
		this.tree = new JavascriptTree();
		this.autorisations = autorisations;
		this.permissions = permissions;
	}

	/**
	 * Chargement de l'arbre des groupes DSI pour le groupe demandé.
	 * 
	 * @param groupe
	 *            Groupe à charger
	 * @param groupeSelectionne
	 *            Rubrique sélectionnée
	 * 
	 * @throws Exception
	 *             the exception
	 */
	public void load(final InfosGroupeDsi groupe, final InfosGroupeDsi groupeSelectionne) throws Exception {
		// Ajoute les éléments à l'arbre
		final Iterator<InfosGroupeDsi> listSousGroupesIt = groupe.getListeSousGroupesSortedByLibelle().iterator();
		InfosGroupeDsi sousGroupe = null;
		boolean selectable = true;
		while (listSousGroupesIt.hasNext()) {
			sousGroupe = listSousGroupesIt.next();
			selectable = isGroupeSelectable(sousGroupe); // élément sélectionnable ou non
			if (isGroupeVisible(sousGroupe, selectable)) {
				// on ajoute l'élément
				tree.addItem(new JavascriptTreeItem(sousGroupe.getCodeGroupePere(), sousGroupe.getCode(), sousGroupe.getIntitule(), sousGroupe.contains(groupeSelectionne), // élément ouvert ou non
					selectable));
				if (!sousGroupe.getListeSousGroupes().isEmpty()) {
					load(sousGroupe, groupeSelectionne);
				}
			}
		}
	}

	/**
	 * Affiche l'arbre Javascript.
	 * 
	 * @return Le code Javascript d'affichage de l'arbre
	 */
	public String print() {
		return tree.print();
	}

	/**
	 * Teste si le groupe DSI est sélectionnable.
	 * 
	 * @param groupeDsi
	 *            Le groupe DSI à tester
	 * 
	 * @return true si le groupe DSI est sélectionnable
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private boolean isGroupeSelectable(final InfosGroupeDsi groupeDsi) throws Exception {
		// Controle le périmètre de l'utilisateur sur le groupe
		boolean selectableItem = groupeDsi.isSelectionnable();
		if (selectableItem && permissions.length() > 0) {
			// patch filtre sur les groupes dynamiques
			if (permissions.startsWith("DYN")) {
				selectableItem = !(groupeDsi.getRequete().length() > 0);
			} else {
				selectableItem = Groupedsi.controlerPermission(autorisations, permissions, groupeDsi.getCode());
			}
		}
		return selectableItem;
	}

	/**
	 * Teste si le groupe est visible dans l'arbre (groupe sélectionnable ou ayant un fils visible.
	 * 
	 * @param groupe
	 *            Le groupe à tester
	 * @param selectable
	 *            true si le groupe est sélectionnable
	 * 
	 * @return true si le groupe est visible dans l'arbre
	 * 
	 * @throws Exception
	 *             the exception
	 */
	private boolean isGroupeVisible(final InfosGroupeDsi groupe, final boolean selectable) throws Exception {
		boolean visible = selectable;
		if (!selectable) // groupe non sélectionnable
		{
			// on vérifie ses groupes fils
			final Iterator<InfosGroupeDsi> listSousGroupesIt = groupe.getListeSousGroupes().iterator();
			InfosGroupeDsi sousGroupe = null;
			while (listSousGroupesIt.hasNext() && !visible) {
				sousGroupe = listSousGroupesIt.next();
				visible = isGroupeVisible(sousGroupe, isGroupeSelectable(sousGroupe));
			}
		}
		return visible;
	}
}
