/**
 * Copyright (C) 2015 - 2018 Kosmos contact@kosmos.fr
 *
 * Projet: core
 * Version: 6.02.48
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.univ.xhtml;

import java.util.Iterator;

import com.univ.objetspartages.om.AutorisationBean;
import com.univ.objetspartages.om.InfosStructure;
import com.univ.objetspartages.om.PermissionBean;
import com.univ.objetspartages.om.Structure;

// TODO: Auto-generated Javadoc
/**
 * Classe représentant un arbre Javascript de structures.
 * 
 * @author FBI
 */
public class JavascriptTreeStructure {

	/** Arbre Javascript. */
	private JavascriptTree tree = null;

	/** Liste des autorisations de l'utilisateur. */
	private AutorisationBean autorisations = null;

	/** Permission demandée (ex : FICHE/0016/M). */
	private String permissions = null;

	/** Filtre sur les types de structure. */
	private TypeStructureFilter filter = null;

	/** Affichage back ou front. */
	private boolean front = false;

	/**
	 * Constructeur.
	 * 
	 * @param autorisations
	 *            Liste des permissions de l'utilisateur
	 * @param permissions
	 *            Permission demandée (ex : FICHE/0016/M)
	 * @param typeStructureList
	 *            Liste des types de structure autorisées (éléments séparés par des ;)
	 * @param front
	 *            true si affichage en front
	 */
	public JavascriptTreeStructure(final AutorisationBean autorisations, final String permissions, final String typeStructureList, final boolean front) {
		this.tree = new JavascriptTree();
		if (typeStructureList.length() > 0) {
			this.filter = new TypeStructureFilter(typeStructureList);
		}
		this.autorisations = autorisations;
		this.permissions = permissions;
		this.front = front;
	}

	/**
	 * Chargement de l'arbre des structures pour la structure demandée.
	 * 
	 * @param structure
	 *            Structure à charger
	 * @param structureSelectionnee
	 *            Structure sélectionnée
	 *
	 */
	public void load(final InfosStructure structure, final InfosStructure structureSelectionnee) {
		final Iterator<InfosStructure> listSousStructuresIt = structure.getListeSousStructuresSortedByLibelleLong().iterator();
		InfosStructure sousStructure = null;
		boolean selectable = true;
		while (listSousStructuresIt.hasNext()) {
			sousStructure = listSousStructuresIt.next();
			// si on est en front, vérifie que la structure est bien visible
			if (!front || sousStructure.isVisibleInFront()) {
				selectable = isStructureSelectable(sousStructure); // élément sélectionnable ou non
				if (isStructureVisible(sousStructure, selectable)) {
					// on ajoute l'élément
					tree.addItem(new JavascriptTreeItem(sousStructure.getCodeRattachement(), sousStructure.getCode(), sousStructure.getLibelleLong(),
						sousStructure.contains(structureSelectionnee), // élément ouvert ou non
						selectable));
					if (!sousStructure.getListeSousStructures().isEmpty()) {
						load(sousStructure, structureSelectionnee);
					}
				}
			}
		}
	}

	/**
	 * Affiche l'arbre Javascript.
	 * 
	 * @return Le code Javascript d'affichage de l'arbre
	 */
	public String print() {
		return tree.print();
	}

	/**
	 * Teste si la structure est sélectionnable.
	 * 
	 * @param structure
	 *            La structure à tester
	 * 
	 * @return true si la structure est sélectionnable
	 *
	 */
	private boolean isStructureSelectable(final InfosStructure structure) {
		boolean selectableItem = true;
		// Controle le périmètre de l'utilisateur sur la structure
		if (permissions.length() > 0) {
			selectableItem = Structure.controlerPermission(autorisations, permissions, structure.getCode());
		}
		// Controle le filtre sur les types de structures
		if (filter != null && selectableItem) {
			selectableItem = filter.accept(structure.getTypeStructure());
		}
		// si la fiche en cours de saisie est une fiche annuaire (ou étudiant ou ancien étudiant),
		// que l'utilisateur est redacteur de sa fiche,
		// et qu'il n'a pas de role concernant les fiches annuaire,
		// il peut choisir la structure
		final PermissionBean permissionCourante = new PermissionBean(permissions);
		if (!selectableItem && autorisations != null && autorisations.isRedacteurFicheCourante() && (permissionCourante.getObjet().equals("0006") || permissionCourante.getObjet().equals(
			"0007") || permissionCourante.getObjet().equals("0013")) && !autorisations.possedePermission(permissionCourante)) {
			selectableItem = true;
		}
		return selectableItem;
	}

	/**
	 * Teste si la structure est visible dans l'arbre (structure sélectionnable ou ayant un fils visible.
	 * 
	 * @param structure
	 *            La structure à tester
	 * @param selectable
	 *            true si la structure est sélectionnable
	 * 
	 * @return true si la structure est visible dans l'arbre
	 *
	 */
	private boolean isStructureVisible(final InfosStructure structure, final boolean selectable) {
		boolean visible = selectable;
		if (!selectable) // structure non sélectionnable
		{
			// on vérifie ses structures filles
			final Iterator<InfosStructure> listSousStructuresIt = structure.getListeSousStructures().iterator();
			InfosStructure sousStructure = null;
			while (listSousStructuresIt.hasNext() && !visible) {
				sousStructure = listSousStructuresIt.next();
				// Si on est en front, vérifie que la structure est bien visible
				if (!front || sousStructure.isVisibleInFront()) {
					visible = isStructureVisible(sousStructure, isStructureSelectable(sousStructure));
				}
			}
		}
		return visible;
	}
}

/**
 * Classe implémentant un filtre sur les types de structure.
 */
class TypeStructureFilter {

	/** Liste des types de structures autorisés. */
	private String authorizedTypeStructureList = null;

	/**
	 * Constructeur.
	 * 
	 * @typeStructureList Liste des types de structures autorisés (séparation des types de structure par des ';')
	 */
	TypeStructureFilter(final String typeStructureList) {
		this.authorizedTypeStructureList = typeStructureList;
	}

	/**
	 * Teste si le type de structure est accepté.
	 * 
	 * @param typeStructure
	 *            Le type de structure à tester
	 * @return true si le type est accepté
	 */
	boolean accept(final String typeStructure) {
		return typeStructure == null || typeStructure.length() == 0 || authorizedTypeStructureList.indexOf(typeStructure) != -1;
	}
}
